package org.lsst.ccs.drivers.thorlabs;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import java.util.Date;

/**
 *  Program to test the Thorlabs filter wheel driver.
 */

public class TestFW {

    private final ThorlabsFW thfw;

   /**
    *  Constructor
    */
    public TestFW()
    {
        thfw = new ThorlabsFW();
    }


    /**
     * Determines current time
     *
     * @return current time as a String
     */
    @Command(name="timestamp", description="Prints current time")
    public String timestamp() {
	Date now = new Date();
        return now.toString();
    }


   /**
    *  Opens connection to a device
    * 
    *  @param  ident     The identification (serial port name)
    *  @param  baudRate  BAUD rate (9600 or 115200)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="ident", description="Identification")
                     String ident, @Argument(name="baudRate",
		     description="Baud rate, 9600 or 115200") int baudRate) 
                     throws DriverException
    {
        thfw.open(ident, baudRate);
    }

   /**
    *  Opens connection to a device using default baud rate
    * 
    *  @param  ident     The identification (serial port name)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        thfw.open(ident);
    }


   /**
    *  Closes connection
    * 
    *  @throws  DriverException
    */
    @Command(name="close", description="Closes connection to device")
    public void close() throws DriverException {
        thfw.close();
    }


   /**
    * Read all settings and data from filter wheel.
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    */
    @Command(name="readAll", description="Read all controller settings and data")
    public String readAll()
    {

        String table = "Read all filter wheel settings and data\n" + timestamp() +"\n";

        ThorlabsFW.CmndFW cmndN[] = ThorlabsFW.CmndFW.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                String respN = thfw.queryFW(cmndN[i]);
                table += respN;
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";
        return table;
    }

   /**
    *  Read filter wheel position.
    *
    *  @return  String showing filter wheel position
    *  @throws  Driver Exception
    */
    @Command(name="readFWpos", description="Read filter wheel position")
    public String readFWpos()  throws DriverException
    {
        return thfw.queryFW(ThorlabsFW.CmndFW.POSITION);
    }


   /** 
    *  Command to move filter wheel
    * 
    *  @param   Position to move to (1 to maximum)
    *  @throws  DriverException
    */
    @Command(name="setPosition", description="Set filter wheel positions")
    public void setPosition(@Argument(name="<int>",
     description="New position, 1 to maximum") int value) 
     throws DriverException 
    {
        thfw.setFW(ThorlabsFW.CmndFW.POSITION, value);
    }

    /* Filter-wheel mode-setting commands   */

   /**
    *  Set total number of positions
    * 
    *  @param   Total number of positions for physical FW (6 or 12)
    *  @throws  DriverException
    */
    @Command(name="setFWsize", description="Set number of FW positions")
    public void setFWsize(@Argument(name=" <6|12>",
     description="Number of filter wheel positions") int value) 
     throws DriverException 
    {
        thfw.setFW(ThorlabsFW.CmndFW.POSITION_COUNT, value);
    }

   /**
    *  Set usage of trigger connector
    * 
    *  @param   0 for input, 1 for output (signal when move is complete)
    *  @throws  DriverException
    */
    @Command(name="setTrigMode", description="Set usage of trigger connector")
    public void setTrigMode(@Argument(name=" <0|1>", 
     description="trigger connector mode <input|output>") int value) 
     throws DriverException 
    {
        thfw.setFW(ThorlabsFW.CmndFW.TRIGGER_MODE, value);
    }

   /**
    *  Set filter wheel speed
    * 
    *  @param   0 for slow speed, 1 for high speed
    *  @throws  DriverException
    */
    @Command(name="setSpeed", description="Set filter wheel speed")
    public void setSpeed(@Argument(name=" <0|1>", 
     description="filter wheel speed <low|high>") int value) 
     throws DriverException 
    {
        thfw.setFW(ThorlabsFW.CmndFW.SPEED_MODE, value);
    }

   /**
    *  Set sensor mode for idle FW ("inactive" may avoid stray light)
    * 
    *  @param   0 (inactive) or 1 (active)
    *  @throws  DriverException
    */
    @Command(name="setSensorMode", description="Set sensor mode for idle FW")
    public void setSensorMode(@Argument(name=" <0|1>", 
     description="sensor mode <inactive|active> if FW idle") int value) 
     throws DriverException 
    {
        thfw.setFW(ThorlabsFW.CmndFW.SENSOR_MODE, value);
    }

   /**
    *  Set baud rate
    * 
    *  @param   0 (9600) or 1 (115200)
    *  @throws  DriverException
    */
    @Command(name="setDataRate", 
             description="Set baud rate.  Danger: last-resort reset only!")
    public void setDataRate(@Argument(name=" <0|1>", 
     description="baud rate <9600|115200>") int value) 
     throws DriverException 
    {
        thfw.setFW(ThorlabsFW.CmndFW.DATA_RATE, value);
    }

   /**
    *  Save settings for next power-up
    * 
    *  @throws  DriverException
    */
    @Command(name="saveSettings", description="Save settings for next power-up")
    public void saveSettings() throws DriverException 
    {
        thfw.saveFW();
    }

    /**
     *  Set debug mode on or off
     *
     *  <true|false>
     */
    @Command(name="setDebug",description="set debug mode <true|false>")
    public void setDebug(@Argument(name="<true|false>",
			 description="debug mode") boolean mode)
    {
        thfw.setDebug(mode);
    }

}
