package org.lsst.ccs.drivers.sorensen;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *  Program to test the Sorensen DCS-series device driver
 * 
 *  @author Owen Saxton
 */
public class TestDcs extends TestScpi {

    protected final Dcs dcs;

    protected enum onOff {
        OFF, ON;
    }


    /**
     *  Constructor
     */
    public TestDcs()
    {
        super(new Dcs());
        dcs = (Dcs)scpi;
    }


    /**
     *  Sets the voltage.
     *
     *  @param  value  The voltage to set
     *  @throws  DriverException
     */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set") double value) throws DriverException
    {
        dcs.setVoltage(value);
    }


    /**
     *  Shows the set voltage.
     * 
     *  @return  The set voltage
     *  @throws  DriverException
     */
    @Command(name="showvoltage", description="Show the set voltage")
    public double getVoltage() throws DriverException
    {
        return dcs.getVoltage();
    }


    /**
     *  Reads the actual voltage.
     * 
     *  @return  The actual voltage
     *  @throws  DriverException
     */
    @Command(name="readvoltage", description="Read the actual voltage")
    public double readVoltage() throws DriverException
    {
        return dcs.readVoltage();
    }


    /**
     *  Sets the soft voltage limit.
     * 
     *  @param  value  The voltage limit to set
     *  @throws  DriverException
     */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value", description="Voltage limit to set") double value) throws DriverException
    {
        dcs.setVoltageLimit(value);
    }


    /**
     *  Shows the soft voltage limit.
     * 
     *  @return  The voltage limit
     *  @throws  DriverException
     */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public double getVoltageLimit() throws DriverException
    {
        return dcs.getVoltageLimit();
    }


    /**
     *  Sets the current.
     * 
     *  @param  value  The current to set
     *  @throws  DriverException
     */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set") double value) throws DriverException
    {
        dcs.setCurrent(value);
    }


    /**
     *  Shows the set current.
     * 
     *  @return  The set current
     *  @throws  DriverException
     */
    @Command(name="showcurrent", description="Show the set current")
    public double getCurrent() throws DriverException
    {
        return dcs.getCurrent();
    }


    /**
     *  Reads the actual current.
     * 
     *  @return  The actual current
     *  @throws  DriverException
     */
    @Command(name="readcurrent", description="Read the actual current")
    public double readCurrent() throws DriverException
    {
        return dcs.readCurrent();
    }


    /**
     *  Sets the soft current limit.
     * 
     *  @param  value  The current limit to set
     *  @throws  DriverException
     */
    @Command(name="setcurrlim", description="Set the soft current limit")
    public void setCurrentLimit(@Argument(name="value", description="Current limit to set") double value) throws DriverException
    {
        dcs.setCurrentLimit(value);
    }


    /**
     *  Shows the soft current limit.
     * 
     *  @return  The current limit
     *  @throws  DriverException
     */
    @Command(name="showcurrlim", description="Show the soft current limit")
    public double getCurrentLimit() throws DriverException
    {
        return dcs.getCurrentLimit();
    }


    /**
     *  Turns the output on or off.
     * 
     *  @param  state  The output state to set: on or off
     *  @throws  DriverException
     */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state", description="Output state: on or off") onOff state) throws DriverException
    {
        dcs.setOutput(state == onOff.ON);
    }


    /**
     *  Shows the output state.
     * 
     *  @return  The output state: on or off
     *  @throws  DriverException
     */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return dcs.getOutput() ? "on" : "off";
    }


    /**
     *  Turns local mode on or off.
     * 
     *  @param  mode  The local mode to set: on or off
     *  @throws  DriverException
     */
    @Command(name="setlocal", description="Turn local mode on or off")
    public void setLocal(@Argument(name="mode", description="Local mode: on or off") onOff mode) throws DriverException
    {
        dcs.setLocal(mode == onOff.ON);
    }


    /**
     *  Shows the local mode.
     * 
     *  @return  The local mode: on or off
     *  @throws  DriverException
     */
    @Command(name="showlocal", description="Show the local mode")
    public String showLocal() throws DriverException
    {
        return dcs.getLocal() ? "on" : "off";
    }


    /**
     *  Shows the source status block.
     *
     *  @return  The source status block
     *  @throws  DriverException
     */
    @Command(name="showstatusblock", description="Show the source status block")
    public Dcs.StatusBlock getStatusBlock() throws DriverException
    {
        return dcs.getStatusBlock();
    }

}
