package org.lsst.ccs.drivers.sorensen;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *  Enables access to Sorensen DCS series power supplies
 *
 *  @author  Owen Saxton
 */

public class Dcs extends Scpi {

    /**
     *  Public constants.
     */
    public static final int
        STAT_CHANNEL            = 0,
        STAT_ONLINE             = 1,
        STAT_STATUS_FLAGS       = 2,
        STAT_STATUS             = 3,
        STAT_ACCUM_STATUS       = 4,
        STAT_FAULT_MASK         = 5,
        STAT_FAULT              = 6,
        STAT_ERROR              = 7,
        STAT_SERIAL             = 8,
        STAT_VOLTAGE            = 9,
        STAT_CURRENT            = 10,
        STAT_OVER_VOLTAGE       = 11,
        STAT_VOLTAGE_DAC_GAIN   = 12,
        STAT_VOLTAGE_DAC_OFFSET = 13,
        STAT_CURRENT_DAC_GAIN   = 14,
        STAT_CURRENT_DAC_OFFSET = 15,
        STAT_OVP_DAC_GAIN       = 16,
        STAT_OVP_DAC_OFFSET     = 17,
        STAT_VOLTAGE_ADC_GAIN   = 18,
        STAT_VOLTAGE_ADC_OFFSET = 19,
        STAT_CURRENT_ADC_GAIN   = 20,
        STAT_CURRENT_ADC_OFFSET = 21,
        STAT_MODEL              = 22,
        STAT_OVP_CALIBRATED     = 23,
        N_STAT_FIELDS           = 24;

    /**
     *  Private fields.
     */
    private static final int
        DEFAULT_PORT = 9221,
        BAUD_RATE    = 19200;

    /**
     *  Status block contents.
     */
    public static class StatusBlock {

        int     channel;
        boolean online;
        int     statusFlags;
        int     status;
        int     accumStatus;
        int     faultMask;
        int     fault;
        int     error;
        String  serial;
        double  voltage;
        double  current;
        double  overVoltage;
        double  voltageDacGain;
        double  voltageDacOffset;
        double  currentDacGain;
        double  currentDacOffset;
        double  ovpDacGain;
        double  ovpDacOffset;
        double  voltageAdcGain;
        double  voltageAdcOffset;
        double  currentAdcGain;
        double  currentAdcOffset;
        String  model;
        boolean ovpCalibrated;

        @Override
        public String toString()
        {
            return   "Channel number:         " + channel +
                   "\nOnline:                 " + online +
                   "\nStatus flags register:  " + toHex4(statusFlags) +
                   "\nStatus register:        " + toHex4(status) +
                   "\nAccumulated status:     " + toHex4(accumStatus) +
                   "\nFault mask register:    " + toHex4(faultMask) +
                   "\nFault register:         " + toHex4(fault) +
                   "\nError register:         " + toHex4(error) +
                   "\nSerial number:          " + serial +
                   "\nModel voltage:          " + voltage +
                   "\nModel current:          " + current +
                   "\nModel over-voltage:     " + overVoltage +
                   "\nVoltage DAC gain:       " + voltageDacGain +
                   "\nVoltage DAC offset:     " + voltageDacOffset +
                   "\nCurrent DAC gain:       " + currentDacGain +
                   "\nCurrent DAC offset:     " + currentDacOffset +
                   "\nOVP DAC gain:           " + ovpDacGain +
                   "\nOVP DAC offset:         " + ovpDacOffset +
                   "\nVoltage ADC gain        " + voltageAdcGain +
                   "\nVoltage ADC offset:     " + voltageAdcOffset +
                   "\nCurrent ADC gain:       " + currentAdcGain +
                   "\nCurrent ADC offset:     " + currentAdcOffset +
                   "\nModel string:           " + model +
                   "\nOVP calibrated:         " + ovpCalibrated;
        }

        private String toHex4(int value)
        {
            return String.format("0x%04x", value);
        }

    }


    /**
     *  Opens a connection to the power supply.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network
     *                   serial number for FTDI device
     *                   device name for serial
     *  @param  parm1  The device parameter, or 0 for default:
     *                   port number for network (default 9221)
     *                   baud rate for FTDI & serial (default 19200)
     *  @param  parm2  The second device parameter (ignored)
     * @throws  DriverException
     */
    @Override
    public void open(ConnType type, String ident, int parm1, int parm2) throws DriverException
    {
        super.open(type, ident,
                   parm1 != 0 ? parm1 : type == ConnType.NET ? DEFAULT_PORT : BAUD_RATE, 0);
        try {
            checkIdentification("SORENSEN", CHECK_EQUALS, "DCS", CHECK_STARTS_WITH);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


    /**
     *  Sets the voltage.
     *
     *  @param  value  The voltage value to set
     *  @throws  DriverException
     */
    public void setVoltage(double value) throws DriverException
    {
        write("source:voltage " + value);
    }


    /**
     *  Reads back the set voltage.
     *
     *  @return  The set voltage
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public double getVoltage() throws DriverException
    {
        return readDouble("source:voltage?");
    }


    /**
     *  Reads the actual voltage.
     *
     *  @return  The actual voltage
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public double readVoltage() throws DriverException
    {
        return readDouble("measure:voltage?");
    }


    /**
     *  Sets the soft voltage limit.
     *
     *  @param  value  The voltage limit value to set
     *  @throws  DriverException
     */
    public void setVoltageLimit(double value) throws DriverException
    {
        write("source:voltage:limit " + value);
    }


    /**
     *  Reads back the soft voltage limit.
     *
     *  @return  The voltage limit
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public double getVoltageLimit() throws DriverException
    {
        return readDouble("source:voltage:limit?");
    }


    /**
     *  Sets the current.
     *
     *  @param  value  The current value to set
     *  @throws  DriverException
     */
    public void setCurrent(double value) throws DriverException
    {
        write("source:current " + value);
    }


    /**
     *  Reads back the set current.
     *
     *  @return  The set current
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public double getCurrent() throws DriverException
    {
        return readDouble("source:current?");
    }


    /**
     *  Reads the actual current.
     *  @return  The actual current
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public double readCurrent() throws DriverException
    {
        return readDouble("measure:current?");
    }


    /**
     *  Sets the soft current limit.
     *
     *  @param  value  The current limit value to set
     *  @throws  DriverException
     */
    public void setCurrentLimit(double value) throws DriverException
    {
        write("source:current:limit " + value);
    }


    /**
     *  Reads back the soft current limit.
     *
     *  @return  The current limit
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public double getCurrentLimit() throws DriverException
    {
        return readDouble("source:current:limit?");
    }


    /**
     *  Turns the output on or off.
     *
     *  @param  on  If true, turn the output on, otherwise turn it off.
     *  @throws  DriverException
     */
    public void setOutput(boolean on) throws DriverException
    {
        write("output:state " + (on ? "on" : "off"));
    }


    /**
     *  Gets the output state.
     *
     *  @return  Whether or not output is on
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public boolean getOutput() throws DriverException
    {
        return read("output:state?").equals("1");
    }


    /**
     *  Turns local mode on or off.
     *
     *  @param  on  If true, turn local mode on, otherwise turn it off.
     *  @throws  DriverException
     */
    public void setLocal(boolean on) throws DriverException
    {
        write("system:local " + (on ? "on" : "off"));
    }


    /**
     *  Gets the local mode.
     *
     *  @return  Whether or not local mode is in effect
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public boolean getLocal() throws DriverException
    {
        return read("system:local?").equals("1");
    }


    /**
     *  Gets the source status block.
     *
     *  @return  The status block.
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public StatusBlock getStatusBlock() throws DriverException
    {
        StatusBlock block = new StatusBlock();
        String[] status
          = read("source:status:block?").split(",", N_STAT_FIELDS);
        if (status.length < N_STAT_FIELDS - 1) {
            throw new DriverException("Too few status block fields: " + status.length);
        }
        try {
            block.channel = Integer.decode(status[STAT_CHANNEL]);
            block.online = status[STAT_ONLINE].equals("1");
            block.statusFlags = fromHex(status[STAT_STATUS_FLAGS]);
            block.status = fromHex(status[STAT_STATUS]);
            block.accumStatus = fromHex(status[STAT_ACCUM_STATUS]);
            block.faultMask = fromHex(status[STAT_FAULT_MASK]);
            block.fault = fromHex(status[STAT_FAULT]);
            block.error = fromHex(status[STAT_ERROR]);
            block.serial = status[STAT_SERIAL];
            block.voltage = Double.valueOf(status[STAT_VOLTAGE]);
            block.current = Double.valueOf(status[STAT_CURRENT]);
            block.overVoltage = Double.valueOf(status[STAT_OVER_VOLTAGE]);
            block.voltageDacGain = Double.valueOf(status[STAT_VOLTAGE_DAC_GAIN]);
            block.voltageDacOffset = Double.valueOf(status[STAT_VOLTAGE_DAC_OFFSET]);
            block.currentDacGain = Double.valueOf(status[STAT_CURRENT_DAC_GAIN]);
            block.currentDacOffset = Double.valueOf(status[STAT_CURRENT_DAC_OFFSET]);
            block.ovpDacGain = Double.valueOf(status[STAT_OVP_DAC_GAIN]);
            block.ovpDacOffset = Double.valueOf(status[STAT_OVP_DAC_OFFSET]);
            block.voltageAdcGain = Double.valueOf(status[STAT_VOLTAGE_ADC_GAIN]);
            block.voltageAdcOffset = Double.valueOf(status[STAT_VOLTAGE_ADC_OFFSET]);
            block.currentAdcGain = Double.valueOf(status[STAT_CURRENT_ADC_GAIN]);
            block.currentAdcOffset = Double.valueOf(status[STAT_CURRENT_ADC_OFFSET]);
            block.model = status[STAT_MODEL];
            block.ovpCalibrated = (status.length > STAT_OVP_CALIBRATED)
                                    ? status[STAT_OVP_CALIBRATED].equals("1")
                                    : false;
            return block;
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


    /**
     *  Decodes a hexadecimal string.
     *
     *  @param  value  The string of hexadecimal digits.
     *  @return  The decoded value.
     *  @throws  DriverException
     */
    private int fromHex(String value) throws DriverException
    {
        try {
            return Integer.decode("0x" + value);
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }

}
