package org.lsst.ccs.drivers.sorensen;

import java.io.IOException;
import java.io.PrintStream;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ****************************************************************************** 
 ** 
 **  Program to monitor a Sorensen power supply
 ** 
 **  @author Owen Saxton
 ** 
 ****************************************************************************** 
 */
public class MonitorDcs {

    private final static PrintStream out = System.out;
    private final Dcs dcs = new Dcs();
    private final Timer monTimer = new Timer();
    private double period = 5;
    private boolean running;
    private PrintStream log;


   /**
    *************************************************************************** 
    ** 
    **  Inner class to run the monitor.
    **
    *************************************************************************** 
    */
    class Monitor extends TimerTask {

        @Override
        public void run()
        {
            if (!running) {
                cancel();
            }
            else {
                double voltage, current;
                try {
                    voltage = dcs.readVoltage();
                    current = dcs.readCurrent();
                    out.print("\r");
                    out.format("%.3f  %.3f  ", voltage, current);
                    if (log != null) {
                        Date ts = new Date();
                        log.format("%tY-%<tm-%<td\t%<tH:%<tM:%<tS\t%.3f\t%.3f\n",
                                   ts, voltage, current);
                    }
                }
                catch (DriverException e) {
                    out.println("Cannot read data: " + e);
                    cancel();
                }
            }
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor.
    **
    **  Opens connection to the power supply.
    ** 
    *************************************************************************** 
    */
    public MonitorDcs() throws DriverException
    {
        dcs.open(Scpi.CONN_TYPE_SERIAL, "/dev/ttyUSB0", 19200);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="voltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        dcs.setVoltage(value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="voltage", description="Read the actual voltage")
    public double readVoltage() throws DriverException
    {
        return dcs.readVoltage();
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current.
    ** 
    *************************************************************************** 
    */
    @Command(name="current", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        dcs.setCurrent(value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual current.
    ** 
    *************************************************************************** 
    */
    @Command(name="current", description="Read the actual current")
    public double readCurrent() throws DriverException
    {
        return dcs.readCurrent();
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the sampling period.
    ** 
    *************************************************************************** 
    */
    @Command(name="period", description="Set the sampling period")
    public void setPeriod(@Argument(name="period", description="Sampling period")
                          double period)
    {
        this.period = period;
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the sampling period.
    ** 
    *************************************************************************** 
    */
    @Command(name="period", description="Show the sampling period")
    public double showPeriod()
    {
        return(period);
    }


   /**
    *************************************************************************** 
    ** 
    **  Starts the monitoring.
    ** 
    *************************************************************************** 
    */
    @Command(name="start", description="Start the monitoring")
    public void start()
    {
        running = true;
        monTimer.schedule(new Monitor(), 0, (long)(1000 * period));
    }


   /**
    *************************************************************************** 
    ** 
    **  Stops the monitoring.
    ** 
    *************************************************************************** 
    */
    @Command(name="stop", description="Stop the monitoring")
    public void stop()
    {
        running = false;
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens the log file.
    ** 
    *************************************************************************** 
    */
    @Command(name="open", description="Open the log file")
    public void open(@Argument(name="name", description="Log file name")
                     String name) throws IOException
    {
        close();
        log = new PrintStream(name);
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes the log file.
    ** 
    *************************************************************************** 
    */
    @Command(name="close", description="Close the log file")
    public void close() throws IOException
    {
        PrintStream logx = log;
        if (logx != null) {
            log = null;
            logx.close();
        }
    }

}
