package org.lsst.ccs.drivers.serial;

import java.util.Arrays;
import jssc.SerialPortException;
import jssc.SerialPortTimeoutException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 * A serial port driver.
 * 
 * Currently implemented on top of JSSC.
 * https://code.google.com/p/java-simple-serial-connector/
 * 
 * @author aubourg
 * 
 */
public class SerialPort {

    jssc.SerialPort portImpl;

    /**
     * Constructor.
     */
    public SerialPort() {
    }

    /**
     * Constructor. Opens the port.
     * 
     * @param name
     *            the os-dependant name of the port (eg /dev/ttyXX).
     *
     * @throws DriverException
     */
    public SerialPort(String name) throws DriverException {
        this();
        openPort(name);
    }

    /**
     * Opens the port.
     * 
     * @param name
     *            the os-dependent name of the port (eg /dev/ttyXX).
     *
     * @throws DriverException
     */
    public void openPort(String name) throws DriverException {
        if (portImpl != null)
            throw new IllegalStateException("Device connection already open");
        portImpl = new jssc.SerialPort(name);
        try {
            portImpl.openPort();
        } catch (SerialPortException e) {
            throw new DriverException("cannot open port", e);
        }
    }

    /**
     * closes the port.
     *
     * @return whether or not the close was successful
     *
     * @throws DriverException
     */
    public boolean closePort() throws DriverException {
        try {
            if (portImpl.closePort()) {
                portImpl = null;
                return true;
            }
            return false;
        } catch (SerialPortException e) {
            throw new DriverException("cannot close port", e);
        }
    }

    /**
     * get the number of bytes available in the buffer (ie non blocking)
     * 
     * @return the number of bytes.
     *
     * @throws DriverException
     */
    public int getInputBufferBytesCount() throws DriverException {
        try {
            return portImpl.getInputBufferBytesCount();
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * read as many bytes as available without blocking.
     * 
     * @return the newly allocated byte array.
     *
     * @throws DriverException
     */
    public byte[] readBytes() throws DriverException {
        try {
            return portImpl.readBytes();
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * read bytes from the serial port. If not enough bytes are available might
     * fail by throwing a TimeoutException after "timeout" milliseconds.
     * 
     * @param byteCount
     *            the number of bytes to read
     * @param timeout
     *            milliseconds before failing on timeout
     * @return the newly allocated byte array.
     *
     * @throws DriverException
     */
    public byte[] readBytes(int byteCount, int timeout) throws DriverException {
        try {
            return portImpl.readBytes(byteCount, timeout);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        } catch (SerialPortTimeoutException e) {
            throw new DriverTimeoutException("serial port timeout", e);
        }
    }

    /**
     * read a given number of bytes from the serial port.
     * 
     * @param byteCount
     *            the number of bytes to read
     * @return the newly allocated byte array.
     *
     * @throws DriverException
     */
    public byte[] readBytes(int byteCount) throws DriverException {
        try {
            return portImpl.readBytes(byteCount);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * read all available bytes from the serial port, return as a String
     * 
     * @return the newly allocated String.
     *
     * @throws DriverException
     */
    public String readString() throws DriverException {
        try {
            return portImpl.readString();
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * read bytes from the serial port. If not enough bytes are available might
     * fail by throwing a TimeoutException after "timeout" milliseconds.
     * 
     * @param byteCount
     *            the number of bytes to read
     * @param timeout
     *            milliseconds before failing on timeout
     * @return the newly allocated byte array.
     *
     * @throws DriverException
     */
    public String readString(int byteCount, int timeout) throws DriverException {
        try {
            return portImpl.readString(byteCount, timeout);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        } catch (SerialPortTimeoutException e) {
            throw new DriverTimeoutException("serial port timeout", e);
        }
    }

    /**
     * read bytes from the serial port.
     * 
     * @param byteCount
     *            the number of bytes to read
     * @return the newly allocated byte array.
     *
     * @throws DriverException
     */
    public String readString(int byteCount) throws DriverException {
        try {
            return portImpl.readString(byteCount);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * set the flow control mode.
     * 
     * @param mask
     *            bitfield
     * 
     *            0 : NONE
     * 
     *            1 : RTSCTS_IN
     * 
     *            2 : RTSCTS_OUT
     * 
     *            4 : XONXOFF_IN
     * 
     *            8 : XONXOFF_OUT
     * 
     * @return true if everything was ok.
     *
     * @throws DriverException
     */
    public boolean setFlowControlMode(int mask) throws DriverException {
        try {
            return portImpl.setFlowControlMode(mask);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * Set the communication parameters
     * 
     * @param baudRate
     *            baud rate (110, 300, 600, 1200, 4800, 9600, 14400, 19200,
     *            38400, 57600, 115200, 128000, 256000)
     * @param dataBits
     *            data bits (5, 6, 7, 8)
     * @param stopBits
     *            stop bits (1, 2, 3 for 1.5)
     * @param parity
     *            parity (0 none, 1 odd, 2 even, 3 mark, 4 space)
     * @return true if everything is ok
     *
     * @throws DriverException
     */
    public boolean setParams(int baudRate, int dataBits, int stopBits,
            int parity) throws DriverException {
        try {
            return portImpl.setParams(baudRate, dataBits, stopBits, parity);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * sends a single byte
     * 
     * @param singleByte the byte to send
     * 
     * @return whether everything was okay
     *
     * @throws DriverException
     */
    public boolean writeByte(byte singleByte) throws DriverException {
        try {
            return portImpl.writeByte(singleByte);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * sends a byte array
     * 
     * @param buffer the byte array to send
     *
     * @return whether everything is okay
     *
     * @throws DriverException
     */
    public boolean writeBytes(byte[] buffer) throws DriverException {
        try {
            return portImpl.writeBytes(buffer);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * sends part of a byte array
     * 
     * @param buffer the byte array to send
     *
     * @param offset the offset of the first byte to send
     *
     * @param leng   the number of bytes to send
     *
     * @return whether everything is okay
     *
     * @throws DriverException
     */
    public boolean writeBytes(byte[] buffer, int offset, int leng)
        throws DriverException {
        try {
            return portImpl.writeBytes(Arrays.copyOfRange(buffer, offset,
                                                          offset + leng));
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * sends a String as byte array
     * 
     * @param string the string to send
     *
     * @return whether everything is okay
     *
     * @throws DriverException
     */
    public boolean writeString(String string) throws DriverException {
        try {
            return portImpl.writeString(string);
        } catch (SerialPortException e) {
            throw new DriverException("serial port error", e);
        }
    }

    /**
     * sends a String + CR + LF as byte array
     * 
     * @param s the string to send
     *
     * @throws DriverException
     */
    public void writeStringCRLF(String s) throws DriverException {
        StringBuilder cmd = new StringBuilder(s);
        cmd.append((char) 13);
        cmd.append((char) 10);
        writeString(cmd.toString());
    }

    /**
     * sends a String + LF as byte array
     * 
     * @param s the string to send
     *
     * @throws DriverException
     */
    public void writeStringLF(String s) throws DriverException {
        StringBuilder cmd = new StringBuilder(s);
        cmd.append((char) 10);
        writeString(cmd.toString());
    }


    /**
     * sends a String + CR as byte array
     * 
     * @param s the string to send
     *
     * @throws DriverException
     */
    public void writeStringCR(String s) throws DriverException {
        StringBuilder cmd = new StringBuilder(s);
        cmd.append((char) 13);
        writeString(cmd.toString());
    }

    /**
     * reads a String until CR+LF occurs.
     * 
     * Default timeout of 1000 ms
     *
     * @return the read string (without the CRLF)
     *
     * @throws DriverException
     */
    public String readUntilCRLF() throws DriverException {
        return readUntilCRLF(1000);
    }

    /**
     * reads a String until LF occurs.
     * 
     * Default timeout of 1000 ms
     *
     * @return the read string (without the LF)
     *
     * @throws DriverException
     */
    public String readUntilLF() throws DriverException {
        return readUntilLF(1000);
    }

    /**
     * reads a String until "> " or "< " occurs. For Thorlabs Laser
     * 
     * Default timeout of 1000 ms
     *
     * @return the read string (without the "> " or "< ")
     *
     * @throws DriverException
     */
    public String readUntilSUPSP() throws DriverException {
        return readUntilSUPSP(1000);
    }

    /**
     * reads a String until "> " or "< " occurs, with explicit timeout
     * 
     * @param timeout
     *            The timeout in ms
     *
     * @return the read string (without the "> " or "< ")
     *
     * @throws DriverException
     */
    public String readUntilSUPSP(int timeout) throws DriverException {
        StringBuilder buf = new StringBuilder();
        while (true) {
            byte[] b = readBytes(1, timeout);
            if (b[0] == 62 || b[0] == 60) {
                byte sb = b[0];
                b = readBytes(1, timeout);
                if (b[0] == 32) {
                    return buf.toString();
                }
                buf.append((char) sb);
            }
            buf.append((char) b[0]);
        }
    }

    /**
     * reads a String until LF occurs, with explicit timeout
     * 
     * @param timeout
     *            The timeout in ms
     *
     * @return the read string (without the LF)
     *
     * @throws DriverException
     */
    public String readUntilLF(int timeout) throws DriverException {
        StringBuilder buf = new StringBuilder();
        while (true) {
            byte[] b = readBytes(1, timeout);
            if (b[0] == 10) {
               return buf.toString();
            }
            buf.append((char) b[0]);
        }
    }

    /**
     * reads a String until CR+LF occurs, with explicit timeout
     * 
     * @param timeout
     *            The timeout in ms
     *
     * @return the read string (without the CRLF)
     *
     * @throws DriverException
     */
    public String readUntilCRLF(int timeout) throws DriverException {
        StringBuilder buf = new StringBuilder();
        while (true) {
            byte[] b = readBytes(1, timeout);
            if (b[0] == 13) {
                byte sb = b[0];
                b = readBytes(1, timeout);
                if (b[0] == 10) {
                    return buf.toString();
                }
                buf.append((char) sb);
            }
            buf.append((char) b[0]);
        }
    }

    /**
     * Reads available data into a byte array.
     * 
     * @param buff
     *            The buffer to receive the response data
     * 
     * @param offset
     *            The offset to the first available byte in the buffer
     * 
     * @param timeout
     *            The timeout in ms
     *
     * @return The number of bytes read
     *
     * @throws DriverException
     */
    public int read(byte[] buff, int offset, int timeout) throws DriverException {
        return read(buff, offset, buff.length - offset, timeout);
    }

    /**
     * Reads available data into a byte array.
     * 
     * @param buff
     *            The buffer to receive the response data
     * 
     * @param offset
     *            The offset to the first available byte in the buffer
     *
     * @param mleng
     *            The maximum number of bytes to read
     *
     * @param timeout
     *            The timeout in ms
     *
     * @return The number of bytes read
     *
     * @throws DriverException
     */
    public int read(byte[] buff, int offset, int mleng, int timeout) throws DriverException {
        if (offset + mleng > buff.length) {
            mleng = buff.length - offset;
        }
        if (mleng <= 0) return 0;
        byte[] data;
        int leng = 0, count = getInputBufferBytesCount();
        if (count == 0) {
            if (timeout > 0) {
                data = readBytes(1, timeout);
            } else {
                data = readBytes(1);
            }
            buff[offset++] = data[0];
            leng = 1;
            mleng--;
            count = getInputBufferBytesCount();
        }
        if (count > mleng) {
            count = mleng;
        }
        if (count > 0) {
            data = readBytes(count);
            System.arraycopy(data, 0, buff, offset, count);
            leng += count;
        }

        return leng;
    }

}
