package org.lsst.ccs.drivers.scpi;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.TestAscii;

/**
 *  Program to test the SCPI device driver
 * 
 *  @author Owen Saxton
 */
public class TestScpi extends TestAscii {

    protected final Scpi scpi;


    /**
     *  Constructor.
     */
    public TestScpi()
    {
        super(new Scpi());
        scpi = (Scpi)dev;
    }


    /**
     *  Constructor.
     *
     *  @param  scpi  The SCPI device to use
     */
    public TestScpi(Scpi scpi)
    {
        super(scpi);
        this.scpi = scpi;
    }


    /**
     *  Writes a command with error checking.
     *
     *  @param  command  The command word
     *  @param  args     The command arguments
     *  @throws  DriverException
     */
    @Command(name="writecommand", description="Write command")
    public void writeCommand(@Argument(description="Command word")
                             String command,
                             @Argument(description="Arguments")
                             String... args) throws DriverException
    {
        scpi.writeCommand(makeCommandString(command, args));
    }


    /**
     *  Writes a command with error checking and reads the string response.
     * 
     *  @param  command  The command word
     *  @param  args     The command arguments
     *  @return  The response string
     *  @throws  DriverException
     */
    @Command(name="readstring", description="Write command and read string response")
    public String readString(@Argument(description="Command word")
                             String command,
                             @Argument(description="Arguments")
                             String... args) throws DriverException
    {
        return scpi.readString(makeCommandString(command, args));
    }


    /**
     *  Writes a command with error checking and reads the integer response.
     * 
     *  @param  command  The command word
     *  @param  args     The command arguments
     *  @return  The response string, converted to an integer
     *  @throws  DriverException
     */
    @Command(name="readinteger", description="Write command and read integer response")
    public int readInteger(@Argument(description="Command word")
                           String command,
                           @Argument(description="Arguments")
                           String... args) throws DriverException
    {
        return scpi.readInteger(makeCommandString(command, args));
    }


    /**
     *  Writes a command with error checking and reads the double response.
     * 
     *  @param  command  The command word
     *  @param  args     The command arguments
     *  @return  The response string, converted to a double float
     *  @throws  DriverException
     */
    @Command(name="readdouble", description="Write command and read double response")
    public double readDouble(@Argument(description="Command wprd")
                             String command,
                             @Argument(description="Arguments")
                             String... args) throws DriverException
    {
        return scpi.readDouble(makeCommandString(command, args));
    }


    /**
     *  Writes a command with error checking and reads the float response.
     * 
     *  @param  command  The command word
     *  @param  args     The command arguments
     *  @return  The response string, converted to a float
     *  @throws  DriverException
     */
    @Command(name="readfloat", description="Write command and read float response")
    public float readFloat(@Argument(description="Command word")
                           String command,
                           @Argument(description="Arguments")
                           String... args) throws DriverException
    {
        return scpi.readFloat(makeCommandString(command, args));
    }


    /**
     *  Clears saved status.
     * 
     *  @throws  DriverException
     */
    @Command(name = "clear", description="Clear saved status")
    public void clear() throws DriverException
    {
        scpi.clearStatus();
    }


    /**
     *  Resets to power-on state.
     * 
     *  @throws  DriverException
     */
    @Command(name="reset", description="Reset to power-on state")
    public void reset() throws DriverException
    {
        scpi.reset();
    }


    /**
     *  Enables operation complete reporting.
     * 
     *  @throws  DriverException
     */
    @Command(name="enableopc", description="Enable operation complete reporting")
    public void enableOpc() throws DriverException
    {
        scpi.enableOperationComplete();
    }


    /**
     *  Sets the event status enable register.
     * 
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setese", description="Set event status enable register")
    public void setEse(@Argument(description="Value to set")
                       int value) throws DriverException
    {
        scpi.setEventStatusEnable(value);
    }


    /**
     *  Sets the service request enable register.
     *
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setsre", description="Set service request enable register")
    public void setSre(@Argument(description="Value to set")
                       int value) throws DriverException
    {
        scpi.setServiceRequestEnable(value);
    }


    /**
     *  Waits for command completion.
     * 
     *  @throws  DriverException
     */
    @Command(name="wait", description="Wait for previous command completion")
    public void waitComp() throws DriverException
    {
        scpi.waitCompletion();
    }


    /**
     *  Displays the event status register.
     * 
     *  @return  The event status register as a hexadecimal string
     *  @throws  DriverException
     */
    @Command(name="showesr", description="Display the event status register")
    public String showEsr() throws DriverException
    {
        return String.format("0x%02x", scpi.getEventStatus());
    }


    /**
     *  Displays the event status enable register.
     * 
     *  @return  The event status enable register as a hexadecimal string
     *  @throws  DriverException
     */
    @Command(name="showese", description="Display the event status enable register")
    public String showEse() throws DriverException
    {
        return String.format("0x%02x", scpi.getEventStatusEnable());
    }


    /**
     *  Displays the operation complete state.
     * 
     *  @return  The operation complete state
     *  @throws  DriverException
     */
    @Command(name="showopc", description="Display the operation complete state")
    public int showOpc() throws DriverException
    {
        return scpi.getOperationComplete();
    }


    /**
     *  Displays the service request enable register.
     * 
     *  @return  The service request enable register as a hexadecimal string
     *  @throws  DriverException
     */
    @Command(name="showsre", description="Display the service request enable register")
    public String showSre() throws DriverException
    {
        return String.format("0x%02x", scpi.getServiceRequestEnable());
    }


    /**
     *  Displays the status byte.
     * 
     *  @return  The status byte as a hexadecimal string
     *  @throws  DriverException
     */
    @Command(name="showstatus", description="Display the status byte")
    public String showStatus() throws DriverException
    {
        return String.format("0x%02x", scpi.getStatusByte());
    }


    /**
     *  Runs the self test.
     * 
     *  @return  The status of the self test
     *  @throws  DriverException
     */
    @Command(name="selftest", description="Run the self test")
    public int selfTest() throws DriverException
    {
        return scpi.runSelfTest();
    }


    /**
     *  Displays the first error item.
     * 
     *  @return  The first error item as a string
     *  @throws  DriverException
     */
    @Command(name="showerror", description="Display the first error item")
    public String showError() throws DriverException
    {
        return scpi.getError();
    }


    /**
     *  Displays SCPI device identification.
     * 
     *  @return  The device information as a string
     *  @throws  DriverException
     */
    @Command(name="showident", description="Show SCPI device identification")
    public String showIdentification() throws DriverException
    {
        String[] id = scpi.getIdentification();
        return   "Manufacturer:   " + id[Scpi.IDENT_MANUFACTURER] +
               "\nModel name:     " + id[Scpi.IDENT_MODEL_NAME] +
               "\nSerial number:  " + id[Scpi.IDENT_SERIAL_NUMBER] +
               "\nF/W version:    " + id[Scpi.IDENT_FW_VERSION];
    }

}
