package org.lsst.ccs.drivers.scpi;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 *****************************************************************************
 **
 **  General access routines for a SCPI device.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class Scpi extends Ascii {

   /**
    **************************************************************************
    **
    **  Public constants.
    **
    **************************************************************************
    */
    /** Identification field index - manufacturer */
    public final static int IDENT_MANUFACTURER  = 0;

    /** Identification field index - model name */
    public final static int IDENT_MODEL_NAME    = 1;

    /** Identification field index - serial number */
    public final static int IDENT_SERIAL_NUMBER = 2;

    /** Identification field index - firmware version */
    public final static int IDENT_FW_VERSION    = 3;

    /** Identification check type - equals string */
    public final static int CHECK_EQUALS      = 0;
   
    /** Identification check type - starts with string */
    public final static int CHECK_STARTS_WITH = 1;

    /** Identification check type - ends with string */
    public final static int CHECK_ENDS_WITH   = 2;

    /** Identification check type - contains string */
    public final static int CHECK_CONTAINS    = 3;

    /** Identification check type - matches regular expression */
    public final static int CHECK_MATCHES     = 4;

    /** Event status register bit mask - query error */
    public final static int ESR_QUERY_ERROR = 0x04;

    /** Event status register bit mask - device dependent error */
    public final static int ESR_DEVICE_ERROR = 0x08;

    /** Event status register bit mask - execution error */
    public final static int ESR_EXECUTION_ERROR = 0x10;

    /** Event status register bit mask - command error */
    public final static int ESR_COMMAND_ERROR = 0x20;

    /** Event status register bit mask - any error */
    public final static int ESR_ANY_ERROR = ESR_QUERY_ERROR | ESR_DEVICE_ERROR
                                            | ESR_EXECUTION_ERROR | ESR_COMMAND_ERROR;

    /** Status byte bit mask - error queue not empty */
    public final static int SB_ERROR_QUEUE = 0x04;

    /** Status byte bit mask - questionable status summary */
    public final static int SB_QUESTIONABLE_STATUS = 0x08;

    /** Status byte bit mask - message available */
    public final static int SB_MESSAGE_AVAILABLE = 0x10;

    /** Status byte bit mask - event status byte summary */
    public final static int SB_EVENT_STATUS = 0x20;

    /** Status byte bit mask - master status summary */
    public final static int SB_MASTER_STATUS = 0x40;

    /** Status byte bit mask - operation status summary */
    public final static int SB_OPERATION_STATUS = 0x80;


   /**
    **************************************************************************
    **
    **  Writes a command with SCPI error checking.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void writeCommand(String command) throws DriverException
    {
        flush();
        write("*CLS;" + command);
        checkError();
    }


   /**
    **************************************************************************
    **
    **  Reads a string with SCPI error checking after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned string value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public synchronized String readString(String command) throws DriverException
    {
        try {
            return read("*CLS;" + command);
        }
        catch (DriverTimeoutException e) {
            checkError();
            throw e;
        }
    }


   /**
    **************************************************************************
    **
    **  Reads a string array with SCPI error checking after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned array of string values
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public String[] readStringArray(String command) throws DriverException
    {
        return readString("*CLS;" + command).split(",");
    }


   /**
    **************************************************************************
    **
    **  Reads an integer with SCPI error checking after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned integer value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int readInteger(String command) throws DriverException
    {
        try {
            return Integer.valueOf(readString(command).trim());
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads an integer array with SCPI error checking after writing a
    **  command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned array of integer values
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int[] readIntegerArray(String command) throws DriverException
    {
        try {
            String[] reply = readStringArray(command);
            int[] iReply = new int[reply.length];
            for (int j = 0; j < iReply.length; j++) {
                iReply[j] = Integer.valueOf(reply[j].trim());
            }
            return iReply;
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads a double with SCPI error checking after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned double float value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public double readDouble(String command) throws DriverException
    {
        try {
            return Double.valueOf(readString(command));
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads a double array with SCPI error checking after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned array of double float values
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public double[] readDoubleArray(String command) throws DriverException
    {
        try {
            String[] reply = readStringArray(command);
            double[] dReply = new double[reply.length];
            for (int j = 0; j < dReply.length; j++) {
                dReply[j] = Double.valueOf(reply[j]);
            }
            return dReply;
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads a float with SCPI error checking after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned float value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public float readFloat(String command) throws DriverException
    {
        try {
            return Float.valueOf(readString(command));
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads a float array with SCPI error checking after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned array of float float values
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public float[] readFloatArray(String command) throws DriverException
    {
        try {
            String[] reply = readStringArray(command);
            float[] fReply = new float[reply.length];
            for (int j = 0; j < fReply.length; j++) {
                fReply[j] = Float.valueOf(reply[j]);
            }
            return fReply;
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Tests the identification.
    **
    **  @param  manufact  The manufacturer to check for, or null if not
    **                    checking.
    **
    **  @param  manCheck  The type of check to make for the manufacturer.
    **
    **  @param  model     The model to check for, or null if not checking.
    **
    **  @param  modCheck  The type of check to make for the model.
    **
    **  @return  Whether or not the identification matches
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public boolean testIdentification(String manufact, int manCheck,
                                      String model, int modCheck)
        throws DriverException
    {
        String[] id = getIdentification();

        return checkString(id[IDENT_MANUFACTURER], manufact, manCheck)
                 && checkString(id[IDENT_MODEL_NAME], model, modCheck);
    }


   /**
    **************************************************************************
    **
    **  Checks the identification.
    **
    **  @param  manufact  The manufacturer to check for, or null if not
    **                    checking.
    **
    **  @param  manCheck  The type of check to make for the manufacturer.
    **
    **  @param  model     The model to check for, or null if not checking.
    **
    **  @param  modCheck  The type of check to make for the model.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void checkIdentification(String manufact, int manCheck,
                                    String model, int modCheck)
        throws DriverException
    {
        String[] id = getIdentification();
        if (!checkString(id[IDENT_MANUFACTURER], manufact, manCheck)
              || !checkString(id[IDENT_MODEL_NAME], model, modCheck)) {
            closeSilent();
            throw new DriverException("Wrong device: " + id[IDENT_MANUFACTURER]
                                      + ":" + id[IDENT_MODEL_NAME]);
        }
    }


   /**
    **************************************************************************
    **
    **  Clears all status reporting structures.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void clearStatus() throws DriverException
    {
        write("*CLS");
    }


   /**
    **************************************************************************
    **
    **  Sets the event status enable register.
    **
    **  @param  value  The value to set
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void setEventStatusEnable(int value) throws DriverException
    {
        write("*ESE " + value);
    }


   /**
    **************************************************************************
    **
    **  Gets the event status enable register value.
    **
    **  @return  The event status enable register value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int getEventStatusEnable() throws DriverException
    {
        return readInt("*ESE?");
    }


   /**
    **************************************************************************
    **
    **  Gets the event status register value.
    **
    **  @return  The event status register value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int getEventStatus() throws DriverException
    {
        return readInt("*ESR?");
    }


   /**
    **************************************************************************
    **
    **  Gets the device identification strings.
    **
    **  @return  A four-element array containing, in order:
    **             Manufacturer;
    **             Model name;
    **             Serial number;
    **             Firmware version.
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public String[] getIdentification() throws DriverException
    {
        int ID_LENG = 4;

        String[] ident = readStringArray("*IDN?");
        if (ident.length == ID_LENG) {
            return ident;
        }
        else {
            String[] idCopy = new String[ID_LENG];
            for (int j = 0; j < ID_LENG; j++) {
                idCopy[j] = (j < ident.length) ? ident[j] : "";
            }
            return idCopy;
        }
    }


   /**
    **************************************************************************
    **
    **  Enables operation complete reporting.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void enableOperationComplete() throws DriverException
    {
        write("*OPC");
    }


   /**
    **************************************************************************
    **
    **  Gets the operation complete state.
    **
    **  @return  The operation complete state
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int getOperationComplete() throws DriverException
    {
        return readInt("*OPC?");
    }


   /**
    **************************************************************************
    **
    **  Resets to power-on state.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void reset() throws DriverException
    {
        write("*RST");
    }


   /**
    **************************************************************************
    **
    **  Sets the service request enable register.
    **
    **  @param  value  The value to set
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void setServiceRequestEnable(int value) throws DriverException
    {
        write("*SRE " + value);
    }


   /**
    **************************************************************************
    **
    **  Gets the service request enable register value.
    **
    **  @return  The service request enable register value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int getServiceRequestEnable() throws DriverException
    {
        return readInt("*SRE?");
    }


   /**
    **************************************************************************
    **
    **  Gets the status byte value.
    **
    **  @return  The status byte value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int getStatusByte() throws DriverException
    {
        return readInt("*STB?");
    }


   /**
    **************************************************************************
    **
    **  Executes internal self-test.
    **
    **  @return  The result of the self-test
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int runSelfTest() throws DriverException
    {
        return readInt("*TST?");
    }


   /**
    **************************************************************************
    **
    **  Waits for previous command completion.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void waitCompletion() throws DriverException
    {
        write("*WAI");
    }


   /**
    **************************************************************************
    **
    **  Gets the first item from the error queue.
    **
    **  @return  The error queue item
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public String getError() throws DriverException
    {
        return read("SYST:ERR?");
    }


   /**
    **************************************************************************
    **
    **  Makes a channel list string from a channel number.
    **
    **  @param  chan  The channel number
    **
    **  @return  The created channel list
    **
    **************************************************************************
    */
    public String makeChannelList(int chan)
    {
        return " (@" + chan + ")";
    }


   /**
    **************************************************************************
    **
    **  Makes a channel list string from an array of channels.
    **
    **  @param  chan  The array of channel numbers
    **
    **  @return  The created channel list
    **
    **************************************************************************
    */
    public String makeChannelList(int[] chan)
    {
        if (chan.length == 0) return "";
        StringBuilder list = new StringBuilder(" (@" + chan[0]);
        for (int j = 1; j < chan.length; j++) {
            list.append(",").append(chan[j]);
        }
        list.append(")");

        return list.toString();
    }


   /**
    **************************************************************************
    **
    **  Reads an integer after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The returned integer value
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    private int readInt(String command) throws DriverException
    {
        try {
            return Integer.decode(read(command));
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Checks for command error.
    **
    **************************************************************************
    */
    private void checkError() throws DriverException
    {
        try {
            if ((getEventStatus() & ESR_ANY_ERROR) == 0) return;
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
        throw new DriverException("SCPI error: " + getError());
    }


   /**
    **************************************************************************
    **
    **  Checks one string against another in the specified way.
    **
    **  @param  string1  String to be checked, or null to pass check
    **
    **  @param  string2  String to be checked against
    **
    **  @param  type     The type of check to make
    **
    **  @return  Whether the check is successful or not
    **
    **************************************************************************
    */
    private static boolean checkString(String string1, String string2, int type)
    {
        if (string1 == null) return true;

        switch (type) {
        case CHECK_EQUALS:
            return string1.equals(string2);
        case CHECK_STARTS_WITH:
            return string1.startsWith(string2);
        case CHECK_ENDS_WITH:
            return string1.endsWith(string2);
        case CHECK_CONTAINS:
            return string1.contains(string2);
        case CHECK_MATCHES:
            return string1.matches(string2);
        default:
            return string1.equals(string2);
        }
    }

}
