package org.lsst.ccs.drivers.scpi;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 **
 **  Program to enable direct talking to a SCPI device.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TalkScpi {

    private final static PrintStream out = System.out;
    private final static String[] terms = {"\r\n", "\r", "\n"};
    private final Scpi scpi = new Scpi();

    private enum ConnType {

        NET(Scpi.CONN_TYPE_NETWORK),
        FTDI(Scpi.CONN_TYPE_FTDI),
        SERIAL(Scpi.CONN_TYPE_SERIAL);

        int value;

        ConnType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    private enum Terminator {
        CRLF, CR, LF;
    }


   /**
    ***************************************************************************
    **
    **  Inner class for holding command line options.
    **
    ***************************************************************************
    */
    private static class Options {

        @Option(name="-c", metaVar="<connType>",
                usage="connection type: net (default), ftdi or serial")
        private ConnType type = ConnType.NET;

        @Option(name="-p", metaVar="<parity>",
                usage="Parity: none (default), odd, even, mark or space")
        private Scpi.Parity parity = Scpi.Parity.NONE;

        @Option(name="-d", metaVar="<dbits>",
                usage="No. data bits: eight (default) or seven")
        private Scpi.DataBits dataBits = Scpi.DataBits.EIGHT;

        @Option(name="-s", metaVar="<sbits>",
                usage="No. stop bits: one (default) or two")
        private Scpi.StopBits stopBits = Scpi.StopBits.ONE;

        @Option(name="-f", metaVar="<flow>",
                usage="Flow control: none (default), rts, dtr or xon")
        private Scpi.FlowCtrl flowCtrl = Scpi.FlowCtrl.NONE;

        @Option(name="-r", metaVar="<timeout>",
                usage="Read timeout (sec) (default = 1.0)")
        private double timeout = 1.0;

        @Option(name="-t", metaVar="<term>",
                usage="Command terminator: crlf (default), cr or lf")
        private Terminator term = Terminator.CRLF;

        @Option(name="-help", usage="Displays this help")
        private boolean showHelp;

        @Argument(index=0, required=true, metaVar="<ident>",
                  usage="Device identifier (node, serial no. or device name)")
        private String ident;

        @Argument(index=1, required=true, metaVar="<parm>",
                  usage="Device parameter (port number or baud rate)")
        private int parm;

    }


   /**
    ***************************************************************************
    **
    **  Main program.
    **
    **  @param  args  Command-line arguments
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        Options optns = new Options();
        CmdLineParser parser = new CmdLineParser(optns);
        try {
            parser.parseArgument(args);
            if (optns.showHelp) {
                showHelp(parser);
            }
            else {
                (new TalkScpi()).run(optns);
            }
        }
        catch (CmdLineException e) {
            if (!optns.showHelp) {
                out.println(e.getMessage());
            }
            showHelp(parser);
        }
        catch (Exception e) {
            out.println(e);
        }
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Sends commands to the device.
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    private void run(Options optns) throws Exception
    {
        scpi.open(optns.type.getValue(), optns.ident, optns.parm,
                  scpi.makeDataCharacteristics(optns.dataBits, optns.stopBits,
                                               optns.parity, optns.flowCtrl));
        scpi.setTimeout(optns.timeout);
        scpi.setTerminator(terms[optns.term.ordinal()]);
        ConsoleReader readC = new ConsoleReader();
        out.println("Connected. Type CTRL-D to exit.");

        while (true) {
            String line = readC.readLine(">> ");
            if (line == null) break;
            if (line.equals("")) continue;
            try {
                if (line.endsWith("?")) {
                    out.println(scpi.readString(line));
                }
                else {
                    scpi.writeCommand(line);
                }
            }
            catch (DriverException e) {
                out.println(e);
            }
        }

        out.println();
        scpi.close();
    }


   /**
    ***************************************************************************
    **
    **  Displays help.
    **
    ***************************************************************************
    */
    private static void showHelp(CmdLineParser parser)
    {
        out.println("Usage: TalkScpi [options...] <ident> <parm>");
        out.println("Parameters & options:");
        parser.printUsage(out);
    }

}
