package org.lsst.ccs.drivers.scpi;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *************************************************************************** 
 ** 
 **  Program to test the SCPI device driver
 ** 
 **  @author Owen Saxton
 ** 
 *************************************************************************** 
 */
public class TestScpi {

    protected Scpi scpi = new Scpi();

    public enum ConnType {

        NET(Scpi.CONN_TYPE_NETWORK),
        FTDI(Scpi.CONN_TYPE_FTDI),
        SERIAL(Scpi.CONN_TYPE_SERIAL);

        int value;

        ConnType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type",
                               description="Connection type: net, serial or ftdi")
                     ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident,
                     @Argument(name="param", description="Device parameter")
                     int param
                     ) throws DriverException
    {
        scpi.open(type.getValue(), ident, param);
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes connection to a device.
    ** 
    *************************************************************************** 
    */
    @Command(name="close", description="Close connection to device")
    public void close() throws DriverException
    {
        scpi.close();
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command.
    ** 
    *************************************************************************** 
    */
    @Command(name="write", description="Write command to device")
    public void write(@Argument(name="command", description="Command to write")
                      String command
                      ) throws DriverException
    {
        scpi.write(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads a response.
    ** 
    *************************************************************************** 
    */
    @Command(name="read", description="Read response from device")
    public String read() throws DriverException
    {
        return scpi.read();
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command and reads the response.
    ** 
    *************************************************************************** 
    */
    @Command(name="read", description="Write command and read response")
    public String read(@Argument(name="command", description="Command to write")
                       String command
                       ) throws DriverException
    {
        return scpi.read(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command with error checking.
    ** 
    *************************************************************************** 
    */
    @Command(name="writecommand",
             description="Write command")
    public void writeCommand(@Argument(name="command",
                                       description="Command to write")
                             String command
                             ) throws DriverException
    {
        scpi.writeCommand(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command with error checking and reads the string response.
    ** 
    *************************************************************************** 
    */
    @Command(name="readstring",
             description="Write command and read string response")
    public String readString(@Argument(name="command",
                                       description="Command to write")
                             String command
                             ) throws DriverException
    {
        return scpi.readString(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command with error checking and reads the integer response.
    ** 
    *************************************************************************** 
    */
    @Command(name="readinteger",
             description="Write command and read integer response")
    public int readInteger(@Argument(name="command",
                                     description="Command to write")
                           String command
                           ) throws DriverException
    {
        return scpi.readInteger(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command with error checking and reads the double response.
    ** 
    *************************************************************************** 
    */
    @Command(name="readdouble",
             description="Write command and read double response")
    public double readDouble(@Argument(name="command",
                                       description="Command to write")
                             String command
                             ) throws DriverException
    {
        return scpi.readDouble(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command with error checking and reads the float response.
    ** 
    *************************************************************************** 
    */
    @Command(name="readfloat",
             description="Write command and read float response")
    public float readFloat(@Argument(name="command",
                                     description="Command to write")
                           String command
                           ) throws DriverException
    {
        return scpi.readFloat(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the response timeout.
    ** 
    *************************************************************************** 
    */
    @Command(name="timeout", description="Set read response timeout")
    public void timeout(@Argument(name="time", description="Timeout value (sec)")
                        double time
                        ) throws DriverException
    {
        scpi.setTimeout(time);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the command terminator.
    ** 
    *************************************************************************** 
    */
    @Command(name="setterm", description="Set command terminator")
    public void setTerm(@Argument(name="term", description="Terminator string")
                        String term
                        ) throws DriverException
    {
        char[] cTerm = new char[term.length()];
        term.getChars(0, cTerm.length, cTerm, 0);
        int jout = 0;
        for (int jin = 0; jin < cTerm.length; jin++) {
            if (cTerm[jin] == '\\' && jin < cTerm.length - 1) {
                jin++;
                switch (cTerm[jin]) {
                case 'r': cTerm[jin] = '\r'; break;
                case 'n': cTerm[jin] = '\n'; break;
                case 't': cTerm[jin] = '\t'; break;
                case 'b': cTerm[jin] = '\b'; break;
                case 'f': cTerm[jin] = '\f'; break;
                }
            }
            cTerm[jout++] = cTerm[jin];
        }
        scpi.setTerminator(new String(cTerm, 0, jout));
    }


   /**
    *************************************************************************** 
    ** 
    **  Clears saved status.
    ** 
    *************************************************************************** 
    */
    @Command(name = "clear", description="Clear saved status")
    public void clear() throws DriverException
    {
        scpi.clearStatus();
    }


   /**
    *************************************************************************** 
    ** 
    **  Resets to power-on state.
    ** 
    *************************************************************************** 
    */
    @Command(name="reset", description="Reset to power-on state")
    public void reset() throws DriverException
    {
        scpi.reset();
    }


   /**
    *************************************************************************** 
    ** 
    **  Enables operation complete reporting.
    ** 
    *************************************************************************** 
    */
    @Command(name="enableopc",
             description="Enable operation complete reporting")
    public void enableOpc() throws DriverException
    {
        scpi.enableOperationComplete();
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the event status enable register.
    ** 
    *************************************************************************** 
    */
    @Command(name="setese", description="Set event status enable register")
    public void setEse(@Argument(name="value", description="Value to set")
                       int value
                       ) throws DriverException
    {
        scpi.setEventStatusEnable(value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the service request enable register.
    ** 
    *************************************************************************** 
    */
    @Command(name="setsre", description="Set service request enable register")
    public void setSre(@Argument(name="value", description="Value to set")
                       int value
                       ) throws DriverException
    {
        scpi.setServiceRequestEnable(value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Waits for command completion.
    ** 
    *************************************************************************** 
    */
    @Command(name="wait", description="Wait for previous command completion")
    public void waitComp() throws DriverException
    {
        scpi.waitCompletion();
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays the event status register.
    ** 
    *************************************************************************** 
    */
    @Command(name="showesr", description="Display the event status register")
    public String showEsr() throws DriverException
    {
        return String.format("0x%02x", scpi.getEventStatus());
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays the event status enable register.
    ** 
    *************************************************************************** 
    */
    @Command(name="showese",
             description="Display the event status enable register")
    public String showEse() throws DriverException
    {
        return String.format("0x%02x", scpi.getEventStatusEnable());
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays the operation complete state.
    ** 
    *************************************************************************** 
    */
    @Command(name="showopc", description="Display the operation complete state")
    public int showOpc() throws DriverException
    {
        return scpi.getOperationComplete();
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays the service request enable register.
    ** 
    *************************************************************************** 
    */
    @Command(name="showsre",
             description="Display the service request enable register")
    public String showSre() throws DriverException
    {
        return String.format("0x%02x", scpi.getServiceRequestEnable());
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays the status byte.
    ** 
    *************************************************************************** 
    */
    @Command(name="showstatus", description="Display the status byte")
    public String showStatus() throws DriverException
    {
        return String.format("0x%02x", scpi.getStatusByte());
    }


   /**
    *************************************************************************** 
    ** 
    **  Runs the self test.
    ** 
    *************************************************************************** 
    */
    @Command(name="selftest", description="Run the self test")
    public int selfTest() throws DriverException
    {
        return scpi.runSelfTest();
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays the first error item.
    ** 
    *************************************************************************** 
    */
    @Command(name="showerror", description="Display the first error item")
    public String showError() throws DriverException
    {
        return scpi.getError();
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays SCPI device identification.
    ** 
    *************************************************************************** 
    */
    @Command(name="showident", description="Show SCPI device identification")
    public String showIdentification() throws DriverException
    {
        String[] id = scpi.getIdentification();
        return   "Manufacturer:   " + id[Scpi.IDENT_MANUFACTURER] +
               "\nModel name:     " + id[Scpi.IDENT_MODEL_NAME] +
               "\nSerial number:  " + id[Scpi.IDENT_SERIAL_NUMBER] +
               "\nF/W version:    " + id[Scpi.IDENT_FW_VERSION];
    }

}
