package org.lsst.ccs.drivers.scpi;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ftdi.Ftdi;
import org.lsst.ccs.drivers.ftdi.FtdiException;

/**
 *****************************************************************************
 **
 **  Communications routines for serial SCPI devices connected via FTDI.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class ScpiCommFtdi implements ScpiComm {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    Ftdi ftdi = new Ftdi();


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  serial  The serial number string of the interface
    **
    **  @param  baud    The baud rate
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String serial, int baud) throws DriverException
    {
        try {
            ftdi.open(0, serial);
            ftdi.setBaudrate(baud);
        }
        catch (FtdiException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        try {
            ftdi.close();
        }
        catch (FtdiException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write, including the terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void write(byte[] command) throws DriverException
    {
        try {
            ftdi.write(command);
        }
        catch (FtdiException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads available response data.
    **
    **  @param  buff    The buffer to receive the response data
    **
    **  @param  offset  The offset to the first available byte in the buffer
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int read(byte[] buff, int offset) throws DriverException
    {
        try {
            int leng = 0, count = ftdi.getQueueStatus();
            if (count == 0) {
                leng = ftdi.read(buff, offset, 1);
                if (leng == 0) {
                    throw new DriverTimeoutException("Read timed out");
                }
                offset++;
                count = ftdi.getQueueStatus();
            }
            if (offset + count > buff.length) {
                count = buff.length - offset;
            }
            if (count > 0) {
                leng += ftdi.read(buff, offset, count);
            }
            return leng;
        }
        catch (FtdiException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Flushes any available response data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void flush() throws DriverException
    {
        try {
            int count = ftdi.getQueueStatus();
            if (count > 0) {
                byte[] buff = new byte[count];
                ftdi.read(buff, 0, count);
            }
        }
        catch (FtdiException e) {
            throw new DriverException(e);
        }
    }

    
   /**
    **************************************************************************
    **
    **  Sets the receive timeout.
    **
    **  @param  timeout  The receive timeout (ms).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        try {
            ftdi.setTimeouts(timeout, 0);
        }
        catch (FtdiException e) {
            throw new DriverException(e);
        }
    }

}
