package org.lsst.ccs.drivers.scpi;

import jssc.SerialPort;
import jssc.SerialPortException;
import jssc.SerialPortTimeoutException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *****************************************************************************
 **
 **  Communications routines for serial SCPI devices.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class ScpiCommSerial implements ScpiComm {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    private SerialPort port;
    private int timeout;


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  device  The device name of the serial port
    **
    **  @param  baud    The baud rate
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String device, int parm) throws DriverException
    {
        try {
            port = new SerialPort(device);
            port.openPort();
            port.setParams(parm, 8, 1, 0);
        }
        catch (SerialPortException e) {
            throw new DriverException(e.getMessage(), e);
        }
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        try {
            port.closePort();
        }
        catch (SerialPortException e) {
            throw new DriverException(e.getMessage(), e);
        }
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write, including the terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void write(byte[] command) throws DriverException
    {
        try {
            port.writeBytes(command);
        }
        catch (SerialPortException e) {
            throw new DriverException(e.getMessage(), e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads available response data.
    **
    **  @param  buff    The buffer to receive the response data
    **
    **  @param  offset  The offset to the first available byte in the buffer
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int read(byte[] buff, int offset) throws DriverException
    {
        try {
            byte[] data;
            int leng = 0, count = port.getInputBufferBytesCount();
            if (count == 0) {
                if (timeout > 0) {
                    data = port.readBytes(1, timeout);
                }
                else {
                    data = port.readBytes(1);
                }
                buff[offset++] = data[0];
                leng = 1;
                count = port.getInputBufferBytesCount();
            }
            if (offset + count > buff.length) {
                count = buff.length - offset;
            }
            if (count > 0) {
                data = port.readBytes(count);
                System.arraycopy(data, 0, buff, offset, count);
                leng += count;
            }
            return leng;
        }
        catch (SerialPortTimeoutException e) {
            throw new DriverTimeoutException("Read timed out");
        }
        catch (SerialPortException e) {
            throw new DriverException(e.getMessage(), e);
        }
    }


   /**
    **************************************************************************
    **
    **  Sets the receive timeout.
    **
    **  @param  timeout  The receive timeout (ms).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        this.timeout = timeout;
    }

}
