package org.lsst.ccs.drivers.scpi;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketTimeoutException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *****************************************************************************
 **
 **  Communications routines for network SCPI devices.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class ScpiCommNet implements ScpiComm {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    private Socket sock;
    private InputStream in;
    private OutputStream out;


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  host  The host name or IP address
    **
    **  @param  port  The port number
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String host, int port) throws DriverException
    {
        try {
            sock = new Socket(host, port);
            in = sock.getInputStream();
            out = sock.getOutputStream();
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        try {
            sock.close();
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write, including the terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void write(byte[] command) throws DriverException
    {
        try {
            out.write(command);
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Reads available response data.
    **
    **  @param  buff    The buffer to receive the response data
    **
    **  @param  offset  The offset to the first available byte in the buffer
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int read(byte[] buff, int offset) throws DriverException
    {
        try {
            int leng = in.read(buff, offset, 1);
            if (leng <= 0) {
                throw new DriverException("Connection closed");
            }
            offset += leng;
            int count = in.available();
            if (offset + count > buff.length) {
                count = buff.length - offset;
            }
            if (count > 0) {
                leng += in.read(buff, offset, count);
            }
            return leng;
        }
        catch (SocketTimeoutException e) {
            throw new DriverTimeoutException("Read timed out");
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    **************************************************************************
    **
    **  Sets the receive timeout.
    **
    **  @param  timeout  The receive timeout (ms).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        try {
            sock.setSoTimeout(timeout);
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }

}
