package org.lsst.ccs.drivers.rotator;

import java.io.Serializable;

/**
 * Rotator state represents the combined state of the rotator system, which includes
 * the RotatorSummaryState, the OfflineState, EnabledState and application state.
 * @author tonyj
 */
public class RotatorState implements Serializable {

    private static final long serialVersionUID = 1L;

    public enum RotatorSummaryState {
        STANDBY, DISABLED, ENABLED, OFFLINE, FAULT
    }

    public enum OfflineState {
        PUBLISH_ONLY, AVAILABLE
    }

    public enum EnabledState {
        STATIONARY, MOVING_POINT_TO_POINT, SLEWING_OR_TRACKING, CONTROLLED_STOP, INITIALIZING, RELATIVE, CONSTANT_VOLOCITY
    }
    public static final int HEX_FOLLOWING_ERROR_MASK = 0x00000001;
    public static final int HEX_MOVE_COMPLETE_MASK = 0x00000002;
    public static final int COMMAND_REJECT_MASK = 0x00000020;
    public static final int SAFETY_INTERLOCK = 0x00000040;
    public static final int EXTEND_LIMIT_SWITCH = 0x00000080;
    public static final int RETRACT_LIMIT_SWITCH = 0x00000100;
    public static final int ETHERCAT_PROBLEM = 0x00000200;
    public static final int DDS_COMMAND_SOURCE = 0x00000400;
    public static final int MOTION_TIMEOUT = 0x00000800;
    public static final int DRIVE_FAULT = 0x00002000;
    public static final int SIMULINK_FAULT = 0x00004000;
    public static final int ENCODER_FAULT = 0x00008000;
    private final RotatorSummaryState rotatorSummaryState;
    private final OfflineState offlineState;
    private final EnabledState enabledState;
    private final int applicationStatus;

    RotatorState(RotatorSummaryState rotatorSummaryState, OfflineState offlineState, EnabledState enabledState, int applicationStatus) {
        this.rotatorSummaryState = rotatorSummaryState;
        this.offlineState = offlineState;
        this.enabledState = enabledState;
        this.applicationStatus = applicationStatus;
    }
    
    /**
     * Rotator summary state. A RotatorSummaryState enumeration value. The state names and
     * meanings are the same as standard CSC summaryState, but the numeric
     * values are ordered differently. We report the controller state in
     * addition to summaryState because we need to report a summaryState
     * even when the controller is not connected, and to aid diagnostics,
     * e.g. in case the controller reports an unrecognized state.
     *
     * @return
     */
    public RotatorSummaryState getRotatorSummaryState() {
        return rotatorSummaryState;
    }

    /**
     * Substate in OFFLINE mode. An OfflineSubstate enumeration value.
     *
     * @return
     */
    public OfflineState getOfflineState() {
        return offlineState;
    }

    /**
     * Substate in ENABLED mode. An EnabledSubstate enumeration value.
     *
     * @return
     */
    public EnabledState getEnabledState() {
        return enabledState;
    }

    /**
     * Application state. A bitmask of ApplicationStatus values.
     *
     * @return
     */
    public int getApplicationStatus() {
        return applicationStatus;
    }

    @Override
    public String toString() {
        return "RotatorState{" + "controllerState=" + rotatorSummaryState + ", offlineState=" + offlineState + ", enabledState=" + enabledState + ", applicationStatus=" + Integer.toHexString(applicationStatus) + '}';
    }
    
    
}
