package org.lsst.ccs.drivers.rotator;

import java.util.concurrent.TimeoutException;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * Methods to be implemented by the driver. This is based on the functionality
 * documented in the SAL interface which is here:
 * https://github.com/lsst-ts/ts_xml/tree/develop/sal_interfaces/Rotator
 *
 * @author tonyj
 */
public interface RotatorInterface {

    /**
     * Move the rotator to the specified position and stop.During the move the
     * enabled substate will be MovingPointToPoint, then it will return to
     * Stationary.
     *
     * @param position The target position in degrees
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * @throws java.util.concurrent.TimeoutException
     */
    public void move(double position) throws DriverException, TimeoutException;

    /**
     * Stop any current rotation.
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * @throws java.util.concurrent.TimeoutException
     */
    public void stop() throws DriverException, TimeoutException;

    /**
     * Limit the minimum and maximum acceleration level of the rotator to be set
     * within the global limit of +/- 1 deg/s2.
     *
     * @param aLimit
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * @throws java.util.concurrent.TimeoutException
     */
    public void configureAcceleration(double aLimit) throws DriverException, TimeoutException;

    /**
     * The configure Velocity command limits the minimum and maximum velocity
     * level of the rotator to be set within the global limit of +/- 3.5 deg/s.
     *
     * @param vLimit
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * @throws java.util.concurrent.TimeoutException
     */
    public void configureVelocity(double vLimit) throws DriverException, TimeoutException;

    /**
     * Clear the Fault state, sending the controller into Offline substate
     * PublishOnly.You must use the engineering user interface to transition
     * from that to Offline substate Available before the CSC can control the
     * rotator.
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * @throws java.util.concurrent.TimeoutException
     */
    public void clearError() throws DriverException, TimeoutException;

    public void addStateChangeListener(RotatorStateChangeListener stateChangeListener);

    public void removeStateChangeListener(RotatorStateChangeListener stateChangeListener);

    public void addPositionListener(PositionListener positionListener);

    public void removePositionListener(PositionListener positionListener);
}
