package org.lsst.ccs.drivers.rotator;

import java.time.Duration;
import java.util.Date;
import java.util.List;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.rotator.RotatorState.RotatorSummaryState;
import org.lsst.ccs.drivers.rotator.RotatorState.EnabledState;
import org.lsst.ccs.drivers.rotator.RotatorState.OfflineState;
import org.lsst.sal.SALCommand;
import org.lsst.sal.SALCommandResponse;
import org.lsst.sal.SALEvent;
import org.lsst.sal.SALException;
import org.lsst.sal.SALTelemetry;
import org.lsst.sal.rotator.SALRotator;
import org.lsst.sal.rotator.command.ClearErrorCommand;
import org.lsst.sal.rotator.command.ConfigureAccelerationCommand;
import org.lsst.sal.rotator.command.ConfigureVelocityCommand;
import org.lsst.sal.rotator.command.DisableCommand;
import org.lsst.sal.rotator.command.EnableCommand;
import org.lsst.sal.rotator.command.EnterControlCommand;
import org.lsst.sal.rotator.command.ExitControlCommand;
import org.lsst.sal.rotator.command.MoveCommand;
import org.lsst.sal.rotator.command.PositionSetCommand;
import org.lsst.sal.rotator.command.StandbyCommand;
import org.lsst.sal.rotator.command.StartCommand;
import org.lsst.sal.rotator.command.StopCommand;
import org.lsst.sal.rotator.event.ControllerStateEvent;
import org.lsst.sal.rotator.event.HeartbeatEvent;
import org.lsst.sal.rotator.event.SummaryStateEvent;
import org.lsst.sal.rotator.event.SummaryStateEvent.SummaryState;
import org.lsst.sal.rotator.telemetry.ApplicationTelemetry;

/**
 * A driver for the camera rotator. Works by talking to the rotator controller
 * software using SAL.
 * 
 * @author tonyj
 */
public class RotatorDriver implements RotatorInterface, AutoCloseable {

    private final SALRotator mgr = SALRotator.create();
    private static final Logger LOG = Logger.getLogger(RotatorDriver.class.getName());
    private final Future<Void> eventListener;
    private final Future<Object> telemetryListener;
    private final BlockingDeque<Object> eventQueue = new LinkedBlockingDeque<>();
    private final Future<?> eventDelivery;
    private final List<PositionListener> positionListeners = new CopyOnWriteArrayList<>();
    private final List<RotatorStateChangeListener> stateListeners = new CopyOnWriteArrayList<>();
    private volatile long lastHeartbeat;
    private volatile SummaryStateEvent.SummaryState summaryState;
    private volatile RotatorState rotatorState;
    private volatile Position position;

    public RotatorDriver() {
        // Don't use common pool, because it may not have enough threads
        this(Executors.newFixedThreadPool(4));
    }

    public RotatorDriver(ExecutorService executor) {
        eventListener = executor.submit(() -> {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    SALEvent event = mgr.getNextEvent(Duration.ofMinutes(1));
                    LOG.log(Level.FINE, "Received {0}", event);
                    if (event == null) {
                        LOG.info("Still waiting for an event");
                    } else if (event instanceof HeartbeatEvent) {
                        HeartbeatEvent beat = (HeartbeatEvent) event;
                        lastHeartbeat = System.currentTimeMillis();
                    } else if (event instanceof SummaryStateEvent) {
                        SummaryStateEvent sse = (SummaryStateEvent) event;
                        summaryState = convertEnum(sse.getSummaryState() - 1, SummaryStateEvent.SummaryState.class);
                    } else if (event instanceof ControllerStateEvent) {
                        ControllerStateEvent cse = (ControllerStateEvent) event;
                        rotatorState = new RotatorState(convertEnum(cse.getControllerState(), RotatorSummaryState.class), convertEnum(cse.getOfflineSubstate(), OfflineState.class),
                                convertEnum(cse.getEnabledSubstate(), EnabledState.class), cse.getApplicationStatus());
                        eventQueue.offer(new RotatorStateChangedEvent(rotatorState));
                    } else {
                        LOG.log(Level.FINE, "Received unhandled {0}", event);
                    }
                } catch (Throwable x) {
                    LOG.log(Level.SEVERE, "Error ", x);
                }
            }
            return null;
        });
        telemetryListener = executor.submit(() -> {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    SALTelemetry telemetry = mgr.getTelemetry(Duration.ofMinutes(1));
                    LOG.log(Level.FINE, "Received {0}", telemetry);
                    if (telemetry == null) {
                        LOG.info("Still waiting for telemetry");
                    } else if (telemetry instanceof ApplicationTelemetry) {
                        ApplicationTelemetry at = (ApplicationTelemetry) telemetry;
                        Position newPosition = new Position(at.getDemand(), at.getPosition(), at.getError());
                        if (newPosition.isSignificantlyDifferent(position)) {
                            position = newPosition;
                            PositionEvent pe = new PositionEvent(position);
                            eventQueue.offer(pe);
                        }
                    } else {
                        LOG.log(Level.FINE, "Received unhandled {0}", telemetry);
                    }
                } catch (Throwable x) {
                    LOG.log(Level.SEVERE, "Error ", x);
                }
            }
            return null;
        });
        eventDelivery = executor.submit(() -> {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    Object event = eventQueue.take();
                    if (event instanceof PositionEvent) {
                        deliverPositionEvent((PositionEvent) event);
                    } else if (event instanceof RotatorStateChangedEvent) {
                        deliverStateChangedEvent((RotatorStateChangedEvent) event);
                    }
                } catch (Throwable x) {
                    LOG.log(Level.SEVERE, "Error ", x);
                }            
            }
            return null;
        });
    }

    @Override

    public void close() throws DriverException {
        try {
            eventListener.cancel(true);
            telemetryListener.cancel(true);
            eventDelivery.cancel(false);
            mgr.close();
        } catch (SALException ex) {
            throw new DriverException("Error during close", ex);
        }
    }

    @Override
    public void move(double position) throws DriverException, TimeoutException {
        executeCommand(new PositionSetCommand(position));
        executeCommand(new MoveCommand(true), Duration.ofSeconds(60));
    }

    private void executeCommand(SALCommand command) throws TimeoutException, DriverException {
        executeCommand(command, Duration.ofSeconds(60));
    }

    public void enable() throws DriverException, TimeoutException {
        executeCommand(new EnableCommand());
    }

    public void disable() throws DriverException, TimeoutException {
        executeCommand(new DisableCommand());
    }

    public void standby() throws DriverException, TimeoutException {
        executeCommand(new StandbyCommand());
    }

    public void start(String config) throws DriverException, TimeoutException {
        executeCommand(new StartCommand(config));
    }

    public void enterControl() throws DriverException, TimeoutException {
        executeCommand(new EnterControlCommand());
    }

    public void exitControl() throws DriverException, TimeoutException {
        executeCommand(new ExitControlCommand());
    }
    
    @Override
    public void stop() throws DriverException, TimeoutException {
        executeCommand(new StopCommand(0));
    }

    @Override
    public void configureAcceleration(double aLimit) throws DriverException, TimeoutException {
        executeCommand(new ConfigureAccelerationCommand(aLimit));
    }

    @Override
    public void configureVelocity(double vLimit) throws DriverException, TimeoutException {
        executeCommand(new ConfigureVelocityCommand(vLimit));
    }

    @Override
    public void clearError() throws TimeoutException, DriverException {
        executeCommand(new ClearErrorCommand(false));
    }

    @Override
    public void addStateChangeListener(RotatorStateChangeListener stateChangeListener) {
        stateListeners.add(stateChangeListener);
    }

    @Override
    public void removeStateChangeListener(RotatorStateChangeListener stateChangeListener) {
        stateListeners.remove(stateChangeListener);
    }

    private void deliverStateChangedEvent(RotatorStateChangedEvent stateChangeEvent) {
        for (RotatorStateChangeListener l : stateListeners) {
            l.stateChanged(stateChangeEvent);
        }
    }

    @Override
    public void addPositionListener(PositionListener positionListener) {
        positionListeners.add(positionListener);
    }

    @Override
    public void removePositionListener(PositionListener positionListener) {
        positionListeners.remove(positionListener);
    }

    private void deliverPositionEvent(PositionEvent positionEvent) {
        for (PositionListener l : positionListeners) {
            l.positionChanged(positionEvent);
        }
    }

    public Date getLastHeartbeat() {
        return new Date(lastHeartbeat);
    }

    public SummaryState getSummaryState() {
        return summaryState;
    }

    public RotatorState getRotatorState() {
        return rotatorState;
    }

    public Position getPosition() {
        return position;
    }

    private <T extends Enum> T convertEnum(int value, Class<T> enumClass) {
        T[] values = enumClass.getEnumConstants();
        if (value < 0 || value >= values.length) {
            throw new IllegalArgumentException("Invalid value " + value + " for enum " + enumClass);
        }
        return values[value];
    }

    private void executeCommand(SALCommand command, Duration timeout) throws TimeoutException, DriverException {
        try {
            SALCommandResponse response = mgr.issueCommand(command);
            int rc = response.waitForCompletion(timeout);
            if (rc != 303) {
                throw new DriverException("Command failed rc=" + rc);
            }
        } catch (SALException ex) {
            throw new DriverException("Error during command execution: "+command, ex);
        }
    }
}

