package org.lsst.ccs.drivers.reflex;

import java.io.Serializable;

import org.lsst.ccs.drivers.reflex.ReflexController.BoardType;
import org.lsst.ccs.utilities.logging.Logger;

public class ReflexStatus implements Serializable {

    private static final long serialVersionUID = 3374708539867878786L;

    int errorCode;
    int errorSource; // board number
    int errorLine;
    int statusIndex;
    int powerState;

    int boardTypes[] = new int[12]; // 0xttrrmmnn
    int romID[] = new int[12];
    int buildNumber[] = new int[12];
    int featureFlags[] = new int[12];

    Logger log = Logger.getLogger("org.lsst.reflex");

    BoardStatus boardStatus[] = new BoardStatus[12];

    public static class BoardStatus implements Serializable {
        private static final long serialVersionUID = -2316945244308302886L;
        int temperature;

        public void dump() {
            System.out.printf("temperature %2.3f\n",
                    (temperature - 273150) / 1000.);
        }
    }

    public static class BackplaneStatus extends BoardStatus {
        private static final long serialVersionUID = 3362594551204925405L;
        int status;

        public void dump() {
            super.dump();
            System.out.printf("status %08X\n", status);
        }

    }

    public static class InterfaceStatus extends BoardStatus {
        private static final long serialVersionUID = 3389851569787973971L;
    }

    public static class PowerAStatus extends BoardStatus {
        private static final long serialVersionUID = -7218331917285754309L;
        int V5VD, I5VD, V5VA, I5VA, Vm5V, Im5V;

        public void dump() {
            super.dump();
            System.out.printf("+5VD  %7.3f %7.3f\n", (V5VD / 1000.),
                    (I5VD / 1000.));
            System.out.printf("+5VA  %7.3f %7.3f\n", (V5VA / 1000.),
                    (I5VA / 1000.));
            System.out.printf("-5V   %7.3f %7.3f\n", (Vm5V / 1000.),
                    (Im5V / 1000.));
        }

    }

    public static class PowerBStatus extends BoardStatus {
        private static final long serialVersionUID = -2936950124927843267L;
        int V30V, I30V, V15V, I15V, Vm15V, Im15V;
        int TECSetPoint, TECActual;

        public void dump() {
            super.dump();
            System.out.printf("+30V  %7.3f %7.3f\n", (V30V / 1000.),
                    (I30V / 1000.));
            System.out.printf("+15V  %7.3f %7.3f\n", (V15V / 1000.),
                    (I15V / 1000.));
            System.out.printf("-15V  %7.3f %7.3f\n", (Vm15V / 1000.),
                    (Im15V / 1000.));
            System.out.printf("TEC   %7.3f %7.3f\n",
                    ((TECSetPoint - 273150) / 1000.),
                    ((TECActual - 273150) / 1000.));
        }

    }

    public static class AD8X120Status extends BoardStatus {
        private static final long serialVersionUID = -3168912407885345768L;
        int status;

        public void dump() {
            super.dump();
            System.out.printf("status %08X\n", status);
        }

    }

    public static class AD8X100Status extends BoardStatus {
        private static final long serialVersionUID = 5811084942281260759L;
        int status;

        public void dump() {
            super.dump();
            System.out.printf("status %08X\n", status);
        }

    }

    public static class DriverStatus extends BoardStatus {
        private static final long serialVersionUID = 3602032819063285937L;
        int status;

        public void dump() {
            super.dump();
            System.out.printf("status %08X\n", status);
        }

    }

    public static class BiasStatus extends BoardStatus {
        private static final long serialVersionUID = 7837103979251692628L;
        int status;
        int lv[] = new int[8];
        int hv[] = new int[8];
        int lc[] = new int[8];
        int hc[] = new int[8];

        public void dump() {
            super.dump();
            System.out.printf("status %08X\n", status);
            for (int i = 0; i < 8; i++) {
                System.out.printf("LV%d %7.3f %7.3f    HV%d %7.3f %7.3f\n",
                        i + 1, lv[i] / 1000., lc[i] / 1000., i + 1,
                        hv[i] / 1000., hc[i] / 1000.);
            }
        }

    }

    public BoardType getBoardType(int i) {
        int tt = boardTypes[i] >> 24;
        return BoardType.forType(tt);
    }

    public String getBoardRevision(int i) {
        int rr = (boardTypes[i] & 0x00FF0000) >> 16;
        int mm = (boardTypes[i] & 0x0000FF00) >> 8;
        int nn = (boardTypes[i] & 0x000000FF);
        char c = Character.toChars(0x0041 + rr)[0];
        return String.format("%c %d.%d.%d", c, mm, nn, buildNumber[i]);
    }

    int[] findBoards(BoardType t) {
        int[] bn = new int[12];
        int n = 0;
        for (int i = 0; i < 12; i++) {
            if (getBoardType(i) == t) {
                bn[n++] = i;
            }
        }
        int[] b = new int[n];
        System.arraycopy(bn, 0, b, 0, n);
        return b;
    }

    public void checkRevisions() {
        int backplaneRev = (boardTypes[0] & 0x00FF0000) >> 16;
        if (backplaneRev == 0) {
            log.fatal("FATAL ERROR cannot handle Backplane revision A");
            System.exit(-1);
        }
        if (backplaneRev != 2) {
            log.warn("Non nominal backplane revision, only Rev C tested :"
                    + getBoardRevision(0));
        }

        int interfaceBuild = buildNumber[1];

        if (interfaceBuild <= 430) {
            log.fatal("FATAL ERROR cannot handle interface build <= 430 : "
                    + interfaceBuild);
            System.exit(-1);
        }

        if (interfaceBuild != 455) {
            log.warn("Non nominal interface build, only build 455 tested :"
                    + getBoardRevision(1));
        }

        int pbindex = findBoards(BoardType.POWERB)[0];
        int pbRev = (boardTypes[pbindex] & 0x00FF0000) >> 16;

        if (pbRev < 4) {
            log.fatal("FATAL ERROR cannot handle POWER B revisions lower than E");
            System.exit(-1);
        }
        if (pbRev != 4) {
            log.warn("Non nominal POWER B revision, only Rev E tested :"
                    + getBoardRevision(pbindex));
        }

    }

    public void dump() {
        System.out.printf("error code %s\n", errorCode);
        System.out.printf("error source %s\n", errorSource);
        System.out.printf("error line %s\n", errorLine);
        System.out.printf("status index %s\n", statusIndex);
        System.out.printf("power state %s\n", powerState);

        for (int i = 0; i < 12; i++) {
            System.out.printf("\n*** board %d type %s Rev. %s\n", i,
                    getBoardType(i), getBoardRevision(i));
            BoardStatus bs = boardStatus[i];
            if (bs != null)
                bs.dump();
        }
    }

    public BoardType[] getBoardTypes() {

        BoardType[] tt = new BoardType[13];
        for (int i = 0; i < 13; i++) {
            int t = boardTypes[i] >> 24;
            tt[i] = BoardType.forType(t);
        }
        return tt;

    }

}
