package org.lsst.ccs.drivers.reflex;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.edtcl.CLImage;
import org.lsst.ccs.drivers.edtcl.CLSerial;
import org.lsst.ccs.drivers.edtcl.EDTException;
import org.lsst.ccs.drivers.reflex.ReflexStatus.AD8X100Status;
import org.lsst.ccs.drivers.reflex.ReflexStatus.AD8X120Status;
import org.lsst.ccs.drivers.reflex.ReflexStatus.BackplaneStatus;
import org.lsst.ccs.drivers.reflex.ReflexStatus.BiasStatus;
import org.lsst.ccs.drivers.reflex.ReflexStatus.BoardStatus;
import org.lsst.ccs.drivers.reflex.ReflexStatus.DriverStatus;
import org.lsst.ccs.drivers.reflex.ReflexStatus.InterfaceStatus;
import org.lsst.ccs.drivers.reflex.ReflexStatus.PowerAStatus;
import org.lsst.ccs.drivers.reflex.ReflexStatus.PowerBStatus;

public class ReflexController {

    // commanding with some retrying...

    String sendCommand(String cmd) throws DriverException {
        String rep = null;
        try {
            rep = CLSerial.sendCommand(cmd);
            if (rep.startsWith("<"))
                return rep;
            else {
                throw new DriverException("Reflex invalid answer " + rep);
            }
        } catch (Exception e) {
            System.out.println("serial comm problem, retrying");
            e.printStackTrace();
            try {
                CLSerial.sendCommand(null); // to empty the buffer
            } catch (EDTException ee) {
                ee.printStackTrace();
            }

        }
        rep = CLSerial.sendCommand(cmd);
        if (rep.startsWith("<"))
            return rep;
        else {
            throw new DriverException("Reflex invalid answer " + rep);
        }
    }

    public void reboot() throws DriverException {
        CLSerial.writeCommand(">REBOOT");
    }

    public void warmboot() throws DriverException {
        CLSerial.writeCommand(">WARMBOOT");
    }

    public int readRegister(int register) throws DriverException {
        String cmd = String.format(">R%08X", register);
        // System.out.println(cmd);
        String rep = sendCommand(cmd);
        // System.out.println(rep);
        if (rep.charAt(0) == '<') {
            long l = Long.parseLong(rep.substring(1), 16);
            int i = (l > 0x07FFFFFF) ? (int) (l - 0x100000000L) : (int) l;
            return i;
        } else {
            throw new DriverException("invalid response from CameraLink: "
                    + rep);
        }
    }

    // undocumented
    public int getSystemState() throws DriverException {
        String rep = sendCommand(">TS");
        if (rep.charAt(0) == '<') {
            return Integer.parseInt(rep.substring(1), 16);
        } else {
            throw new DriverException("invalid response from CameraLink: "
                    + rep);
        }
    }

    void checkEmptyResponse(String rep) throws DriverException {
        if (!rep.equals("<"))
            throw new DriverException("invalid response from CameraLink: "
                    + rep);
    }

    public void writeRegister(int register, int value) throws DriverException {
        String cmd = String.format(">W%08X%08X", register, value);
        // System.out.println(cmd);
        String rep = sendCommand(cmd);
        // System.out.println(rep);
        checkEmptyResponse(rep);
    }

    public void clearError() throws DriverException {
        checkEmptyResponse(sendCommand(">E"));
    }

    public void configureAll() throws DriverException {
        checkEmptyResponse(sendCommand(">A"));
        // this can take some time to be done and the controller is sometimes
        // unresponsive. Let's sleep a bit.
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    public void configureSystem() throws DriverException {
        checkEmptyResponse(sendCommand(">C"));
    }

    public static enum BoardSlot {
        SYSTEM(-1), BACKPLANE(0), INTERFACE(1), POWERA(2), POWERB(3), DM1(4), DM2(
                5), DM3(6), DM4(7), DM5(8), DM6(9);
        int num;

        BoardSlot(int num) {
            this.num = num;
        };

        public static BoardSlot forInt(int num) {
            for (BoardSlot b : values()) {
                if (b.num == num)
                    return b;
            }
            throw new IllegalArgumentException(Integer.toString(num));
        }

        public int getSlotNum() {
            return num;
        }
    };

    public static enum BoardType {
        SYSTEM(-1), NONE(0x00), BACKPLANE_X6(0x10), CAMERALINK(0x20), POWERA(
                0x30), POWERB(0x40), AD8X120(0x50), AD8X100(0x51), DRIVER(0x60), BIAS(
                0x70);
        int tt;

        BoardType(int tt) {
            this.tt = tt;
        };

        public static BoardType forType(int tt) {
            for (BoardType b : values()) {
                if (b.tt == tt)
                    return b;
            }
            throw new IllegalArgumentException(Integer.toString(tt));
        }
    }

    public void configureBoard(BoardSlot b) throws DriverException {
        configureBoard(b.num);
    }

    public void configureBoard(int board) throws DriverException {
        String cmd = String.format(">B%1X", board);
        checkEmptyResponse(sendCommand(cmd));
    }

    public void powerOn() throws DriverException {
        checkEmptyResponse(sendCommand(">P1"));
    }

    public void powerOff() throws DriverException {
        checkEmptyResponse(sendCommand(">P0"));
    }

    public void timingGo() throws DriverException {
        checkEmptyResponse(sendCommand(">TG"));
    }

    public void timingHalt() throws DriverException {
        checkEmptyResponse(sendCommand(">TH"));
    }

    public void timingLoadParameters() throws DriverException {
        checkEmptyResponse(sendCommand(">TP"));
    }

    public void setTimingLoadAddress(int addr) throws DriverException {
        String cmd = String.format(">TA%03X", addr);
        checkEmptyResponse(sendCommand(cmd));
    }

    public void loadTimingLine(String line) throws DriverException {
        String cmd = String.format(">T%s", line);
        // System.out.println(cmd);
        checkEmptyResponse(sendCommand(cmd));
    }

    // undocumented but apparently necessary
    // Force all boards to be polled for their status

    public void pollStatus() throws DriverException {
        checkEmptyResponse(sendCommand(">L2"));
    }

    public ReflexStatus getStatus() throws DriverException {
        pollStatus();
        ReflexStatus s = new ReflexStatus();
        s.errorCode = readRegister(0);
        s.errorSource = readRegister(1);
        s.errorLine = readRegister(2);
        s.statusIndex = readRegister(3);
        s.powerState = readRegister(4);

        for (int i = 0; i < 12; i++) {
            s.boardTypes[i] = readRegister(0x5 + i);
        }

        for (int i = 0; i < 12; i++) {
            s.romID[i] = readRegister(0x11 + i);
        }

        for (int i = 0; i < 12; i++) {
            s.buildNumber[i] = readRegister(0x1D + i);
        }

        for (int i = 0; i < 12; i++) {
            s.featureFlags[i] = readRegister(0x29 + i);
        }

        for (int i = 0; i < 12; i++) {
            BoardType bt = s.getBoardType(i);
            BoardStatus bs = null;
            int pfx = (i + 1) << 16;
            switch (bt) {
            case SYSTEM: // should not exist here
                break;
            case BACKPLANE_X6:
                BackplaneStatus bps = new BackplaneStatus();
                bps.status = readRegister(0x0010001);
                bs = bps;
                break;
            case CAMERALINK:
                bs = new InterfaceStatus();
                break;
            case POWERA:
                PowerAStatus pas = new PowerAStatus();
                pas.V5VD = readRegister(0x0030001);
                pas.I5VD = readRegister(0x0030002);
                pas.V5VA = readRegister(0x0030003);
                pas.I5VA = readRegister(0x0030004);
                pas.Vm5V = readRegister(0x0030005);
                pas.Im5V = readRegister(0x0030006);
                bs = pas;
                break;
            case POWERB:
                PowerBStatus pbs = new PowerBStatus();
                pbs.V30V = readRegister(0x0040001);
                pbs.I30V = readRegister(0x0040002);
                pbs.V15V = readRegister(0x0040003);
                pbs.I15V = readRegister(0x0040004);
                pbs.Vm15V = readRegister(0x0040005);
                pbs.Im15V = readRegister(0x0040006);
                pbs.TECSetPoint = readRegister(0x0040007);
                pbs.TECActual = readRegister(0x0040008);
                bs = pbs;
                break;
            case AD8X120:
                AD8X120Status ad120 = new AD8X120Status();
                ad120.status = readRegister(pfx + 0x0001);
                bs = ad120;
                break;
            case AD8X100:
                AD8X100Status ad100 = new AD8X100Status();
                ad100.status = readRegister(pfx + 0x0001);
                bs = ad100;
                break;
            case DRIVER:
                DriverStatus ds = new DriverStatus();
                ds.status = readRegister(pfx + 0x0001);
                bs = ds;
                break;
            case BIAS:
                BiasStatus bis = new BiasStatus();
                bis.status = readRegister(pfx + 0x0001);
                for (int j = 0; j < 8; j++) {
                    bis.lv[j] = readRegister(pfx + j + 0x02);
                    bis.hv[j] = readRegister(pfx + j + 0x0a);
                    bis.lc[j] = readRegister(pfx + j + 0x12);
                    bis.hc[j] = readRegister(pfx + j + 0x1a);
                }
                bs = bis;
                break;
            case NONE:
                bs = null;
                break;
            }
            if (bs != null) {
                try {
                    bs.temperature = readRegister(pfx);
                } catch (Exception e) {
                    System.out.println(e);
                }
            }
            s.boardStatus[i] = bs;

        }
        lastStatus = s;
        return s;
    }

    ReflexStatus lastStatus;

    BoardControl[] controls = new BoardControl[13]; // 0=system, 1=backplane,
                                                    // etc.

    public ReflexController() throws DriverException {
        getStatus().checkRevisions();
        clearError();

        controls[0] = new SystemControl(this, BoardSlot.SYSTEM);
        for (int i = 0; i < 12; i++) {
            switch (lastStatus.getBoardType(i)) {
            case BACKPLANE_X6:
                controls[i + 1] = new BackplaneControl(this,
                        BoardSlot.forInt(i));
                break;

            case CAMERALINK:
                controls[i + 1] = new InterfaceControl(this,
                        BoardSlot.forInt(i));
                break;

            case POWERB:
                controls[i + 1] = new PowerBControl(this, BoardSlot.forInt(i));
                break;

            case AD8X100:
                controls[i + 1] = new AD8X100Control(this, BoardSlot.forInt(i));
                break;
            case DRIVER:
                controls[i + 1] = new DriverControl(this, BoardSlot.forInt(i));
                break;
            case BIAS:
                controls[i + 1] = new BiasControl(this, BoardSlot.forInt(i));
                break;
            default:
                controls[i + 1] = null;
            }
        }
    }

    public BoardControl[] getControls() {
        return controls; // mediocre encapsulation
    }

    public static interface ImageHandler {
        public void startingAcq(int i);

        public void handleImage(int i, short[][] image);
    }

    public void acquireImages(int w, int h, int n, final ImageHandler handler,
            boolean async) throws DriverException {
        try {
            // System.out.println("ACQ setup");
            CLImage.setup(w, h);
            CLImage.start(n);
            // System.out.println("ACQ started");
            if (async) {
                // number of threads should not be greater than number of
                // buffers in
                // the edt library buffer ring !
                ExecutorService exec = Executors.newFixedThreadPool(2);
                for (int i = 0; i < n; i++) {
                    // System.out.println("ACQ waiting for image");
                    handler.startingAcq(i);
                    final short[][] img = CLImage.getImage();
                    // System.out.println("ACQ got image");
                    final int j = i;
                    exec.execute(new Runnable() {
                        public void run() {
                            handler.handleImage(j, img);
                        };
                    });
                }
                exec.shutdown();
                exec.awaitTermination(10, TimeUnit.SECONDS);

            } else {
                for (int i = 0; i < n; i++) {
                    short[][] img = CLImage.getImage();
                    handler.handleImage(i, img);
                }

            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            // System.out.println("ACQ cleanup");
            CLImage.close();
        }

    }

    public static void main(String[] args) throws DriverException {
        ReflexController c = new ReflexController();
        ReflexStatus s = c.getStatus();
        s.dump();
    }

}