package org.lsst.ccs.drivers.rebps;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ************************************************
 *
 *  Routines for controlling a REB power supply
 *
 *  @author  Owen Saxton
 *
 ************************************************
 */
public class RebPs {

   /**
    *  Constants and data.
    */
    public static final int
        PS_DIGITAL  = 0,
        PS_ANALOG   = 1,
        PS_OD       = 2,
        PS_CLK_HIGH = 3,
        PS_CLK_LOW  = 4,
        PS_HEATER   = 5,
        PS_DPHI     = 5,
        PS_HV_BIAS  = 6,

        CHAN_VOLT_BEF_LDO = 0,
        CHAN_CURR_BEF_LDO = 1,
        CHAN_VOLT_AFT_LDO = 2,
        CHAN_CURR_AFT_LDO = 3,
        CHAN_VOLT_AFT_SW  = 4,
        CHAN_VOLT2_AFT_LDO = 5,
        CHAN_VOLT_DAC = -1,

        NUM_PS      = 7,
        NUM_REBS    = 3;  // For now: should be 6

    private static final int
        MAX_REGS = 64,

        READ_TIMEOUT = 1000,
        DEFAULT_PORT = 8192,

        PGP_PKT_LENG    = 20,
        PGP_OFF_HEADER  = 4,
        PGP_OFF_OC      = 8,
        PGP_OFF_ADDR    = 8,
        PGP_OFF_DATA    = 12,
        PGP_OFF_FOOTER  = 16,
        PGP_STS_TIMEOUT = 0x02,
        PGP_STS_ERROR   = 0x01,

        DEVC_LTC2945      = 0,
        DEVC_MAX11644     = 1,
        DEVC_MAX5803      = 2,

        REG_FPGA_VERSION  = 0x00,
        REG_SCRATCH       = 0x01,
        REG_SERIAL_NO     = 0x02,
        REG_GEOG_ADDR     = 0x04,
        REG_SPARE0        = 0x05,
        REG_STATUS        = 0x06,
        REG_STATUS_LATCH  = 0x0a,
        REG_POWER_CTRL    = 0x0e,
        REG_LED_TEST_CTRL = 0x0f,
        REG_TMO_PS_CFG    = 0x10,
        REG_TMO_PS_WAIT   = 0x11,
        REG_TMO_PS_WAIT_0 = 0x12,
        REG_TMO_PS_WAIT_1 = 0x13,
        REG_TMO_PS_WAIT_3 = 0x14,
        REG_TMO_PS_WAIT_5 = 0x15,
        REG_TMO_PS_WAIT_2 = 0x16,
        REG_TMO_PS_WAIT_6 = 0x17,
        REG_TMO_PS_CHECK  = 0x18,
        REG_TMO_PS_WAIT_DN = 0x19,
        REG_PS_STATE_RAFT0 = 0x1a,
        REG_PS_STATE_RAFT1 = 0x1b,
        REG_PS_FAIL_RAFT0  = 0x1c,
        REG_PS_FAIL_RAFT1  = 0x1d,
        REG_PS_OFF_RAFT0   = 0x1e,
        REG_PS_OFF_RAFT1   = 0x1f,
        REG_FAULT_DUMP     = 0x20,
        REG_DS75LV_WRITE   = 0xe0,
        REG_DS75LV_READ    = 0xe1,
        REG_THRESHOLD      = 0x100,
        REG_REB_BASE       = 0x10000,
        REG_REB_INCR       = 0x10000,
        REG_PS_BASE        = REG_REB_BASE,
        REG_PS_INCR        = 0x1000,
        REG_DEV_INCR       = 0x100,
        REG_REB_DATA_BASE      = REG_REB_BASE + 0x7000,
        REG_REB_RDO_PERIOD     = 0x00,
        REG_REB_RDO_CONFIG     = 0x01,
        REG_REB_RDO_COUNT      = 0x03,
        REG_REB_RDO_DIAG_CNT   = 0x04,
        REG_REB_CHAN_BASE  = REG_REB_DATA_BASE + 0x80,
        XXX = 0;
    private static final byte
        PGP_OC_READ  = 0x00,
        PGP_OC_WRITE = 0x40;

    private static final int[][]
        chanOffset = {{ 1,  0,  2,  3,  4},
                      { 9,  8, 10, 11, 12},
                      {17, 16, 18, 19, 21, 20},
                      {25, 24, 26, 27, 28},
                      {33, 32, 34, 35, 37, 36},
                      {41, 40, 42, 43, 44},
                      {49, 48, 50}},
        chanDesc =   {{0x01, 0x00, 0x02, 0x10, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x12, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x12, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x11, 0x20},
                      {0x01, 0x00, 0x10}},
        devcDesc =   {{DEVC_LTC2945, DEVC_MAX11644},
                      {DEVC_LTC2945, DEVC_MAX11644},
                      {DEVC_LTC2945, DEVC_LTC2945},
                      {DEVC_LTC2945, DEVC_MAX11644},
                      {DEVC_LTC2945, DEVC_LTC2945},
                      {DEVC_LTC2945, DEVC_MAX11644, DEVC_MAX5803},
                      {DEVC_LTC2945, DEVC_MAX5803}};
    private static final int[]
        dacChannel = {-1, -1, -1, -1, -1, 5, 2};
    private static final double[][]
        convConst = {{0.0250, 0.000490,  0.00352, 0.00140,   0.00305},
                     {0.0250, 0.000490,  0.00352, 0.00140,   0.00305},
                     {0.0250, 0.0000641, 0.0206,  0.0000641, 0.0206,  0.0250},
                     {0.0250, 0.000125,  0.00654, 0.000617,  0.006},
                     {0.0250, 0.000758,  0.00352, 0.000531,  0.00050, 0.0250},
                     {0.0250, 0.000125,  0.00654, 0.000750,  0.00600, 1.0},
                     {0.0250, 0.000000296, 1.0}};
    private static final List<Long> oldUnits = new ArrayList<>();
    static {
        oldUnits.add(0x6af054000af5c200L);  // LPNHE
        oldUnits.add(0x6af0540060f5c200L);  // BNL
        oldUnits.add(0x6af054005cf5c200L);  // Penn
    }

    private DatagramSocket sock;
    private int seqno, nSeqErr, nTimeout;
    private final byte[]
        inBuff = new byte[PGP_PKT_LENG + 4 * (MAX_REGS - 1)],
        outBuff = new byte[PGP_PKT_LENG + 4 * (MAX_REGS - 1)];
    private final DatagramPacket
        inPkt = new DatagramPacket(inBuff, inBuff.length),
        outPkt = new DatagramPacket(outBuff, outBuff.length);
    private double clockLoAftSwCoeff;
    private boolean simulated;
    private final Map<Integer, Integer> simRegMap = new HashMap<>();


   /**
    *  Opens a connection to a board.
    *
    *  @param  ipAddr  The IP address, or null or empty for simulation
    *
    *  @throws  DriverException
    */
    public void open(String ipAddr) throws DriverException
    {
        open(ipAddr, DEFAULT_PORT);
    }


   /**
    *  Opens a connection to a board.
    *
    *  @param  ipAddr  The IP address, or null or empty for simulation
    *
    *  @param  port    The port number
    *
    *  @throws  DriverException
    */
    public synchronized void open(String ipAddr, int port) throws DriverException
    {
        if (sock != null) {
            throw new DriverException("Connection is already open");
        }
        nSeqErr = 0;
        nTimeout = 0;
        try {
            DatagramSocket newSock = new DatagramSocket();
            if (ipAddr == null || ipAddr.isEmpty()) {
                simulated = true;
                simInitialize();
            }
            else {
                simulated = false;
                InetAddress inetAddr = InetAddress.getByName(ipAddr);
                newSock.connect(inetAddr, port);
                newSock.setSoTimeout(READ_TIMEOUT);
                outPkt.setAddress(inetAddr);
                outPkt.setPort(port);
                sock = newSock;
            }
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
        clockLoAftSwCoeff = oldUnits.contains(getSerialNo()) ? 13.1 : 21.1;
    }


   /**
    *  Closes the connection.
    *
    *  This method isn't synchronized in order to allow a connection to be
    *  closed while another thread is waiting for a read to complete.  But
    *  some synchronization does need to be implemented to avoid race
    *  conditions leading to null pointer exceptions..
    *
    *  @throws  DriverException
    */
    public void close() throws DriverException
    {
        checkOpen();
        sock.close();
        sock = null;
    }


   /**
    *  Writes a register.
    *
    *  @param  addr   The register address
    *
    *  @param  value  The value to write
    *
    *  @throws  DriverException
    */
    public void writeReg(int addr, int value) throws DriverException
    {
        writeRegs(addr, new int[]{value});
    }


   /**
    *  Writes registers.
    *
    *  @param  addr   The first register address
    *
    *  @param  value  The array of values to write.  If it contains more than
    *                 MAX_REGS (currently 64) values, only MAX_REGS values
    *                 are written.
    *
    *  @throws  DriverException
    */
    public synchronized void writeRegs(int addr, int[] value)
        throws DriverException
    {
        checkOpen();
        int count = Math.min(value.length, MAX_REGS);
        if (simulated) {
            simWriteRegs(addr, value, count);
            return;
        }
        Convert.intToBytesBE(addr, outBuff, PGP_OFF_ADDR);
        for (int j = 0; j < count; j++) {
            Convert.intToBytesBE(value[j], outBuff, PGP_OFF_DATA + 4 * j);
        }
        outBuff[PGP_OFF_OC] = PGP_OC_WRITE;
        for (int j = 0; j < 2; j++) {
            try {
                send(PGP_PKT_LENG + 4 * (count - 1));
                receive();
                break;
            }
            catch (DriverTimeoutException e) {
                if (j == 1) {
                    throw e;
                }
            }
        }
    }


   /**
    *  Reads a register.
    *
    *  @param  addr  The register address
    *
    *  @return  The read value
    *
    *  @throws  DriverException
    */
    public int readReg(int addr) throws DriverException
    {
        return readRegs(addr, 1)[0];
    }


   /**
    *  Reads registers.
    *
    *  @param  addr   The first register address
    *
    *  @param  count  The number of registers to read.  If greater than
    *                 MAX_REGS (currently 64), is set to MAX_REGS.
    *
    *  @return  The array of read values
    *
    *  @throws  DriverException
    */
    public synchronized int[] readRegs(int addr, int count)
        throws DriverException
    {
        checkOpen();
        if (count <= 0) {
            return new int[0];
        }
        count = Math.min(count, MAX_REGS);
        if (simulated) {
            return simReadRegs(addr, count);
        }
        Convert.intToBytesBE(addr, outBuff, PGP_OFF_ADDR);
        Convert.intToBytesBE(count - 1, outBuff, PGP_OFF_DATA);
        outBuff[PGP_OFF_OC] = PGP_OC_READ;
        for (int j = 0; j < 2; j++) {
            try {
                send(PGP_PKT_LENG);
                receive();
                break;
            }
            catch (DriverTimeoutException e) {
                if (j == 1) {
                    throw e;
                }
            }
        }
        int nRead = (inPkt.getLength() - PGP_PKT_LENG) / 4 + 1;
        int[] data = new int[nRead];
        for (int j = 0; j < nRead; j++) {
            data[j] = Convert.bytesToIntBE(inBuff, PGP_OFF_DATA + 4 * j);
        }
        return data;
    }


   /**
    *  Updates a register.
    *
    *  @param  addr   The register address
    *
    *  @param  mask   The mask of bits to be updated
    *
    *  @param  value  The value to use for updating
    *
    *  @return  Previous register value
    *
    *  @throws  DriverException
    */
    public synchronized int updateReg(int addr, int mask, int value)
        throws DriverException
    {
        int prevValue = readReg(addr);
        writeReg(addr, (prevValue & ~mask) | (value & mask));
        return prevValue;
    }


   /**
    *  Gets the firmware version number.
    *
    *  @return  The version number
    *
    *  @throws  DriverException
    */
    public int getFwVersion() throws DriverException
    {
        return readReg(REG_FPGA_VERSION);
    }


   /**
    *  Gets the serial number.
    *
    *  @return  The serial number
    *
    *  @throws  DriverException
    */
    public long getSerialNo() throws DriverException
    {
        int[] value = readRegs(REG_SERIAL_NO, 2);
        return ((long)value[1] << 32) | (value[0] & 0xffffffffL);
    }


   /**
    *  Gets the geographical address.
    *
    *  @return  The geographical address
    *
    *  @throws  DriverException
    */
    public int getGeogAddr() throws DriverException
    {
        return readReg(REG_GEOG_ADDR);
    }


   /**
    *  Sets the temperature resolution.
    *
    *  @param  value  The resolution value (0 - 3)
    *
    *  @throws  DriverException
    */
    public void setTemperatureRes(int value) throws DriverException
    {
        writeReg(REG_DS75LV_WRITE, 0x50000 | ((value & 0x03) << 13));
    }


   /**
    *  Reads the temperature.
    *
    *  @return  The temperature (C)
    *
    *  @throws  DriverException
    */
    public double readTemperature() throws DriverException
    {
        writeReg(REG_DS75LV_WRITE, 0x80000);
        int value = 0;
        for (int count = 0; count < 10; count++) {
            try {
                Thread.sleep(25);
            }
            catch (InterruptedException e) {
            }
            value = readReg(REG_DS75LV_READ);
            if ((value & 0x80000000) != 0) break;
        }
        if ((value & 0x80000000) == 0) {
            throw new DriverException("Timeout reading temperature");
        }
        if ((value & 0x40000000) != 0) {
            throw new DriverException("Error reading temperature");
        }
        return (value & 0xffff) / 256.0;
    }


   /**
    *  Sets the power options for a REB.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  value   The value to set
    *
    *  @throws  DriverException
    */
    public void setPower(int rebNum, int value) throws DriverException
    {
        checkRebNumber(rebNum);
        //int shift = 3 * rebNum;
        //updateReg(REG_POWER_CTRL, 0x07 << shift, value << shift);
        int shift = 8 * (rebNum + 1);
        int prevValue = updateReg(REG_SPARE0, 0xff << shift, value << shift) >> shift;
        if ((prevValue & 1) == 0 && (value & 1) != 0) {
            try {
                Thread.sleep(25);
            }
            catch (InterruptedException e) {
            }
            configure(rebNum);  // MAXIM devices need this every time
        }
    }


   /**
    *  Gets the power options for a REB.
    *
    *  @param  rebNum  The REB number
    *
    *  @return  The bit mask of power options
    *
    *  @throws  DriverException
    */
    public int getPower(int rebNum) throws DriverException
    {
        checkRebNumber(rebNum);
        //return (readReg(REG_POWER_CTRL) >> (3 * rebNum)) & 0x07;
        return (readReg(REG_SPARE0) >> (8 * (rebNum + 1))) & 0xff;
    }


   /**
    *  Gets the overall power state.
    *
    *  @return  The power state
    *
    *  @throws  DriverException
    */
    public int getPowerState() throws DriverException
    {
        return (readReg(REG_POWER_CTRL) >> 18) & 0x3f;
    }


   /**
    *  Gets the power state of a REB.
    *
    *  @param  rebNum  The REB number
    *
    *  @return  The power state
    *
    *  @throws  DriverException
    */
    public boolean isPowerOn(int rebNum) throws DriverException
    {
        checkRebNumber(rebNum);
        return ((getPowerState() >> rebNum) & 1) != 0;
    }


   /**
    *  Configures a device.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  devc    The device number
    *
    *  @throws  DriverException
    */
    public void configure(int rebNum, int psNum, int devc)
        throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        int addr = getDevcBaseAddress(rebNum, psNum, devc);
        switch (getDeviceType(psNum, devc)) {
        case DEVC_LTC2945:
            break;
        case DEVC_MAX11644:
            writeReg(addr, 0x03dadada);
            break;
        case DEVC_MAX5803:
            writeReg(addr + 0x26, 0);
            break;
        }
    }


   /**
    *  Configures all devices of a power supply.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @throws  DriverException
    */
    public void configure(int rebNum, int psNum) throws DriverException
    {
        for (int devc = 0; devc < getNumDevices(psNum); devc++) {
            configure(rebNum, psNum, devc);
        }
    }


   /**
    *  Configures all devices of a REB.
    *
    *  @param  rebNum  The REB number
    *
    *  @throws  DriverException
    */
    public void configure(int rebNum) throws DriverException
    {
        for (int psNum = 0; psNum < NUM_PS; psNum++) {
            configure(rebNum, psNum);
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  chan    The channel number
    *
    *  @return  The value of the channel data
    *
    *  @throws  DriverException
    */
    public double readChannel(int rebNum, int psNum, int chan)
        throws DriverException
    {
        double value = readChanDirect(rebNum, psNum, chan);
        if (psNum == PS_CLK_LOW
              && (chan == CHAN_VOLT2_AFT_LDO || chan == CHAN_VOLT_AFT_SW)) {
            double mult = (chan == CHAN_VOLT2_AFT_LDO) ? 1 : clockLoAftSwCoeff;
            value = mult * (value - readChanDirect(rebNum, psNum, CHAN_VOLT_AFT_LDO));
        }
        return value;
    }


   /**
    *  Reads a channel.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  chan    The channel number
    *
    *  @return  The value of the channel data
    *
    *  @throws  DriverException
    */
    private double readChanDirect(int rebNum, int psNum, int chan)
        throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        chan = convChannel(psNum, chan);
        int desc = getChanDesc(psNum, chan);
        int devc = (desc >> 4) & 0x0f, dChan = desc & 0x0f;
        int addr = getDevcBaseAddress(rebNum, psNum, devc);
        int value = 0;
        switch (getDeviceType(psNum, devc)) {
        case DEVC_LTC2945:
            addr += (dChan == 0) ? 0x14 : (dChan == 1) ? 0x1e : 0x28;
            int[] data = readRegs(addr, 2);
            value = ((data[0] & 0xff) << 4) | ((data[1] & 0xff) >> 4);
            break;
        case DEVC_MAX11644:
            value = (readReg(addr) >> (16 * (1 - dChan))) & 0xfff;
            break;
        case DEVC_MAX5803:
            value = (readReg(addr + 0xa0) >> 4) & 0xfff;
            break;
        }
        return value * getConversion(psNum, chan);
    }


   /**
    *  Reads all channels of a power supply.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @return  The array of power supply values
    *
    *  @throws  DriverException
    */
    public double[] readChannel(int rebNum, int psNum) throws DriverException
    {
        checkRebNumber(rebNum);
        int nChan = getNumChannels(psNum);
        double[] values = new double[nChan];
        for (int chan = 0; chan < nChan; chan++) {
            values[chan] = readChannel(rebNum, psNum, chan);
        }
        return values;
    }


   /**
    *  Reads a sequencer channel.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  chan    The channel number
    *
    *  @return  The channel value
    *
    *  @throws  DriverException
    */
    public double readSeqChan(int rebNum, int psNum, int chan)
        throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        chan = convChannel(psNum, chan);
        int addr = getSeqChanAddress(rebNum, psNum, chan);
        return readReg(addr) * getConversion(psNum, chan);
    }


   /**
    *  Writes to a DAC.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  value   The voltage value to write
    *
    *  @throws  DriverException
    */
    public void writeDac(int rebNum, int psNum, double value)
        throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        int chan = convChannel(psNum, CHAN_VOLT_DAC);
        int devc = getChanDesc(psNum, chan) >> 4;
        int addr = getDevcBaseAddress(rebNum, psNum, devc);
        int val = (int)(value / getConversion(psNum, chan));
        writeReg(addr + 0xa0, (val & 0xfff) << 4);
    }


   /**
    *  Tests a channel number for validity.
    *
    *  @param  psNum  The power supply number
    *
    *  @param  chan   The channel number
    *
    *  @return  Whether the channel number is valid
    *
    *  @throws  DriverException
    */
    public static boolean testChannelNumber(int psNum, int chan)
        throws DriverException
    {
        return chan >= 0 && chan < getNumChannels(psNum);
    }


   /**
    *  Tests a REB number for validity.
    *
    *  @param  rebNum  The REB number
    *
    *  @return  Whether the REB number is valid
    *
    *  @throws  DriverException
    */
    public static boolean testRebNumber(int rebNum) throws DriverException
    {
        return rebNum >= 0 && rebNum < NUM_REBS;
    }


   /**
    *  Gets the sequence error count.
    *
    *  @return  The number of sequence errors
    */
    public int getNumSeqErr()
    {
        return nSeqErr;
    }


   /**
    *  Gets the socket timeout count.
    *
    *  @return  The number of socket timeouts
    */
    public int getNumTimeout()
    {
        return nTimeout;
    }


   /**
    *  Gets the number of devices in a power supply.
    *
    *  @param  psNum  The power supply number
    *
    *  @return  The number of devices
    *
    *  @throws  DriverException
    */
    private static int getNumDevices(int psNum) throws DriverException
    {
        checkPsNumber(psNum);
        return devcDesc[psNum].length;
    }


   /**
    *  Gets the number of channels in a power supply.
    *
    *  @param  psNum  The power supply number
    *
    *  @return  The number of channels
    *
    *  @throws  DriverException
    */
    private static int getNumChannels(int psNum) throws DriverException
    {
        checkPsNumber(psNum);
        return chanDesc[psNum].length;
    }


   /**
    *  Checks whether a connection is open.
    *
    *  @throws  DriverException
    */
    private void checkOpen() throws DriverException
    {
        if (sock == null) {
            throw new DriverException("Connection is not open");
        }
    }


   /**
    *  Sends the "out" packet.
    *
    *  @throws  DriverException
    */
    private void send(int leng) throws DriverException
    {
        seqno += 4;
        Convert.intToBytes(seqno, outBuff, PGP_OFF_HEADER);
        Convert.intToBytes(0, outBuff, leng - PGP_PKT_LENG + PGP_OFF_FOOTER);
        outPkt.setLength(leng);
        try {
            sock.send(outPkt);
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    *  Receives the "in" packet.
    *
    *  @throws  DriverException
    */
    private void receive() throws DriverException
    {
        while (true) {
            inPkt.setLength(inBuff.length);
            try {
                sock.receive(inPkt);
                //if (seqno == Convert.bytesToInt(inBuff, PGP_OFF_HEADER)) break;
                int rSeqno = Convert.bytesToInt(inBuff, PGP_OFF_HEADER);
                if (seqno == rSeqno) break;
                nSeqErr++;
            }
            catch (SocketTimeoutException e) {
                nTimeout++;
                throw new DriverTimeoutException();
            }
            catch (IOException e) {
                throw new DriverException(e);
            }
        }
        int leng = inPkt.getLength();
        int status = Convert.bytesToIntBE(inBuff, leng + PGP_OFF_FOOTER
                                                       - PGP_PKT_LENG);
        if ((status & PGP_STS_TIMEOUT) != 0) {
            throw new DriverException("Register access timeout");
        }
        if ((status & PGP_STS_ERROR) != 0) {
            throw new DriverException("Register access error");
        }
    }


    /**
    *  Converts a channel number.
    *
    *  @throws  DriverException
    */
    private static int convChannel(int psNum, int chan)
        throws DriverException
    {
        if (chan == CHAN_VOLT_DAC) {
            chan = dacChannel[psNum];
        }
        if (chan < 0) {
            throw new DriverException("Power supply has no DAC");
        }
        if (chan >= chanDesc[psNum].length) {
            throw new DriverException("Invalid channel for PS");
        }
        return chan;
    }


   /**
    *  Gets a device type.
    *
    *  @throws  DriverException
    */
    private static int getDeviceType(int psNum, int devc)
        throws DriverException
    {
        try {
            return devcDesc[psNum][devc];
        }
        catch (IndexOutOfBoundsException e) {
            throw new DriverException("Invalid device index for PS");
        }
    }


    /**
    *  Gets a channel descriptor word.
    *
    *  @throws  DriverException
    */
    private static int getChanDesc(int psNum, int chan)
        throws DriverException
    {
        return chanDesc[psNum][chan];
    }


   /**
    *  Gets a device base register address.
    *
    *  @throws  DriverException
    */
    private static int getDevcBaseAddress(int rebNum, int psNum, int devc)
        throws DriverException
    {
        return REG_PS_BASE + REG_REB_INCR * rebNum + REG_PS_INCR * psNum
                 + REG_DEV_INCR * devc;
    }


   /**
    *  Gets a channel register address.
    *
    *  @throws  DriverException
    */
    private static int getSeqChanAddress(int rebNum, int psNum, int chan)
        throws DriverException
    {
        return REG_REB_CHAN_BASE + REG_REB_INCR * rebNum
                 + chanOffset[psNum][chan];
    }


   /**
    *  Gets a conversion constant.
    *
    *  @throws  DriverException
    */
    private static double getConversion(int psNum, int chan)
        throws DriverException
    {
        return convConst[psNum][chan];
    }


   /**
    *  Checks a REB number for validity.
    *
    *  @throws  DriverException
    */
    private static void checkRebNumber(int rebNum) throws DriverException
    {
        if (!testRebNumber(rebNum)) {
            throw new DriverException("Invalid REB number (" + rebNum + ")");
        }
    }


   /**
    *  Checks a power supply number for validity.
    *
    *  @throws  DriverException
    */
    private static void checkPsNumber(int psNum) throws DriverException
    {
        if (psNum < 0 || psNum >= NUM_PS) {
            throw new DriverException("Invalid PS number (" + psNum + ")");
        }
    }


    private static final double[][]
        simValues = {{ 5.0, 0.0,  5.0, 0.0,  5.0},
                     { 7.0, 0.0,  7.0, 0.0,  7.0},
                     {40.0, 0.0, 40.0, 0.0, 40.0, 40.0},
                     {16.0, 0.0, 16.0, 0.0, 16.0},
                     {16.0, 0.0,  0.0, 0.0, 16.0, 16.0},
                     { 8.0, 0.0,  8.0, 0.0,  8.0},
                     { 0.0, 0.0}};


   /**
    *  Initializes the simulation.
    */
    private void simInitialize()
    {
        simRegMap.clear();
        simRegMap.put(REG_SPARE0, 0);
        simRegMap.put(REG_DS75LV_READ, (int)(256 * 23.45) | 0x80000000);
    }

    
   /**
    *  Writes simulated registers.
    *
    *  @param  addr   The first register address
    *
    *  @param  value  The array of values to write.
    *
    *  @param  count  The number of values to write.
    */
    private void simWriteRegs(int addr, int[] value, int count)
        throws DriverException
    {
        for (int j = 0; j < count; j++, addr++) {
            if (addr == REG_SPARE0) {
                simSwitchPower(simRegMap.get(REG_SPARE0), value[j]);
            }
            if (addr < 0x10000 || (addr & 0xfff) != 0x100) {
                simRegMap.put(addr, value[j]);
            }
        }
    }


   /**
    *  Reads simulated registers.
    *
    *  @param  addr   The first register address
    *
    *  @param  count  The number of registers to read.
    *
    *  @return  The array of read values
    */
    private int[] simReadRegs(int addr, int count)
    {
        int[] data = new int[count];
        for (int j = 0; j < count; j++) {
            Integer value = simRegMap.get(addr + j);
            data[j] = value == null ? 0 : value;
        }
        return data;
    }


   /**
    *  Simulates changing the power switch.
    *
    *  @param  oldSwitch  The old switch value
    *
    *  @param  newSwitch  The new switch value
    */
    private void simSwitchPower(int oldSwitch, int newSwitch)
        throws DriverException
    {
        oldSwitch >>= 8;
        newSwitch >>= 8;
        int changed = oldSwitch ^ newSwitch;
        for (int chan = 0; changed != 0; chan++, changed >>= 1) {
            if ((changed & 1) != 0) {
                int reb = chan / 8;
                int ps = chan % 8;
                if (ps == 0) {
                    simMasterOn(reb, newSwitch >> (8 * reb));
                }
                else {
                    simPsOn(reb, ps - 1, (newSwitch & (1 << chan)) != 0);
                }
            }
        }
    }


   /**
    *  Simulates turning on or off the master switch.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  swtch   The switch value
    */
    private void simMasterOn(int rebNum, int swtch)
        throws DriverException
    {
        boolean on = (swtch & 1) != 0;
        for (int ps = 0; ps < NUM_PS; ps++) {
            boolean psOn = on && ((1 << (ps + 1)) & swtch) != 0;
            int psChan = ps == PS_HV_BIAS ? CHAN_VOLT_BEF_LDO : CHAN_VOLT_AFT_SW;
            for (int chan = 0; chan < simValues[ps].length; chan++) {
                if (chan != CHAN_CURR_AFT_LDO && chan != CHAN_CURR_BEF_LDO) {
                    simChanOn(rebNum, ps, chan, chan == psChan ? psOn : on);
                }
            }
        }
    }


   /**
    *  Simulates turning on or off one power supply.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  on      Turn on if true, off if false
    */
    private void simPsOn(int rebNum, int psNum, boolean on)
        throws DriverException
    {
        int chan = psNum == PS_HV_BIAS ? CHAN_VOLT_BEF_LDO : CHAN_VOLT_AFT_SW;
        simChanOn(rebNum, psNum, chan, on);
    }


   /**
    *  Simulates turning on or off one power supply channel.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  chan    The channel number
    *
    *  @param  on      Turn on if true, off if false
    */
    private void simChanOn(int rebNum, int psNum, int chan, boolean on)
        throws DriverException
    {
        double value = on ? simValues[psNum][chan] : 0;
        if (on && psNum == PS_CLK_LOW) {
            if (chan == CHAN_VOLT_AFT_SW) {
                value = value / clockLoAftSwCoeff + simValues[psNum][CHAN_VOLT_AFT_LDO];
            }
            else if (chan == CHAN_VOLT2_AFT_LDO) {
                value += simValues[psNum][CHAN_VOLT_AFT_LDO];
            }
        }
        simSetChannel(rebNum, psNum, chan, value);
    }


   /**
    *  Sets a simulated channel DAC value.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  psNum   The power supply number
    *
    *  @param  chan    The channel number
    *
    *  @param  value   The value to set
    */
    private void simSetChannel(int rebNum, int psNum, int chan, double value)
        throws DriverException
    {
        chan = convChannel(psNum, chan);
        int desc = getChanDesc(psNum, chan);
        int devc = (desc >> 4) & 0x0f, dChan = desc & 0x0f;
        int addr = getDevcBaseAddress(rebNum, psNum, devc);
        int dacVal = (int)(value / getConversion(psNum, chan) + 0.5);
        switch (getDeviceType(psNum, devc)) {
        case DEVC_LTC2945:
            addr += (dChan == 0) ? 0x14 : (dChan == 1) ? 0x1e : 0x28;
            simRegMap.put(addr, dacVal >> 4);
            simRegMap.put(addr + 1, (dacVal & 0x0f) << 4);
            break;
        case DEVC_MAX11644:
            Integer oldVal = simRegMap.get(addr);
            oldVal = oldVal == null ? 0 : oldVal;
            int mask = 0xffff << (16 * dChan);
            simRegMap.put(addr, (dacVal << (16 * (1 - dChan))) | (oldVal & mask));
            break;
        case DEVC_MAX5803:
            simRegMap.put(addr + 0xa0, dacVal >> 4);
            break;
        }
    }
        
}
