package org.lsst.ccs.drivers.rebps;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 ** 
 **  Program to test the REB power supply driver
 ** 
 **  @author Owen Saxton
 ** 
 ******************************************************************************
 */
public class TestRebPs {

    protected static enum OnOff { ON, OFF; };

    private final RebPs ps;


   /**
    *************************************************************************** 
    ** 
    **  Constructor.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    public TestRebPs() throws DriverException
    {
        ps = new RebPs();
    }


   /**
    *************************************************************************** 
    ** 
    **  Connects to a device.
    ** 
    **  @param  ipAddr  The IP address
    **
    **  @param  port    The port number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="connect", description="Open connection to device")
    public void connect(@Argument(name="ipaddr", description="IP address")
                        String ipAddr,
                        @Argument(name="port", description="Port number")
                        int port) throws DriverException
    {
        ps.open(ipAddr, port);
    }


   /**
    *************************************************************************** 
    ** 
    **  Connects to a device.
    ** 
    **  @param  ipAddr  The IP address
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="connect", description="Open connection to device")
    public void connect(@Argument(name="ipaddr", description="IP address")
                        String ipAddr) throws DriverException
    {
        ps.open(ipAddr);
    }


   /**
    *************************************************************************** 
    ** 
    **  Disconnects from the device.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="disconnect", description="Close connection to device")
    public void disconnect() throws DriverException
    {
        ps.close();
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads a register.
    ** 
    **  @param  addr  The register address
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="read", description="Read a register")
    public String read(@Argument(name="addr", description="Register address")
                       int addr) throws DriverException
    {
        return String.format("%08x", ps.readReg(addr));
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads registers.
    ** 
    **  @param  addr   The first register address
    **
    **  @param  count  The number of registers to read
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="read", description="Reads registers")
    public String read(@Argument(name="addr", description="Register address")
                       int addr,
                       @Argument(name="count", description="Register count")
                       int count) throws DriverException
    {
        return formatHex(addr, ps.readRegs(addr, count));
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes registers.
    ** 
    **  @param  addr   The first register address
    **
    **  @param  value  The values to write
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="write", description="Write a register")
    public void write(@Argument(name="addr", description="Register address")
                      int addr,
                      @Argument(name="value", description="Register values")
                      int... value) throws DriverException
    {
        ps.writeRegs(addr, value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Updates a register.
    ** 
    **  @param  addr   The register address
    **
    **  @param  mask   The mask of bits to change
    **
    **  @param  value  The value to write
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="update", description="Update a register")
    public void update(@Argument(name="addr", description="Register address")
                       int addr,
                       @Argument(name="mask", description="Mask of bits to change")
                       int mask,
                       @Argument(name="value", description="Register value")
                       int value) throws DriverException
    {
        ps.updateReg(addr, mask, value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the serial number.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showserial", description="Show the serial number")
    public String showSerial() throws DriverException
    {
        return String.format("%016x", ps.getSerialNo());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the firmware version.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showfirmware", description="Show the firmware version")
    public String showFirmware() throws DriverException
    {
        return String.format("%08x", ps.getFwVersion());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the geographical address.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showgeogaddr", description="Show the geographical address")
    public String showGeogAddr() throws DriverException
    {
        return String.valueOf(ps.getGeogAddr());
    }


   /**
    *************************************************************************** 
    ** 
    **  Configures a device.
    **
    **  @param  rebNum  The REB number
    **
    **  @param  psNum   The power supply number
    **
    **  @param  devc    The device number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="configure", description="Configure a device")
    public void configure(@Argument(name="rebNum", description="REB number")
                          int rebNum,
                          @Argument(name="psNum",
                                    description="Power supply number")
                          int psNum,
                          @Argument(name="device", description="Device number")
                          int devc) throws DriverException
    {
        ps.configure(rebNum, psNum, devc);
    }


   /**
    *************************************************************************** 
    ** 
    **  Configures all devices in a power supply.
    **
    **  @param  rebNum  The REB number
    **
    **  @param  psNum   The power supply number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="configure", description="Configure all devices in a PS")
    public void configure(@Argument(name="rebNum", description="REB number")
                          int rebNum,
                          @Argument(name="psNum",
                                    description="Power supply number")
                          int psNum) throws DriverException
    {
        ps.configure(rebNum, psNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Configures all devices in a REB.
    **
    **  @param  rebNum  The REB number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="configure", description="Configure all devices in a REB")
    public void configure(@Argument(name="rebNum", description="REB number")
                          int rebNum) throws DriverException
    {
        ps.configure(rebNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a channel value.
    **
    **  @param  rebNum  The REB number
    **
    **  @param  psNum   The power supply number
    **
    **  @param  chan    The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readchannel", description="Show a channel value")
    public String readChannel(@Argument(name="rebNum", description="REB number")
                              int rebNum,
                              @Argument(name="psNum",
                                        description="Power supply number")
                              int psNum,
                              @Argument(name="chan",
                                        description="Channel number")
                              int chan) throws DriverException
    {
        return String.format("%.3f", ps.readChannel(rebNum, psNum, chan));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all channel values for a power supply.
    **
    **  @param  rebNum  The REB number
    **
    **  @param  psNum   The power supply number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readchannel", description="Show channel values")
    public String readChannel(@Argument(name="rebNum", description="REB number")
                              int rebNum,
                              @Argument(name="psNum",
                                        description="Power supply number")
                              int psNum) throws DriverException
    {
        double[] values = ps.readChannel(rebNum, psNum);
        StringBuilder text = new StringBuilder(String.format("PS%s:", psNum));
        for (double value : values) {
            text.append(String.format("  %.2f", value));
        }
        return text.toString();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all channel values for a REB.
    **
    **  @param  rebNum  The REB number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readchannel", description="Show channel values")
    public String readChannel(@Argument(name="rebNum", description="REB number")
                              int rebNum) throws DriverException
    {
        StringBuilder text = new StringBuilder(readChannel(rebNum, 0));
        for (int psNum = 1; psNum < RebPs.NUM_PS; psNum++) {
            text.append("\n");
            text.append(readChannel(rebNum, psNum));
        }
        return text.toString();
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes to a power supply DAC.
    **
    **  @param  rebNum  The REB number
    **
    **  @param  psNum   The power supply number
    **
    **  @param  value   The value to write
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="writedac", description="Write to a DAC")
    public void writeDac(@Argument(name="rebNum", description="REB number")
                         int rebNum,
                         @Argument(name="psNum",
                                   description="Power supply number")
                         int psNum,
                         @Argument(name="value", description="DAC value")
                         double value) throws DriverException
    {
        ps.writeDac(rebNum, psNum, value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the temperature resolution.
    **
    **  @param  res  The resolution value (0 - 3)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="settempres", description="Set the temperature resolution")
    public void setTemperatureRes(@Argument(name="resolution",
                                            description="Resolution value")
                                  int res) throws DriverException
    {
        ps.setTemperatureRes(res);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the temperature.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showtemp", description="Show the temperature")
    public String showTemperature() throws DriverException
    {
        return String.format("%.2f", ps.readTemperature());
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the power value for a REB.
    **
    **  @param  rebNum  The reb number
    **
    **  @param  action  The action to take (0 - 7)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setpower", description="Sets the power")
    public void setPower(@Argument(name="reb", description="REB number")
                         int rebNum,
                         @Argument(name="action", description="Action value")
                         int action) throws DriverException
    {
        ps.setPower(rebNum, action);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the power value for a REB.
    **
    **  @param  rebNum  The reb number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showpower", description="Show the power")
    public String showPower(@Argument(name="reb", description="REB number")
                            int rebNum) throws DriverException
    {
        return String.format("0x%02x", ps.getPower(rebNum));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the overall power state.
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showpower", description="Show the overall power state")
    public String showPower() throws DriverException
    {
        return String.valueOf(ps.getPowerState());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the error counters.
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showerrors", description="Show the error counters")
    public String showErrors() throws DriverException
    {
        return String.format("Seq: %s   Tmo: %s",
                             ps.getNumSeqErr(), ps.getNumTimeout());
    }


   /**
    *************************************************************************** 
    ** 
    **  Formats an integer array as hexadecimal.
    **
    *************************************************************************** 
    */
    private String formatHex(int addr, int[] data)
    {
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < data.length; j++) {
            if ((j & 3) == 0) {
                if (j > 0) {
                    text.append('\n');
                }
                text.append(String.format("%06x:", addr + j));
            }
            text.append(String.format(" %08x", data[j]));
        }
        return text.toString();
    }

}
