package org.lsst.ccs.drivers.reb;

import java.nio.ByteBuffer;

/**
 *  Image data
 *
 *  @author Owen Saxton
 */
public class Image extends ImageMetadata {

    /*
     *  Public data
     */
    public static final int
        MAX_CCDS = BaseSet.MAX_STRIPS,
        STATUS_GOOD = 0,
        STATUS_MISSING = 1,
        STATUS_INCOMPLETE = 2;

    /*
     *  Private data.
     */
    private long mdata;               // Reference to the C++ metadata object
    private ImageClient.Impl client;  // The image client used
    private ByteBuffer data;          // The image data, old style, interleaved
    private final ByteBuffer[] idata = new ByteBuffer[MAX_CCDS]; // The image data, new style, per CCD
    protected boolean interleaved;      // True if data is interleaved
    private int status;               // Image status


    /*
     *  Constructors.
     */
    public Image() {
        super();
    }

    // For DAQ V2
    public Image(String name, long timestamp, int length, String partition, int address,
                 long sciId, int sensorType, int numCcds, int stripes, int[] registers,
                 int sciVersion, String clientVersion, String serverVersion,
                 String platform, int[] members, ImageClient.Impl client) {
        super(name, timestamp, length, partition, address, sciId, sensorType, numCcds, stripes,
              registers, sciVersion, clientVersion, serverVersion, platform, members);
        interleaved = false;
        this.client = client;
    }

    public Image(String name, long timestamp, int length, String partition, int address,
                 long sciId, int sensorType, int numCcds, int stripes, int[] registers,
                 int sciVersion, String clientVersion, String serverVersion,
                 String platform, int[] members, ByteBuffer[] data) {
        super(name, timestamp, length, partition, address, sciId, sensorType, numCcds, stripes,
              registers, sciVersion, clientVersion, serverVersion, platform, members);
        interleaved = false;
        for (int j = 0; j < idata.length; j++) {
            idata[j] = data != null && j < data.length ? data[j] : null;
        }
    }

    // For DAQ V1
    public Image(String name, long timestamp, int length, int address,
                 long sciId, int sensorType, int stripes, int[] registers,
                 int sciVersion, String clientVersion, String serverVersion,
                 long event, int[] members, long mdata, ImageClient.Impl client) {
        super(name, timestamp, length, address, sciId, sensorType, stripes,
              registers, sciVersion, clientVersion, serverVersion, event, members);
        interleaved = true;
        this.mdata = mdata;
        this.client = client;
    }

    public Image(String name, long timestamp, int length, int address,
                 long sciId, int sensorType, int stripes, int[] registers,
                 int sciVersion, String clientVersion, String serverVersion,
                 long event, int[] members, ByteBuffer data) {
        super(name, timestamp, length, address, sciId, sensorType, stripes,
              registers, sciVersion, clientVersion, serverVersion, event, members);
        interleaved = true;
        this.data = data;
    }

    // For DAQ V0 & PGP Card
    public Image(long tag, int length, int format, int schema, int version,
                 int address, int cluster, int element, long mdata,
                 ImageClient.Impl client) {
        super(tag, length, format, schema, version, address, cluster, element);
        interleaved = true;
        this.mdata = mdata;
        this.client = client;
    }

    public Image(long tag, int length, int format, int schema, int version,
                 int address, int cluster, int element, ByteBuffer data) {
        super(tag, length, format, schema, version, address, cluster, element);
        interleaved = true;
        this.data = data;
    }


    /**
     *  Gets the image pixel data status.
     *
     *  @return  The status code
     */
    public int getStatus() {
        return status;
    }


    /**
     *  Gets whether the image pixel data is interleaved.
     *
     *  @return  Whether interleaved
     */
    public boolean isInterleaved() {
        return interleaved;
    }


    /**
     *  Gets the old-style (interleaved) image pixel data.
     *
     *  @return  The image's pixel data if it has been successfully read, or
     *           null if it hasn't been. The data returned is a read-only
     *           wrapper around the image data to assure the user cannot
     *           change the state of the image.
     */
    public ByteBuffer getData() {
        return (data != null && data.limit() > 0) ? data.asReadOnlyBuffer() : null;
    }


    /**
     *  Gets the new-style (per-CCD) image pixel data.
     *
     *  @param   ccdNum  The CCD number
     *  @return  The image's pixel data if it has been successfully read, or
     *           null if it hasn't been. The data returned is a read-only
     *           wrapper around the image data to assure the user cannot
     *           change the state of the image.
     */
    public ByteBuffer getData(int ccdNum) {
        ByteBuffer dat = idata[ccdNum];
        return (dat != null && dat.limit() > 0) ? dat.asReadOnlyBuffer() : null;
    }


    /**
     *  Gets the image metadata.
     *
     *  @return  The image's metadata
     */
    public ImageMetadata getMetadata() {
        return new ImageMetadata(name, timestamp, length, partition, address, sciId,
                                 sensorType, numCcds, stripes, registers, sciVersion,
                                 clientVersion, serverVersion, platform, members);
    }


    /**
     *  Deletes the metadata reference.
     */
    public void deleteMetadataRef() {
        if (mdata != 0) {
            client.deleteImageMetadataRef(this);
        }
    }


    /**
     *  Performs finalization.
     *
     *  Allocated non-java memory (in this case the C++ image metadata object)
     *  has to be freed.
     *
     *  @throws Throwable
     */
    @Override
    public void finalize() throws Throwable {
        super.finalize();
        deleteMetadataRef();
    }

    private static final long serialVersionUID = 1313601326490669879L;
}
