package org.lsst.ccs.drivers.reb;


/**
 *  Routines for reading the RTD temperature sensors.
 *
 *  These are read using an Analog Devices AD7794 chip which requires a
 *  certain amount of setup.
 *
 *  @author Owen Saxton
 */
public class TempRtds {

    /**
     *  Public constants
     */
    public static final int
        REG_RTD_COMMAND  = 0x700000,
        REG_RTD_REPLY    = 0x700001,
        REG_RTD_RESET    = 0x700002,
        CMND_READ        = 0x80000,
        NUM_RTD_TEMPS    = 4,
        CCD_TYPE_ITL     = 1,
        CCD_TYPE_E2V     = 2;
 
    public static final int
        STATUS_REG     = 0,        // Status register (8-bit)
        MODE_REG       = 1,        // Mode register (16-bit)
        CONFIG_REG     = 2,        // Configuration register (16-bit)
        DATA_REG       = 3,        // Data register (24- or 16-bit)
        ID_REG         = 4,        // ID register (8-bit)
        IO_REG         = 5,        // IO register (8-bit)
        OFFSET_REG     = 6,        // Offset register (24- or 16-bit)
        SCALE_REG      = 7,        // (Full-)Scale register (24- or 16-bit)

        STS_READY      = 0x80,     // Status: (not) ready
        STS_ERROR      = 0x40,     // Status: voltage/temp data ready
        STS_NOXREF     = 0x20,     // Status: no external reference
        STS_AD7794     = 0x08,     // Status: AD7794 chip, not AD7795
        STS_CHANNEL_M  = 0x07,     // Status: channel mask
        STS_CHANNEL_V  = 0,        // Status: channel offset

        MOD_MODE_M     = 0xe000,   // Mode: mode select mask
        MOD_MODE_V     = 13,       // Mode: mode select offset
        MODE_CONT      = 0,        // Mode select: continuous conversion
        MODE_SINGLE    = 1,        // Mode select: single conversion
        MODE_IDLE      = 2,        // Mode select: idle
        MODE_PWR_DOWN  = 3,        // Mode select: power-down
        MODE_INT_Z_CAL = 4,        // Mode select: internal zero calibration
        MODE_INT_F_CAL = 5,        // Mode select: int. full-scale calibration
        MODE_SYS_Z_CAL = 6,        // Mode select: system zero calibration
        MODE_SYS_F_CAL = 7,        // Mode select: sys. full-scale calibration
        MOD_PSW        = 0x1000,   // Mode: power switch
        MOD_AMP_CM     = 0x0200,   // Mode: amplifier common mode
        MOD_CLOCK_M    = 0x00c0,   // Mode: clock source mask
        MOD_CLOCK_V    = 6,        // Mode: clock source offset
        CLOCK_INT      = 0,        // Clock source: internal 64 KHz
        CLOCK_INT_AVL  = 1,        // Clock source: int 64 KHz, avail on pin
        CLOCK_EXT      = 2,        // Clock source: external 64 KHz
        CLOCK_EXT_HALF = 3,        // Clock source: ext 64 KHz, halved
        MOD_CHOP_DIS   = 0x0010,   // Mode: chop disable
        MOD_FRS_M      = 0x000f,   // Mode: filter rate select mask
        MOD_FRS_V      = 0,        // Mode: filter rate select offset
        FRS_RATE_470   = 1,        // Filter rate: 470 Hz
        FRS_RATE_242   = 2,        // Filter rate: 242 Hz
        FRS_RATE_123   = 3,        // Filter rate: 123 Hz
        FRS_RATE_62    = 4,        // Filter rate: 62 Hz
        FRS_RATE_50    = 5,        // Filter rate: 50 Hz
        FRS_RATE_39    = 6,        // Filter rate: 39 Hz
        FRS_RATE_33    = 7,        // Filter rate: 33.2 Hz
        FRS_RATE_19    = 8,        // Filter rate: 19.6 Hz
        FRS_RATE_17    = 9,        // Filter rate: 16.7 Hz (80 dB rej'n)
        FRS_RATE_16    = 10,       // Filter rate: 16.7 Hz (65 dB rej'n)
        FRS_RATE_12    = 11,       // Filter rate: 12.5 Hz
        FRS_RATE_10    = 12,       // Filter rate: 10 Hz
        FRS_RATE_8     = 13,       // Filter rate: 8.33 Hz
        FRS_RATE_6     = 14,       // Filter rate: 6.25 Hz
        FRS_RATE_4     = 15,       // Filter rate: 4.17 Hz

        CFG_VBIAS_M    = 0xc000,   // Config: bias voltage mask
        CFG_VBIAS_V    = 14,       // Config: bias voltage offset
        VBIAS_OFF      = 0,        // Bias voltage: disabled
        VBIAS_AIN1     = 1,        // Bias voltage: connected to AIN1(-)
        VBIAS_AIN2     = 2,        // Bias voltage: connected to AIN2(-)
        VBIAS_AIN3     = 3,        // Bias voltage: connected to AIN3(-)
        CFG_BURNOUT    = 0x2000,   // Config: burnout current enabled
        CFG_UNIPOLAR   = 0x1000,   // Config: unipolar operation
        CFG_BOOST      = 0x0800,   // Config: boost bias voltage current
        CFG_GAIN_M     = 0x0700,   // Config: gain select mask
        CFG_GAIN_V     = 8,        // Config: gain select offset
        GAIN_1         = 0,        // ADC gain: 1
        GAIN_2         = 1,        // ADC gain: 2
        GAIN_4         = 2,        // ADC gain: 4
        GAIN_8         = 3,        // ADC gain: 8
        GAIN_16        = 4,        // ADC gain: 16
        GAIN_32        = 5,        // ADC gain: 32
        GAIN_64        = 6,        // ADC gain: 64
        GAIN_128       = 7,        // ADC gain: 128
        RANGE_2500     = GAIN_1,   // ADC range: 2.5 V
        RANGE_1250     = GAIN_2,   // ADC range: 1.25 V
        RANGE_625      = GAIN_4,   // ADC range: 0.625 V
        RANGE_312      = GAIN_8,   // ADC range: 0.3125 V
        RANGE_156      = GAIN_16,  // ADC range: 0.15625 V
        RANGE_78       = GAIN_32,  // ADC range: 0.078125 V
        RANGE_39       = GAIN_64,  // ADC range: 0.0390625 V
        RANGE_19       = GAIN_128, // ADC range: 0.01953125 V
        CFG_REFSEL_M   = 0x00c0,   // Config: reference select mask
        CFG_REFSEL_V   = 6,        // Config: reference select offset
        REFSEL_EXT1    = 0,        // Ref select: external on REFIN1
        REFSEL_EXT2    = 1,        // Ref select: external on REFIN2
        REFSEL_INT     = 2,        // Ref select: internal
        CFG_REF_DETECT = 0x0020,   // Config: reference detect enable
        CFG_BUFFERED   = 0x0010,   // Config: buffered mode
        CFG_CHANNEL_M  = 0x000f,   // Config: channel select mask
        CFG_CHANNEL_V  = 0,        // Config: channel select offset
        CHAN_AIN1      = 0,        // Channel: AIN1
        CHAN_AIN2      = 1,        // Channel: AIN2
        CHAN_AIN3      = 2,        // Channel: AIN3
        CHAN_AIN4      = 3,        // Channel: AIN4
        CHAN_AIN5      = 4,        // Channel: AIN5
        CHAN_AIN6      = 5,        // Channel: AIN6
        CHAN_TEMP      = 6,        // Channel: temperature sensor
        CHAN_VDD       = 7,        // Channel: VDD monitor
        CHAN_AIN1A     = 8,        // Channel: AIN1 (again)

        IO_DIOENA      = 0x40,     // IO: digital IO enable
        IO_IO2DAT      = 0x20,     // IO: pin P2 data
        IO_IO1DAT      = 0x10,     // IO: pin P1 data
        IO_IEXCDIR_M   = 0x0c,     // IO: current sources direction mask
        IO_IEXCDIR_V   = 2,        // IO: current sources direction offset
        IEXCDIR_12     = 0,        // Sources direction: IOUT1 & IOUT2
        IEXCDIR_21     = 1,        // Sources direction: IOUT2 & IOUT1
        IEXCDIR_11     = 2,        // Sources direction: both to IOUT1
        IEXCDIR_22     = 3,        // Sources direction: both to IOUT2
        IO_IEXCENA_M   = 0x03,     // IO: current sources enable mask
        IO_IEXCENA_V   = 0,        // IO: current sources enable offset
        IEXCENA_OFF    = 0,        // Sources enable: disabled
        IEXCENA_10     = 1,        // Sources enable: 10 uA
        IEXCENA_210    = 2,        // Sources enable: 210 uA
        IEXCENA_1000   = 3;        // Sources enable: 1 mA

    /**
     *  Private data
     */
    private static final int[]
        REG_SIZE = {1, 2, 2, 3, 1, 1, 3, 3};   // Array of register sizes

    private static final int
//        MODE_BASE = (FRS_RATE_50 << MOD_FRS_V) | (MODE_SINGLE << MOD_MODE_V),
        MODE_BASE = (FRS_RATE_470 << MOD_FRS_V) | (MODE_SINGLE << MOD_MODE_V),
        CONFIG_BASE = CFG_BUFFERED /*| CFG_UNIPOLAR*/ | (GAIN_1 << CFG_GAIN_V),
        IO_BASE = (IEXCENA_210 << IO_IEXCENA_V) | (IEXCDIR_12 << IO_IEXCDIR_V),
        READY_WAIT = 1,         // Make ~400 / (nn in FRS_RATE_nn)
        READY_TIMEOUT = 5000;

    private static final double
        HALF_RANGE      = 8388608.0,  // 0x800000
        INT_REF_VOLTAGE = 1.17,
        INT_TEMP_SCALE  = 100.0,
        INT_VOLT_SCALE  = 6.0;

    private static final double
        REF_RESIST      = 3000.0,
        RTD_RESIST_1000 = 1000.0,
        RTD_RESIST_500  = 500.0,
        RTD_RESIST_100  = 100.0,
        RTD_WREB_CORRN  = -8.0,
        TEMP_CONV       = 0.003850;

    private final BaseSet bss;
    private boolean forceScience = false;


    /**
     *  Constructor.
     *
     *  @param  bss  The associated base set object
     */
    public TempRtds(BaseSet bss)
    {
        this.bss = bss;
    }


    /**
     *  Sets whether a science CCD is being used.
     *
     *  Needed when a WREB is using a science CCD
     *
     *  @param  state  Whether a science CCD is being used
     */
    public void setScienceCCD(boolean state)
    {
        forceScience = state;
    }


    /**
     *  Initializes for RTD reading.
     *
     *  Not really necessary any more (4/5/17)
     *
     *  @throws  REBException
     */
    public void initialize() throws REBException
    {
        //reset();   // Probably not a good idea - should be unnecessary
        writeRegister(MODE_REG, MODE_BASE);
        writeRegister(CONFIG_REG, CONFIG_BASE);
        writeRegister(IO_REG, IO_BASE);
    }


    /**
     *  Gets the CCD type.
     *
     *  This uses the measured resistance of the first RTD.  E2V CCDs have
     *  a PT100 RTD; ITLs have either a PT1000 or PT500 RTD.
     *
     *  @return  The encoded CCD type (ITL or E2V)
     *  @throws  REBException If the resistance cannot be read, or the type cannot be determined.
     */
    public int getCCDType() throws REBException
    {
        double resist = readResistance(0);
        int result =  resist < 0.3 * RTD_RESIST_100 ? 0 :
               resist < 2 * RTD_RESIST_100 ? CCD_TYPE_E2V :
               resist < 2 * RTD_RESIST_1000 ? CCD_TYPE_ITL : 0;
        if (result == 0) {
            throw new REBException("CCD type cannot be determined, measured resistance is "+resist);
        }
        return result;
    }


    /**
     *  Reads an RTD temperature.
     *
     *  There are three possible RTD types:
     *    PT1000: Used by ITL science and guiding CCDs
     *    PT500:  Used by wavefront CCDs
     *    PT100:  Used by the raft RTD and by E2V science and guiding CCDs
     *
     *  @param  chan  The RTD channel number: 0 - 2 for CCDs; 3 for RTM RTD
     *  @return  The temperature (C), or NaN if conversion failed
     *  @throws  REBException
     */
    public double readTemperature(int chan) throws REBException
    {
        if (chan < 0 || chan >= NUM_RTD_TEMPS) {
            throw new REBException("Invalid RTD channel");
        }
        chan = (chan == NUM_RTD_TEMPS - 1) ? CHAN_AIN6 : bss.convertStripNum(chan);
        boolean isWreb = chan != CHAN_AIN6 && !forceScience && bss.getRebType() == BaseSet.REB_TYPE_WAVEFRONT;
        double resist = readResistance(chan) + (isWreb ? RTD_WREB_CORRN : 0.0);
        double nomResist = isWreb ? RTD_RESIST_500 : (resist < 2 * RTD_RESIST_100) ? RTD_RESIST_100 : RTD_RESIST_1000;
        return (resist / nomResist - 1.0) / TEMP_CONV;
    }


    /**
     *  Reads an RTD resistance.
     *
     *  @param  chan  The RTD channel number: 0 - 2 for CCDs; 3 for RTM RTD
     *  @return  The resistance (ohms), or NaN if conversion failed
     *  @throws  REBException
     */
    private synchronized double readResistance(int chan) throws REBException
    {
        int ref = chan == CHAN_AIN6 && bss.isVersion(BaseSet.OPTN_CCD_TEMP, BaseSet.VERSION_1) ? REFSEL_EXT2 : REFSEL_EXT1;
        writeRegister(IO_REG, IO_BASE);
        writeRegister(CONFIG_REG, CONFIG_BASE | (chan << CFG_CHANNEL_V) | (ref << CFG_REFSEL_V));
        writeRegister(MODE_REG, MODE_BASE);
        if (waitReady()) {
            return REF_RESIST * Math.abs(readRegister(DATA_REG) / HALF_RANGE - 1.0);
        }
        else {
            return BaseSet.ERROR_VALUE;
        }
    }


    /**
     *  Reads the internal temperature.
     *
     *  @return  The temperature (C), or NaN if conversion failed
     *  @throws  REBException
     */
    public synchronized double readIntTemperature() throws REBException
    {
        writeRegister(CONFIG_REG, CONFIG_BASE | (CHAN_TEMP << CFG_CHANNEL_V));
        writeRegister(MODE_REG, MODE_BASE);
        if (waitReady()) {
            return (readRegister(DATA_REG) / HALF_RANGE - 1.0) * INT_REF_VOLTAGE * INT_TEMP_SCALE;
        }
        else {
            return BaseSet.ERROR_VALUE;
        }
    }


    /**
     *  Reads the internal voltage.
     *
     *  @return  The voltage, or NaN if conversion failed
     *  @throws  REBException
     */
    public synchronized double readIntVoltage() throws REBException
    {
        writeRegister(CONFIG_REG, CONFIG_BASE | (CHAN_VDD << CFG_CHANNEL_V));
        writeRegister(MODE_REG, MODE_BASE);
        if (waitReady()) {
            return (readRegister(DATA_REG) / HALF_RANGE - 1.0) * INT_REF_VOLTAGE * INT_VOLT_SCALE;
        }
        else {
            return BaseSet.ERROR_VALUE;
        }
    }


    /**
     *  Resets the chip.
     *
     *  @throws  REBException
     */
    public void reset() throws REBException
    {
        bss.write(REG_RTD_RESET, 0);
        try {
            Thread.sleep(0, 500_000);  // Try for 500 uS, the recommended time
        }
        catch (InterruptedException e) {
        }
    }


    /**
     *  Reads a register.
     *
     *  @param  regnum   The register to read
     *  @return  The value read
     *  @throws  REBException
     */
    public int readRegister(int regnum) throws REBException
    {
        int reg = regnum & 0x07;
        bss.write(REG_RTD_COMMAND, CMND_READ | (reg << 16));
        return bss.read(REG_RTD_REPLY) & ((1 << (8 * REG_SIZE[reg])) - 1);
    }


    /**
     *  Writes a register
     *
     *  @param  regnum  The register to write
     *  @param  value   The value to write
     *  @throws  REBException
     */
    public void writeRegister(int regnum, int value) throws REBException
    {
        int reg = regnum & 0x07;
        int val = (REG_SIZE[reg] == 1) ? (value & 0xff) << 8 : value & 0xffff;
        bss.write(REG_RTD_COMMAND, reg << 16 | val);
    }


    /**
     *  Waits for the chip to become ready after a conversion
     *
     *  @return  Whether conversion was successful
     *  @throws  REBException
     */
    public boolean waitReady() throws REBException
    {
        long start = System.currentTimeMillis();
        boolean ready = false;
        int status = readRegister(STATUS_REG);
        while (!(ready = ((status & STS_READY) == 0))
                 && System.currentTimeMillis() - start <= READY_TIMEOUT) {
            try {
                Thread.sleep(READY_WAIT);
            }
            catch (InterruptedException e) {
            }
            status = readRegister(STATUS_REG);
        }
        if (!ready) {
            throw new REBException("Conversion timed out");
        }
        return (status & STS_ERROR) == 0;
    }

}
