package org.lsst.ccs.drivers.reb.sim;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.GlobalClient;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.TestReg;

/**
 * A simulation of the DAQ v1 GlobalClient
 *
 * @author tonyj
 */
public class GlobalClientSimulation implements GlobalClient.Impl {

    private final HandleAndIdManager manager;
    private List<AddressSpace> addressSpaces;
    private final DAQSimulation daq;
    private boolean initializeOnTrigger = false;
    private boolean supportMultiMains;

    GlobalClientSimulation(HandleAndIdManager manager, DAQSimulation daq, boolean supportMultiMains) {
        this.manager = manager;
        this.daq = daq;
        this.supportMultiMains = supportMultiMains;
    }

    //Implementation of default setRegisterList method.
    //This notifies the GlobalClient of which registers contain
    //image metadata
    @Override
    public void setRegisterList(int rebType, int[] registers) {
        daq.setRegisterList(rebType, registers);
    }

    @Override
    public long triggerImage(String name) {
        initializeAddressSpaces("triggerImage");
        daq.setImageName(name);
        for (AddressSpace as : addressSpaces) {
            // FIXME: We should have some lock around this, or a better way of triggering
            int state = as.read(BaseSet.REG_STATE);
            state |= 1 << BaseSet.RSET_SEQUENCER;
            as.write(BaseSet.REG_TRIGGER, state);
        }
        return 1_000_000 * System.currentTimeMillis();
    }

    private void initializeAddressSpaces(String name) throws SimulationException {
        if (addressSpaces == null) {
            if (initializeOnTrigger) {
                addressSpaces = new ArrayList<>();
                for (AddressSpace as : manager.getAllAddressSpaces()) {
                    addressSpaces.add(as);
                }
            } else {
                throw new SimulationException(name + " called but no ids registered");
            }
        }
    }

    @Override
    public long startSequencer(int opcode) throws REBException {
        if (supportMultiMains) {
            initializeAddressSpaces("startSequencer");
            for (AddressSpace as : addressSpaces) {
                if (as instanceof REB3Simulation) {
                    REB3Simulation reb = (REB3Simulation) as;
                    reb.getSequencer().trigger(opcode);
                }
            }
        }
        return 1_000_000 * System.currentTimeMillis();
    }

    @Override
    public long triggerImage(String name, String folder, int opcode, String annotation, int... ids) throws REBException {
        if (!supportMultiMains) {
            return triggerImage(name);
        } else {
            initializeAddressSpaces("triggerImage");
            daq.setImageName(name);
            for (AddressSpace as : addressSpaces) {
                if (as instanceof REB3Simulation) {
                    REB3Simulation reb = (REB3Simulation) as;
                    reb.getSequencer().trigger(opcode);
                }
            }
            return 1_000_000 * System.currentTimeMillis();
        }
    }

    @Override
    public void deleteGlobalClient() {
        addressSpaces = null;
    }

    @Override
    public void newGlobalClient(int[] ids, String[] ifc) throws REBException {
        if (initializeOnTrigger || addressSpaces != null) {
            throw new SimulationException("Unsupported repeat call to newGlobalClient");
        }
        List<AddressSpace> seqs = new ArrayList<>();
        // Locate the sequencers with the given ids.
        for (int id : ids) {
            AddressSpace as = manager.getAddressSpaceForId(id);
            seqs.add(as);
        }
        this.addressSpaces = seqs;
    }

    @Override
    public void newGlobalClient(String part) throws REBException {
        if (initializeOnTrigger || addressSpaces != null) {
            throw new SimulationException("Unsupported repeat call to newGlobalClient");
        }
        initializeOnTrigger = true;
    }

    @Override
    public int getHwType() {
        return supportMultiMains ? TestReg.HdwType.DAQ4.getValue() :  TestReg.HdwType.DAQ1.getValue();
    }
}
