package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * A factory class for creating RegClient and ImageClient instances. Replacing
 * the default implementation with an alternative allows simulated clients to be
 * substituted for the real ones.
 *
 * @author tonyj
 */
public class ClientFactory {

    private static final String libPathPropertyName = "org.lsst.ccs.ld.library.path";
    private static final Map<String, Integer> daqTypeMap = new HashMap<>();
    static {
        daqTypeMap.put("Daq1", RegClient.HDW_TYPE_DAQ1);
        daqTypeMap.put("Daq2", RegClient.HDW_TYPE_DAQ2);
        daqTypeMap.put("Daq4", RegClient.HDW_TYPE_DAQ4);
    }
    private static final Map<Integer, String> invDaqTypeMap = new HashMap<>();
    static {
        for (String name : daqTypeMap.keySet()) {
            invDaqTypeMap.put(daqTypeMap.get(name), name);
        }
    }

    private static final Logger LOG = Logger.getLogger(ClientFactory.class.getName());
    private static boolean daqLibReported = false;

    public RegClient.Impl createRegClient(int hdw) throws REBException {
        int hdwType = checkHdwType(hdw);
        switch (hdwType) {
        case RegClient.HDW_TYPE_DAQ0:
            return new Daq0Client.Registers();
        case RegClient.HDW_TYPE_DAQ1:
            return new Daq1Client.Registers();
        case RegClient.HDW_TYPE_DAQ2:
            return new Daq2Client.Registers();
        case RegClient.HDW_TYPE_DAQ4:
            return new Daq4Client.Registers();
        case RegClient.HDW_TYPE_PCI0:
        case RegClient.HDW_TYPE_PCI1:
            return new PciClient.Registers();
        default:
            throw new REBException("Invalid hardware type (" + hdwType + ")");
        }
    }

    public ImageClient.Impl createImageClient(int hdw, RegClient reg) throws REBException {
        int hdwType = checkHdwType(hdw);
        switch (hdwType) {
        case RegClient.HDW_TYPE_DAQ0:
            return new Daq0Client.Images();
        case RegClient.HDW_TYPE_DAQ1:
            return new Daq1Client.Images();
        case RegClient.HDW_TYPE_DAQ2:
            return new Daq2Client.Images();
        case RegClient.HDW_TYPE_DAQ4:
            return new Daq4Client.Images();
        case RegClient.HDW_TYPE_PCI0:
            return new PciClient.Images(reg, 0);
        case RegClient.HDW_TYPE_PCI1:
            return new PciClient.Images(reg, 1);
        default:
            throw new REBException("Invalid hardware type (" + hdwType + ")");
        }
    }

    public GlobalClient.Impl createGlobalClient(int hdw) throws REBException {
        int hdwType = checkHdwType(hdw);
        switch (hdwType) {
        case RegClient.HDW_TYPE_DAQ0:
            return new Daq0Client.Global();
        case RegClient.HDW_TYPE_DAQ1:
            return new Daq1Client.Global();
        case RegClient.HDW_TYPE_DAQ2:
            return new Daq2Client.Global();
        case RegClient.HDW_TYPE_DAQ4:
            return new Daq4Client.Global();
        case RegClient.HDW_TYPE_PCI0:
        case RegClient.HDW_TYPE_PCI1:
            return new PciClient.Global();
        default:
            throw new REBException("Invalid hardware type (" + hdwType + ")");
        }
    }

    private static int checkHdwType(int hdw) {
        if (hdw == RegClient.HDW_TYPE_DAQ0 || hdw == RegClient.HDW_TYPE_PCI0 || hdw == RegClient.HDW_TYPE_PCI1) return hdw;
        Properties props = BootstrapResourceUtils.getBootstrapSystemProperties();
        String libPath = props.getProperty(libPathPropertyName);
        if (libPath == null) {
            LOG.severe("Bootstrap property " + libPathPropertyName + " not defined");
            return hdw;
        }
        Integer libType = daqTypeMap.get(libPath);
        if (libType == null) {
            LOG.log(Level.SEVERE,"Bootstrap property " + libPathPropertyName + " has invalid value ({0})", libPath);
            return hdw;
        }
        if (hdw == RegClient.HDW_TYPE_DAQ) {
            if (!daqLibReported) {
                LOG.log(Level.INFO, "Using {0} library", libPath);
                daqLibReported = true;
            }
            return libType;
        }
        if (hdw != libType) {
            LOG.log(Level.SEVERE, "{0} libray doesn''t match requested hardware type ({1})",
                    new Object[]{libPath, invDaqTypeMap.get(hdw)});
        }
        return hdw;
    }

}
