package org.lsst.ccs.drivers.reb.sim;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.GlobalClient;
import org.lsst.ccs.drivers.reb.REBException;

/**
 * A simulation of the DAQ v1 GlobalClient
 *
 * @author tonyj
 */
public class GlobalClientSimulation implements GlobalClient.Impl {

    private final HandleAndIdManager manager;
    private List<AddressSpace> addressSpaces;
    private final DAQSimulation daq;
    private boolean initializeOnTrigger = false;

    GlobalClientSimulation(HandleAndIdManager manager, DAQSimulation daq) {
        this.manager = manager;
        this.daq = daq;
    }

    //Implementation of default setRegisterList method.
    //This notifies the GlobalClient of which registers contain
    //image metadata
    @Override
    public void setRegisterList(int rebType, int[] registers) {
        daq.setRegisterList(rebType, registers);
    }

    @Override
    public long triggerImage(String name) {
        if (addressSpaces == null) {
            if (initializeOnTrigger) {
                addressSpaces = new ArrayList<>();
                for (AddressSpace as : manager.getAllAddressSpaces()) {
                    addressSpaces.add(as);
                }
            }
            else {
                throw new SimulationException("triggerImage called but no ids registered");
            }
        }
        daq.setImageName(name);
        for (AddressSpace as : addressSpaces) {
            // FIXME: We should have some lock around this, or a better way of triggering
            int state = as.read(BaseSet.REG_STATE);
            state |= 1<<BaseSet.RSET_SEQUENCER;
            as.write(BaseSet.REG_TRIGGER, state);
        }
        return 1_000_000 * System.currentTimeMillis();
    }

    @Override
    public void deleteGlobalClient() {
        addressSpaces = null;
    }

    @Override
    public void newGlobalClient(int[] ids, String[] ifc) throws REBException {
        if (initializeOnTrigger || addressSpaces != null) {
            throw new SimulationException("Unsupported repeat call to newGlobalClient");
        }
        List<AddressSpace> seqs = new ArrayList<>();
        // Locate the sequencers with the given ids.
        for (int id : ids) {
            AddressSpace as = manager.getAddressSpaceForId(id);
            seqs.add(as);
        }
        this.addressSpaces = seqs;
    }

    @Override
    public void newGlobalClient(String part) throws REBException {
        if (initializeOnTrigger || addressSpaces != null) {
            throw new SimulationException("Unsupported repeat call to newGlobalClient");
        }
        initializeOnTrigger = true;
    }
}
