package org.lsst.ccs.drivers.reb.sim;

import java.util.Random;
import org.lsst.ccs.drivers.reb.Asic;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.SlowAdcs;
import org.lsst.ccs.drivers.reb.TempAdcs;
import org.lsst.ccs.drivers.reb.TempRtds;

/**
 * A simulation of a REB3 device.
 *
 * @author tonyj
 * @see
 * <a href="https://confluence.slac.stanford.edu/display/LSSTCAM/REB+3+registers+sets">Sequencer
 * User Guide</a>
 *
 */
public class REB3Simulation extends AddressSpace {

    @SuppressWarnings("PointlessBitwiseExpression")
    private static final int MASK_RSET_STATUS = 1 << BaseSet.RSET_STATUS;
    private static final int MASK_RSET_SEQUENCER = 1 << BaseSet.RSET_SEQUENCER;
    private static final int MASK_RSET_TEMP_ADCS = 1 << BaseSet.RSET_TEMP_ADCS;
    private static final int MASK_RSET_TIME_BASE = 1 << BaseSet.RSET_TIME_BASE;
    private static final int MASK_RSET_POWER_ADCS = 1 << BaseSet.RSET_POWER_ADCS;
    private static final int MASK_RSET_FAST_ADCS = 1 << BaseSet.RSET_FAST_ADCS;
    private static final int MASK_MBZ = ~(MASK_RSET_SEQUENCER | MASK_RSET_STATUS | MASK_RSET_TEMP_ADCS | MASK_RSET_TIME_BASE | MASK_RSET_POWER_ADCS | MASK_RSET_FAST_ADCS);

    private static final int RESERVED = 0;
    private static final int NUM_STRIPS = 3;

    private volatile boolean timeEnabled;
    private long baseTime;
    private long enableTime;
    private final long[] triggerTime = new long[32];
    private long serial;
    private final Random random = new Random();
    private SequencerSimulation sequencer;
    private SensorsSimulation sensors;
    private AspicSimulation aspics;

    private boolean backBias = false;
    private final int[][] boardDacs = new int[NUM_STRIPS][6];
    private RTDSimulation rtds;

    /**
     * Create a reb3 simulator.
     *
     * @param id The unique id of this reb
     */
    public REB3Simulation(int id) {
        init(id);
    }

    private void init(int id) {
        add(new RegisterSet.ConstantRegister(BaseSet.REG_SCHEMA, 0));
        add(new RegisterSet.ConstantRegister(BaseSet.REG_VERSION, 0xb0200103));
        add(new RegisterSet.ConstantRegister(BaseSet.REG_ID, id));
        add(new RegisterSet.ConstantRegister(3, RESERVED));
        add(new TimeBaseRegisterSet(BaseSet.REG_TIME_BASE));
        add(new RegisterSet.ConstantRegister(6, RESERVED));
        add(new RegisterSet.ConstantRegister(7, RESERVED));
        add(new StateRegister(BaseSet.REG_STATE));
        add(new TriggerRegister(BaseSet.REG_TRIGGER));
        add(new RegisterSet.LongArrayRegisterSet(BaseSet.REG_TRIG_TIME, triggerTime));

        add(new StartRebSerialNumberRegister(BaseSet.REG_SN_REB_START));
        add(new SerialNumberRegister(BaseSet.REG_SN_REB_VALUE));

        add(new BackBiasRegister(BaseSet.REG_BACK_BIAS));

        for (int stripe = 0; stripe < 3; stripe++) {
            add(new BoardDacsRegister(stripe));
            add(new BoardDacsActivateRegister(stripe));
        }

        sensors = new SensorsSimulation(this);
        aspics = new AspicSimulation(this);
        sequencer = new SequencerSimulation(this);
        rtds = new RTDSimulation(this);

        sequencer.addStateListener((SequencerSimulation.State oldState, SequencerSimulation.State newState) -> {
            if (!newState.isRunning()) {
                triggerTime[BaseSet.RSET_SEQUENCER] = getTime();
            }
        });
    }

    private void setEnableTimer(boolean enabled) {
        if (timeEnabled != enabled) {
            if (enabled) {
                triggerTime[BaseSet.RSET_TIME_BASE] = getTime();
                enableTime = System.nanoTime();
                timeEnabled = true;
            } else {
                baseTime = getTime();
                timeEnabled = false;
                triggerTime[BaseSet.RSET_TIME_BASE] = getTime();
            }
        }
    }

    private void setStartSequencer(boolean enabled) {
        if (enabled) {
            triggerTime[BaseSet.RSET_SEQUENCER] = getTime();
            sequencer.start();
        }
    }

    private long getTime() {
        if (timeEnabled) {
            long nanosSinceEnabled = System.nanoTime() - enableTime;
            // FIXME: Check clock period for REB3
            return baseTime + nanosSinceEnabled / BaseSet.CLOCK_PERIOD_1;
        } else {
            return baseTime;
        }
    }

    /**
     * Access the sequencer within this reb3
     *
     * @return The sequencer.
     */
    public SequencerSimulation getSequencer() {
        return sequencer;
    }

    long getTriggerTime(int offset) {
        return triggerTime[offset];
    }

    private class TimeBaseRegisterSet extends RegisterSet.SimpleLongRegisterSet {

        TimeBaseRegisterSet(int address) {
            super(address, 1);
        }

        @Override
        long readLong(int index) {
            return getTime();
        }

        @Override
        void writeLong(int index, long longValue) {
            if (timeEnabled) {
                throw new SimulationException("Attempt to write to TimeBase while enabled");
            }
            baseTime = longValue;
        }
    }

    private class StateRegister extends RegisterSet.ReadOnlyRegister {

        public StateRegister(int address) {
            super(address);
        }

        @Override
        public int read() {
            int result = 0;
            if (timeEnabled) {
                result |= MASK_RSET_TIME_BASE;
            }
            if (sequencer.getState().isRunning()) {
                result |= MASK_RSET_SEQUENCER;
            }
            return result;
        }
    }

    private class TriggerRegister extends RegisterSet.Register {

        public TriggerRegister(int address) {
            super(address);
        }

        @Override
        public int read() {
            int result = 0;
            if (timeEnabled) {
                result |= MASK_RSET_TIME_BASE;
            }
            if (sequencer.getState().isRunning()) {
                result |= MASK_RSET_SEQUENCER;
            }
            return result;
        }

        @Override
        public void write(int value) {
            if ((value & MASK_MBZ) != 0) {
                throw new SimulationException("Illegal trigger bit set %08x", value);
            }
            setEnableTimer((value & MASK_RSET_TIME_BASE) != 0);
            setStartSequencer((value & MASK_RSET_SEQUENCER) != 0);
        }
    }

    private class StartRebSerialNumberRegister extends RegisterSet.WriteOnlyRegister {

        StartRebSerialNumberRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            // Set serial number
            serial = random.nextLong() & 0xffffffffffffL;
            serial |= 1L << 48;
        }

    }

    private class SerialNumberRegister extends RegisterSet.ReadOnlyLongRegister {

        SerialNumberRegister(int address) {
            super(address);
        }

        @Override
        public long read() {
            return serial;
        }
    }

    private class BackBiasRegister extends RegisterSet.Register {

        BackBiasRegister(int address) {
            super(address);
        }

        @Override
        public int read() {
            return backBias ? 1 : 0;
        }

        @Override
        public void write(int value) {
            backBias = (value & 1) != 0;
        }
    }

    private class BoardDacsRegister extends RegisterSet.WriteOnlyRegister {

        private final int stripe;

        BoardDacsRegister(int stripe) {
            super(BoardDacs.REG_BOARD_DACS + (stripe << 4) + 0x100);
            this.stripe = stripe;
        }

        @Override
        public void write(int value) {
            int o = (value >>> 12) & 0xf;
            boardDacs[stripe][o] = value & 0xfff;
        }
    }

    private class BoardDacsActivateRegister extends RegisterSet.WriteOnlyRegister {

        private final int stripe;

        BoardDacsActivateRegister(int stripe) {
            super(BoardDacs.REG_BOARD_DACS + (stripe << 4) + 0x101);
            this.stripe = stripe;
        }

        @Override
        public void write(int value) {
            // Nothing to do really
        }
    }

    private static class SensorsSimulation {

        SensorsSimulation(AddressSpace addressSpace) {

            addressSpace.add(new VoltageSensorRegister(0, 5));
            addressSpace.add(new VoltageSensorRegister(2, 7));
            addressSpace.add(new VoltageSensorRegister(4, 15));
            addressSpace.add(new VoltageSensorRegister(6, 40));
            addressSpace.add(new CurrentSensorRegister(1, .01f, 25e-6f));
            addressSpace.add(new CurrentSensorRegister(3, .01f, 25e-6f));
            addressSpace.add(new CurrentSensorRegister(5, .001f, 8e-6f));
            addressSpace.add(new CurrentSensorRegister(7, .001f, 8e-6f));

            // Temperatures
            for (int offset = 0; offset < TempAdcs.NUM_TEMP_REGS_M; offset++) {
                addressSpace.add(new TemperatureSensorRegister(offset, 25));
            }
        }

        private class VoltageSensorRegister extends RegisterSet.ReadOnlyRegister {

            private final Random r = new Random();
            private final float delta;
            private final float voltage;

            VoltageSensorRegister(int offset, float voltage) {
                super(PowerAdcs.REG_POWER_ADCS + offset);
                this.voltage = voltage;
                delta = ((float)0.1)*voltage;
            }

            @Override
            public int read() {
                return (int) ((voltage +r.nextFloat()*delta) / 0.025);
            }
        }

        private class CurrentSensorRegister extends RegisterSet.ReadOnlyRegister {

            private final float current;
            private final float resolution;
            private final Random r = new Random();
            private final float delta;

            CurrentSensorRegister(int offset, float current, float resolution) {
                super(PowerAdcs.REG_POWER_ADCS + offset);
                this.current = current;
                this.resolution = resolution;
                delta = ((float)0.1)*current;
            }

            @Override
            public int read() {
                return (int) ((current +r.nextFloat()*delta)/ resolution);
            }
        }

        private class TemperatureSensorRegister extends RegisterSet.ReadOnlyRegister {

            private final float temperature;
            private final float delta;
            private final Random r = new Random();

            TemperatureSensorRegister(int offset, float temperature) {
                super(TempAdcs.REG_TEMP_ADCS + offset);                
                this.temperature = temperature;
                delta = ((float)0.1)*temperature +1;
            }

            @Override
            public int read() {
                return (int) ((temperature + r.nextFloat()*delta)/ TempAdcs.TEMP_SCALE);
            }

        }

    }

    private static class RTDSimulation {

        RTDSimulation(AddressSpace addressSpace) {
            //TODO: Better emulation
            addressSpace.add(new RegisterSet.WriteOnlyRegister(TempRtds.REG_RTD_COMMAND) {
                @Override
                public void write(int value) {
                    // Ignored for now
                }
            });
            addressSpace.add(new RegisterSet.ReadOnlyRegister(TempRtds.REG_RTD_REPLY) {
                @Override
                public int read() {
                    return 0; // FIXME: Implementation needed
                }
            });
            addressSpace.add(new RegisterSet.WriteOnlyRegister(TempRtds.REG_RTD_RESET) {
                @Override
                public void write(int value) {
                    // Ignored for now
                }
            });
        }
    }

    private static class AspicSimulation {

        private int adcMux;
        private int boardMux;
        @SuppressWarnings("MismatchedReadAndWriteOfArray")
        private final int[] aspicTemps = new int[Asic.NUM_SIDES * NUM_STRIPS];

        AspicSimulation(AddressSpace addressSpace) {
            addressSpace.add(new AspicStartRegister(SlowAdcs.REG_ASPIC_START));
            addressSpace.add(new AspicMuxConfigRegister(SlowAdcs.REG_MUX_CONFIG));
            for (int offset = 0; offset < Asic.NUM_SIDES * NUM_STRIPS; offset++) {
                addressSpace.add(new AspicTempRegister(offset));
            }
            addressSpace.add(new MuxTempRegister(SlowAdcs.REG_MUX_READ));
        }

        private class AspicStartRegister extends RegisterSet.WriteOnlyRegister {

            AspicStartRegister(int address) {
                super(address);
            }

            @Override
            public void write(int value) {

            }
        }

        private class AspicMuxConfigRegister extends RegisterSet.WriteOnlyRegister {

            AspicMuxConfigRegister(int address) {
                super(address);
            }

            @Override
            public void write(int value) {
                boardMux = value & 0xf;
                adcMux = (value >>> 4) & 0xf;
            }
        }

        private class AspicTempRegister extends RegisterSet.ReadOnlyRegister {

            private final int offset;

            AspicTempRegister(int offset) {
                super(SlowAdcs.REG_ASPIC_READ + offset);
                this.offset = offset;
            }

            @Override
            public int read() {
                return aspicTemps[offset];
            }
        }

        private class MuxTempRegister extends RegisterSet.ReadOnlyRegister {

            MuxTempRegister(int address) {
                super(address);
            }

            @Override
            public int read() {
                return adcMux << 24;
            }
        }

    }
}
