package org.lsst.ccs.drivers.reb;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *  Java interface to the image client routines
 *
 *  @author Owen Saxton
 */
public class ImageClient {

    /**
     *  Defines the data listener.
     */
    public interface Listener {

        /**
         *  Receives image data.
         *
         *  @param  image  The received image
         */
        public void processImage(Image image);

    }


    /**
     *  Thread that obtains the image data for the listener.
     */
    private class ReadThread extends Thread {

        private final Listener listener;   // Attached listener
        private final Image listenImage;   // Image supplied by listener
        private volatile int state = RUNNING;

        /**
         *  Constructor.
         *
         *  @param  lstnr  The listener
         *  @param  image  The image to use
         */
        private ReadThread(Listener lstnr, Image image)
        {
            listener = lstnr;
            listenImage = image;
        }

        /**
         *  Receives image data.
         */
        @Override
        public void run()
        {
            LOG.log(Level.FINE, "Image listener thread starting for {0}", rebid);
            try {
                while (state == RUNNING) {
                    Image image = impl.waitForImage(listenImage);
                    if (state != RUNNING) break;
                    if (!impl.getImage(image)) continue;
                    Listener l = listener;
                    if (state != RUNNING) break;
                    l.processImage(image);
                }
                if (state == CLOSING) {
                    impl.deleteImageClient();
                }
            } catch (Throwable t) {
                LOG.log(Level.SEVERE, "Error in image receive thread for "+rebid,t);
            } finally {
                LOG.log(Level.FINE, "Image listener thread stoping for {0}", rebid);
            }
        }


        /**
         *  Sets the thread state.
         *
         *  @param  newState  The state to set
         */
        public void setState(int newState)
        {
            state = newState;
        }

    }


    /**
     *  Data fields.
     */
    public static final int
        HDW_TYPE_DAQ0 = RegClient.HDW_TYPE_DAQ0,
        HDW_TYPE_DAQ1 = RegClient.HDW_TYPE_DAQ1,
        HDW_TYPE_DAQ2 = RegClient.HDW_TYPE_DAQ2,
        HDW_TYPE_PCI  = RegClient.HDW_TYPE_PCI,
        HDW_TYPE_PCI0 = RegClient.HDW_TYPE_PCI0,
        HDW_TYPE_PCI1 = RegClient.HDW_TYPE_PCI1;

    private static final int
        RUNNING = 0,
        CLOSING = 1,
        ENDING  = 2;

    private static final Logger LOG = Logger.getLogger(ImageClient.class.getName());
    private ReadThread reader;
    private Listener imgListener;
    private Image listenImage;
    private Impl impl;
    private final RegClient reg;
    private ClientFactory clientFactory = new ClientFactory();
    private String rebid = "Unknown";
    

    /**
     *  Constructor.
     */
    ImageClient() {
        this(null);
    }


    /**
     *  Constructor.
     *
     *  @param  reg  The associated register client
     */
    public ImageClient(RegClient reg)
    {
        this.reg = reg;
    }
    

    /**
     *  Constructor.
     *
     *  @param  reg    The associated register client
     *  @param  rebid  The REB ID (for logging)
     */
    public ImageClient(RegClient reg, String rebid) {
        this.reg = reg;
        this.rebid = rebid;
    }


    /**
     *  Finalizer.
     *
     *  @throws  Throwable
     */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        if (impl != null) {
            impl.deleteImageClient();
        }
    }


    /**
     *  Sets an alternative client factory.
     *
     *  The client factory is used to create the objects that implement
     *  access to either real (by default) or simulated hardware.
     *
     *  @param  clientFactory  The ClientFactory to use
     */
    public void setClientFactory(ClientFactory clientFactory) {
        this.clientFactory = clientFactory;
    }
    

    /**
     *  Sets the REB ID.
     *
     *  @param  rebid  The REB ID (for logging)
     */
    public void setRebId(String rebid)
    {
        this.rebid = rebid;
    }


    /**
     *  Opens a connection using the default interface.
     *
     *  @param  id  The ID of the REB to connect to
     *  @throws  REBException 
     */
    @Deprecated
    public void open(int id) throws REBException
    {
        open(id, null);
    }


    /**
     *  Opens a DAQ2 connection.
     *
     *  @param  id    The ID of the REB to connect to
     *  @param  part  The name of the partition to use.
     *  @throws  REBException 
     */
    public void open(int id, String part) throws REBException
    {
        open(HDW_TYPE_DAQ2, id, part);
    }


    /**
     *  Opens a connection to a REB.
     *
     *  @param  hdw  The hardware type
     *  @param  id   The ID of the REB to connect to
     *  @param  ifc  The name of the partition or interface to use.  If null or
     *               empty, the default is used if available.
     *  @throws  REBException 
     */
    public void open(int hdw, int id, String ifc) throws REBException
    {
        if (impl != null) {
            throw new REBException("Image connection already open");
        }
        Impl newImpl = clientFactory.createImageClient(hdw, reg);
        newImpl.newImageClient(id, ifc);
        impl = newImpl;
        if (imgListener != null) {
            reader = new ReadThread(imgListener, listenImage);
            reader.setDaemon(true);
            reader.start();
        }
    }


    /**
     *  Closes a connection.
     *
     *  @throws  REBException 
     */
    public void close() throws REBException
    {
        checkOpen();
        if (reader != null) {
            reader.setState(ENDING);
        }
        impl.deleteImageClient();
        if (reader != null) {
            try {
                reader.join(1000);
            }
            catch (InterruptedException e) {
            }
            reader = null;
        }
        impl = null;
    }


    /**
     *  Awaits an image.
     *
     *  Waits until a new image has been generated.
     *  @param  image  An Image object in which to save the reference and metadata for
     *                 the new image, or null if a new image object is to be created.
     *  @return  The Image object containing the new image reference and metadata.
     *  @throws  REBException 
     */
    public Image awaitImage(Image image) throws REBException
    {
        checkOpen();
        return impl.waitForImage(image);
    }


    /**
     *  Reads an image.
     *
     *  Gets the pixel data for an image.
     *
     *  @param  image  The Image object containing the valid metadata for an
     *                 image.  If the contained pixel data byte array is not
     *                 null and is large enough, the pixel data us stored
     *                 there.  Otherwise a new array is created.
     *  @return  True if the pixel data was successfully read, false otherwise.
     *  @throws  REBException 
     */
    public boolean readImage(Image image) throws REBException
    {
        checkOpen();
        return impl.getImage(image);
    }


    /**
     *  Deletes an image's metadata reference.
     *
     *  Deletes referenced C++ object.
     *  @param  image  The image whose metadata reference is to be deleted.
     */
    public void deleteImageMetadataRef(Image image)
    {
        impl.deleteImageMetadataRef(image);
    }


    /**
     *  Interrupts an image wait.
     *
     *  @throws  REBException 
     */
    public void interrupt() throws REBException
    {
        checkOpen();
        impl.interrupt();
    }


    /**
     *  Resets the front end system.
     *
     *  @throws  REBException 
     */
    public void resetFrontEnd() throws REBException
    {
        checkOpen();
        impl.reset();
    }


    /**
     *  Sets the data listener.
     *
     *  Sets the data listener object, whose class must implement the Listener
     *  interface, and therefore must contain the processImage method.  This
     *  method is called with a complete Image object as its argument whenever
     *  a new image becomes available.
     *
     *  @param  listener  The listener object to be set as the listener.  Any
     *                    existing listener is replaced.
     *  @param  image     An image object to be used to contain received
     *                    images, or null if a new object is to be created
     *                    each time.
     *  @throws  REBException
     */
    public void setListener(Listener listener, Image image)
    {
        clearListener();
        imgListener = listener;
        listenImage = image;
        if (impl != null) {
            reader = new ReadThread(listener, image);
            reader.setDaemon(true);
            reader.start();
        }
    }


    /**
     *  Clears the data listener.
     *
     *  Clears the data listener object if it exists.
     *
     *  @throws  REBException
     */
    public void clearListener()
    {
        if (reader != null) {
            reader.setState(ENDING);
            try {
                interrupt();
            }
            catch (REBException e) {
                // Just ignore it
            }
            reader = null;
        }
        imgListener = null;
        listenImage = null;
    }


    /**
     *  Checks that connection is open
     *
     *  @throws  REBException 
     */
    private void checkOpen() throws REBException
    {
        if (impl == null) {
            throw new REBException("Image connection not open");
        }
    }


    /**
     *  Inner interface to support choosing the hardware.
     */
    public interface Impl {

        /**
         *  Creates a new object.
         *
         *  Creates needed C++ objects.
         *
         *  @param  id   The ID of the REB to connect to
         *  @param  ifc  The name of the partition or interface to use.  If null or
         *               empty, the default is used if available.
         */
        public void newImageClient(int id, String ifc) throws REBException;


        /**
         *  Deletes an object.
         *
         *  Deletes referenced C++ objects.
         */
        public void deleteImageClient();


        /**
         *  Awaits an image.
         *
         *  Waits until a new image has been generated.
         *
         *  @param  image  An Image object in which to save the reference and
         *                 metadata for the new image, or null if a new image
         *                 object is to be created.
         *  @return  The Image object containing the new image reference and metadata.
         */
        public Image waitForImage(Image image);


        /**
         *  Reads an image.
         *
         *  Gets the pixel data for an image.
         *
         *  @param  image  The Image object containing the valid metadata for an
         *                 image.  If the contained pixel data byte array is not
         *                 null and is large enough, the pixel data us stored
         *                 there.  Otherwise a new array is created.
         *  @return  True if the pixel data was successfully read, false
         *           otherwise.
         */
        public boolean getImage(Image image);


        /**
         *  Interrupts an image wait.
         */
        public default void interrupt() {
        }


        /**
         *  Resets the front end system.
         */
        public default void reset() {
        }


        /**
         *  Deletes an image's metadata reference.
         *
         *  Deletes referenced C++ object.
         *
         *  @param  image  The image whose metadata reference is to be deleted.
         */
        public default void deleteImageMetadataRef(Image image) {
        }

    }

}
