package org.lsst.ccs.drivers.reb.sim;

import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.GlobalClient;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.RegClient;

/**
 * A factory class for creating simulated RegClient, ImageClient and
 * GlobalClient instances. This class also maintains a HandleAndIdManager class,
 * for tracking ids.
 *
 * @author tonyj
 */
public class ClientFactorySimulation extends ClientFactory {

    private final HandleAndIdManager manager;
    private final DAQSimulation daq;

    public ClientFactorySimulation() {
        manager = new HandleAndIdManager(this);
        daq = new DAQSimulation();
        // FIXME: Just here for backwards compatibility
        daq.setDataProvider(createDataProvider());
    }

    @Override
    public RegClient.Impl createRegClient(int hdw) throws REBException {
        switch (hdw) {
            case RegClient.HDW_TYPE_DAQ1:
            case RegClient.HDW_TYPE_DAQ2:
                return new RegClientDaq1Simulation(manager);
            default:
                throw new REBException("Invalid hardware type (" + hdw + ")");
        }
    }

    @Override
    public ImageClient.Impl createImageClient(int hdw, RegClient reg) throws REBException {
        switch (hdw) {
            case ImageClient.HDW_TYPE_DAQ1:
            case ImageClient.HDW_TYPE_DAQ2:
                return new ImageClientDaq1Simulation(manager, daq);
            default:
                throw new REBException("Invalid hardware type (" + hdw + ")");
        }
    }

    @Override
    public GlobalClient.Impl createGlobalClient(int hdw) throws REBException {
        switch (hdw) {
            case ImageClient.HDW_TYPE_DAQ1:
            case ImageClient.HDW_TYPE_DAQ2:
                return new GlobalClientSimulation(manager, daq);
            default:
                throw new REBException("Invalid hardware type (" + hdw + ")");
        }
    }

    AddressSpace createREBSimulation(int id) {
        return new REB3Simulation(id);
    }

    public void setDataProvider(DataProvider dp) {
       daq.setDataProvider(dp);
    }

    @Deprecated 
    protected DataProvider createDataProvider() {
        return new TrivialDataProvider();
    }

    public AddressSpace getAddressSpaceForId(int id) {
        return manager.getAddressSpaceForId(id);
    }
}
