package org.lsst.ccs.drivers.reb;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Java interfaces to REBs connected via the DAQ V2 system.
 *
 *  @author Owen Saxton
 */
public class Daq2Client {

    static void loadLibrary() {
        System.loadLibrary("RebDaq2");
    }


    /**
     *  Register interface.
     */
    public static class Registers implements RegClient.Impl {

        private static final int MAX_RETRY = 2;

        Registers() {
            loadLibrary();
        }

        @Override
        public native long newRegClient(int id, String part);

        @Override
        public native void deleteRegClient(long handle);

        @Override
        public int readReg(long handle, int address) throws REBException {
            int[] values = new int[1];
            readRegs(handle, address, values, 0, 1);
            return values[0];
        }

        @Override
        public void readRegs(long handle, int address, int[] values, int offset, int count) throws REBException {
            for (int j = 0; j <= MAX_RETRY; j++) {
                try {
                    readRegs1(handle, address, values, offset, count);
                }
                catch (DriverException e) {
                    if (j >= MAX_RETRY) {
                        throw e;
                    }
                }
            }
        }

        public native void readRegs1(long handle, int address, int[] values, int offset, int count) throws REBException;

        @Override
        public void writeReg(long handle, int address, int value) throws REBException {
            int[] values = {value};
            writeRegs(handle, address, values, 0, 1);
        }

        @Override
        public void writeRegs(long handle, int address, int[] values, int offset, int count) throws REBException {
            for (int j = 0; j <= MAX_RETRY; j++) {
                try {
                    writeRegs1(handle, address, values, offset, count);
                }
                catch (DriverException e) {
                    if (j >= MAX_RETRY) {
                        throw e;
                    }
                }
            }
        }

        public native void writeRegs1(long handle, int address, int[] values, int offset, int count) throws REBException;

        @Override
        public int updateReg(long handle, int address, int mask, int value) throws REBException {
            int[] values = {0};
            readRegs(handle, address, values, 0, 1);
            int oldVal = values[0];
            values[0] = (values[0] & ~mask) | (value & mask);
            writeRegs(handle, address, values, 0, 1);
            return oldVal;
        }

        @Override
        public native void reset(long handle, int type) throws REBException;

    }


    /**
     *  Image interface.
     */
    public static class Images implements ImageClient.Impl  {

        long handle;

        Images() {
            loadLibrary();
            initSys();
        }

        private native static void initSys();

        @Override
        public native void newImageClient(int id, String part);

        @Override
        public native void deleteImageClient();

        @Override
        public native Image waitForImage(Image image);
        
        @Override
        public native boolean getImage(Image image);

        @Override
        public native void interrupt();

    }


    /**
     *  Global interface.
     */
    public static class Global implements GlobalClient.Impl  {

        long handle;

        Global() {
            loadLibrary();
            initSys();
        }

        private native static void initSys();

        @Override
        public void newGlobalClient(int[] ids, String[] ifc) {
            handle = newGlobalClient1(ifc[0]);
        }

        @Override
        public void newGlobalClient(int[] ids, String part) {
            handle = newGlobalClient1(part);
        }

        private native long newGlobalClient1(String part);

        @Override
        public void deleteGlobalClient() {
            deleteGlobalClient1(handle);
            handle = 0;
        }

        private native void deleteGlobalClient1(long handle);

        @Override
        public void setRegisterList(int rebType, int[] registers) {
            setRegisterList1(handle, rebType, registers);
        }

        private native void setRegisterList1(long handle, int rebType, int[] registers);

        @Override
        public void triggerImage(String name) {
            triggerImage1(handle, name);
        }

        private native void triggerImage1(long handle, String name);

    }

}
