package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 ****************************************************
 *
 *  Java interface to the REB via the PGP PCI card.
 *
 *  @author Owen Saxton
 *
 ****************************************************
 */
public class PciClient {

   /*
    *  Class for holding register access information
    */
    static class RegisterAccess {
        long      handle;
        Registers regs;

        RegisterAccess(Registers regs, long handle) {
            this.handle = handle;
            this.regs = regs;
        }
    }

    private static final Map<String, RegisterAccess> idMap = new HashMap<>();
    private static boolean first = true;

    private static void loadLibrary() {
        if (first) {
            System.loadLibrary("RebPci");
            first = false;
        }
    }

   /**
    *  Register interface.
    */
    public static class Registers implements RegClient.Impl {

        int[] regList = new int[0];
        String imageName;

        Registers() {
            loadLibrary();
        }

        @Override
        public synchronized long newRegClient(int id, String ifc) {
            long handle = newRegClient1(id, ifc);
            idMap.put(genIdIfcKey(id, ifc), new RegisterAccess(this, handle));
            return handle;
        }

        public native long newRegClient1(int id, String ifc);

        @Override
        public synchronized void deleteRegClient(long handle) {
            for (String key : idMap.keySet()) {
                if (idMap.get(key).handle == handle) {
                    idMap.remove(key);
                    break;
                }
            }
            deleteRegClient1(handle);
        }

        public native void deleteRegClient1(long handle);

        @Override
        public int readReg(long handle, int address) throws REBException {
            int[] data = new int[1];
            readRegs(handle, address, data, 0, 1);
            return data[0];
        }

        @Override
        public synchronized native void readRegs(long handle, int address, int[] values, int offset, int count)
            throws REBException;

        @Override
        public void writeReg(long handle, int address, int value)
            throws REBException {
            int[] data = {value};
            writeRegs(handle, address, data, 0, 1);
        }

        @Override
        public synchronized native void writeRegs(long handle, int address, int[] values, int offset, int count)
            throws REBException;

        @Override
        public int updateReg(long handle, int address, int mask, int value)
            throws REBException {
            int oldValue = readReg(handle, address);
            writeReg(handle, address, (oldValue & ~mask) | (value & mask));
            return oldValue;
        }

    }

   /**
    *  Image interface.
    */
    public static class Images implements ImageClient.Impl  {

        long client;
        int version;
        String idIfc;
        RegisterAccess regAccess;


        Images(RegClient reg, int versn) {
            loadLibrary();
            initSys();
            version = versn;
        }

        private native static void initSys();


        @Override
        public void newImageClient(int id, String ifc) throws REBException {
            newImageClient1(id, ifc, version);
            idIfc = genIdIfcKey(id, ifc);
            if (idMap.get(idIfc) == null) {
                Registers regs = new Registers();
                regAccess = new RegisterAccess(regs, regs.newRegClient(id, ifc));
            }
        }

        public native void newImageClient1(int id, String ifc, int version);


        @Override
        public void deleteImageClient() {
            if (regAccess != null) {
                regAccess.regs.deleteRegClient(regAccess.handle);
                regAccess = null;
            }
            deleteImageClient1();
        }

        public native void deleteImageClient1();


        @Override
        public Image waitForImage(Image image) {
            Image newImage = waitForImage1(image);
            RegisterAccess ra = regAccess != null ? regAccess : idMap.get(idIfc);
            if (ra != null) {
                try {
                    newImage.name = ra.regs.imageName;
                    int[] values = new int[2];
                    ra.regs.readRegs(ra.handle, BaseSet.REG_SN_REB_VALUE, values, 0, values.length);
                    newImage.sciId = values[0] | ((long)(values[1] & 0xffff) << 32);
                    ra.regs.readRegs(ra.handle, BaseSet.REG_VERSION, values, 0, values.length);
                    newImage.sciVersion = values[0];
                    newImage.address = values[1];
                    ra.regs.readRegs(ra.handle, BaseSet.REG_TRIG_TIME + 2 * BaseSet.RSET_SEQUENCER,
                                     values, 0, values.length);
                    newImage.timestamp = values[0] + ((long)values[1] << 32);
                    newImage.registers = new int[ra.regs.regList.length];
                    for (int j = 0; j < ra.regs.regList.length; j++) {
                        newImage.registers[j] = ra.regs.readReg(ra.handle, ra.regs.regList[j]);
                    }
                    newImage.clientVersion = "";
                    newImage.serverVersion = "";
                }
                catch (REBException e) {
                }
            }
            return newImage;
        }

        private native Image waitForImage1(Image image);


        @Override
        public native boolean getImage(Image image);


        @Override
        public native void reset();


        @Override
        public native void deleteImageMetadataRef(Image image);
        
    }

   /**
    *  Global interface.
    */
    public static class Global implements GlobalClient.Impl  {

        private String[] idIfcs;

        @Override
        public void newGlobalClient(int[] ids, String[] ifcs) {
            idIfcs = new String[ids.length];
            for (int j = 0; j < ids.length; j++) {
                String ifc = (ifcs == null || j >= ifcs.length || ifcs[j] == null) ? "" : ifcs[j];
                idIfcs[j] = genIdIfcKey(ids[j], ifc);
            }
        }

        @Override
        public void deleteGlobalClient() {
        }

        @Override
        public void setRegisterList(int rebType, int[] registers) {
            int[] regs = registers.clone();
            for (String key : idIfcs) {
                RegisterAccess ra = idMap.get(key);
                if (ra == null) continue;
                ra.regs.regList = regs;
            }
        }

        @Override
        public void triggerImage(String name) throws REBException {
            for (String key : idIfcs) {
                RegisterAccess ra = idMap.get(key);
                if (ra == null) continue;
                ra.regs.imageName = name;
                int value = (ra.regs.readReg(ra.handle, BaseSet.REG_STATE)
                              & ((1 << BaseSet.RSET_STATUS) | (1 << BaseSet.RSET_TIME_BASE)))
                              | (1 << BaseSet.RSET_SEQUENCER);
                ra.regs.writeReg(ra.handle, BaseSet.REG_TRIGGER, value);
            }
        }
    }

   /**
    *  Generate ID/interface key
    */
    private static String genIdIfcKey(int id, String ifc) {
        return id + ":" + (ifc == null ? "" : ifc);
    }

}