package org.lsst.ccs.drivers.reb;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 *  Image metadata
 *
 *  @author Owen Saxton
 */
public class ImageMetadata implements Serializable {

    public static final int
        REB_TYPE_SCIENCE = BaseSet.REB_TYPE_SCIENCE,
        REB_TYPE_GUIDER = BaseSet.REB_TYPE_GUIDER,
        REB_TYPE_WAVEFRONT = BaseSet.REB_TYPE_WAVEFRONT;

    private static final Map<Integer, String> rebTypeMap = new HashMap<>();
    static {
        rebTypeMap.put(REB_TYPE_SCIENCE, "Science");
        rebTypeMap.put(REB_TYPE_GUIDER, "Guider");
        rebTypeMap.put(REB_TYPE_WAVEFRONT, "Wavefront");
    }

    protected String name;       // Image name, assigned by user
    protected long   timestamp;  // Timestamp (in nanoseconds)
    protected int    length;     // The number of pixels in the image data
    protected String partition;  // The partition name
    protected int    address;    // The address of the REB producing the image
    protected long   sciId;      // The REB serial number
    protected int    sensorType; // The REB type (SCIENCE, GUIDER, WAVEFRONT)
    protected int    numCcds;    // The number of CCDs generating image data
    protected int    stripes;    // The number of stripes in the image data
    protected int[]  registers = new int[0];  // Values of registers read
    protected int    sciVersion;    // The version of the REB firmware
    protected String clientVersion; // The version of the DAQ client software
    protected String serverVersion; // The version of the DAQ server software
    protected String platform;      // The platform (ATCA shelf) name
    protected int[]  members = new int[0]; // The addresses of the REBs in the event

    @Deprecated
    protected long tag;     // The image tag
    @Deprecated
    protected int format;   // The format of the data sent from the REB
    @Deprecated
    protected int schema;   // The schema of the firmware producing the image
    @Deprecated
    protected int version;  // The version of the firmware producing the image
    @Deprecated
    protected int cluster;  // The cluster number of the RCE
    @Deprecated
    protected int element;  // The element number of the RCE
    @Deprecated
    protected long event;   // The event ID
    

   /*
    *  Constructors.
    */
    public ImageMetadata() {
    }

    // DAQ V2
    public ImageMetadata(String name, long timestamp, int length, String partition, int address,
                         long sciId, int sensorType, int numCcds, int stripes, int[] registers,
                         int sciVersion, String clientVersion, String serverVersion,
                         String platform, int[] members) {
        this.name = name;
        this.timestamp = timestamp;
        this.length = length;
        this.partition = partition;
        this.address = address;
        this.sciId = sciId;
        this.sensorType = sensorType;
        this.numCcds = numCcds;
        this.stripes = stripes;
        this.registers = new int[registers.length];
        System.arraycopy(registers, 0, this.registers, 0, registers.length);
        this.sciVersion = sciVersion;
        this.clientVersion = clientVersion;
        this.serverVersion = serverVersion;
        this.platform = platform;
        this.members = new int[members.length];
        System.arraycopy(members, 0, this.members, 0, members.length);
    }

    // DAQ V1 and PGP card
    public ImageMetadata(String name, long timestamp, int length, int address,
                         long sciId, int sensorType, int stripes, int[] registers,
                         int sciVersion, String clientVersion, String serverVersion,
                         long event, int[] members) {
        this.name = name;
        this.timestamp = timestamp;
        this.length = length;
        this.address = address;
        this.sciId = sciId;
        this.sensorType = sensorType;
        this.stripes = stripes;
        this.registers = new int[registers.length];
        System.arraycopy(registers, 0, this.registers, 0, registers.length);
        this.sciVersion = sciVersion;
        this.clientVersion = clientVersion;
        this.serverVersion = serverVersion;
        this.platform = "";
        this.event = event;
        this.members = new int[members.length];
        System.arraycopy(members, 0, this.members, 0, members.length);
    }

    // DAQ V0
    @Deprecated
    public ImageMetadata(long tag, int length, int format, int schema,
                         int version, int address, int cluster, int element) {
        this.timestamp = tag;
        this.length = length;
        this.format = format;
        this.schema = schema;
        this.sciVersion = version;
        this.address = address;
        this.cluster = cluster;
        this.element = element;
    }


   /*
    *  Getters
    */
    public String getName() {
        return name;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public int getLength() {
        return length;
    }

    public String getPartition() {
        return partition;
    }

    public int getAddress() {
        return address;
    }

    public long getSciId() {
        return sciId;
    }

    public int getSensorType() {
        return sensorType;
    }

    public int getNumCcds() {
        return numCcds;
    }

    public int getStripes() {
        return stripes;
    }

    public int[] getRegisters() {
        return registers;
    }

    public int getSciVersion() {
        return sciVersion;
    }

    public String getClientVersion() {
        return clientVersion;
    }

    public String getServerVersion() {
        return serverVersion;
    }

    public String getPlatform() {
        return platform;
    }

    public int[] getMembers() {
        return members;
    }

    @Deprecated
    public int getVersion() {
        return sciVersion;
    }

    @Deprecated
    public long getTag() {
        return timestamp;
    }

    @Deprecated
    public int getFormat() {
        return 0;
    }

    @Deprecated
    public int getSchema() {
        return 0;
    }

    @Deprecated
    public int getCluster() {
        return 0;
    }

    @Deprecated
    public int getElement() {
        return 0;
    }

    @Deprecated
    public long getEvent() {
        return event;
    }

    @Override
    public String toString()
    {
        StringBuilder text = new StringBuilder();
        text.append(String.format("Name           : %s\n", name));
        text.append(String.format("Timestamp      : %016x\n", timestamp));
        text.append(String.format("Length         : %s\n", length));
        text.append(String.format("Partition      : %s\n", partition));
        text.append(String.format("Address        : %s\n", address));
        text.append(String.format("SCI ID         : %016x\n", sciId));
        text.append(String.format("Sensor type    : %s (%s)\n", sensorType, rebTypeMap.get(sensorType)));
        text.append(String.format("No. of CCDs    : %s\n", numCcds));
        text.append(String.format("Stripes        : %s\n", stripes));
        text.append(String.format("Registers      :"));
        for (int j = 0; j < registers.length; j++) {
            text.append(String.format(" %s", registers[j]));
        }
        text.append("\n");
        text.append(String.format("SCI version    : %08x\n", sciVersion));
        text.append(String.format("Client version : %s\n", clientVersion));
        text.append(String.format("Server version : %s\n", serverVersion));
        text.append(String.format("Platform       : %s\n", platform));
        text.append(String.format("Members        :"));
        for (int j = 0; j < members.length; j++) {
            text.append(String.format(" %s", members[j]));
        }
        return text.toString();
    }

    private static final long serialVersionUID = 9060283981810765079L;
}
