package org.lsst.ccs.drivers.reb;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 ***************************************************************
 *
 *  Program to test the Java global client interface routines
 *
 *  @author Owen Saxton
 *
 ***************************************************************
 */
public class TestGlobal {

   /*
    *  Private fields
    */
    private final GlobalClient gbl = new GlobalClient();


   /**
    *  Enumeration for hardware type.
    */
    public enum HdwType {

        DAQ0(BaseSet.HDW_TYPE_DAQ0),
        DAQ1(BaseSet.HDW_TYPE_DAQ1),
        PCI(BaseSet.HDW_TYPE_PCI);

        int value;

        HdwType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    *  Connects to a REB.
    *
    *  @param  type     The hardware type (DAQ0, DAQ1, PCI)
    *
    *  @param  id       The REB ID
    *
    *  @throws  REBException
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="type", description="Hardware type")
                        HdwType type,
                        @Argument(name="id", description="REB ID")
                        int id) throws REBException
    {
        connect(type, id, "");
    }


   /**
    *  Connects to a REB.
    *
    *  @param  type     The hardware type (DAQ0, DAQ1, PCI)
    *
    *  @param  id       The REB ID
    *
    *  @param  ifcName  The network interface name or PCI device name
    *
    *  @throws  REBException
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="type", description="Hardware type")
                        HdwType type,
                        @Argument(name="id", description="REB ID")
                        int id,
                        @Argument(name="ifcname",
                                  description="Network interface name")
                        String ifcName) throws REBException
    {
        gbl.open(type.getValue(), new int[]{id}, new String[]{ifcName});
    }


   /**
    *  Disconnects from a REB.
    *
    *  @throws  REBException
    */
    @Command(name="disconnect", description="Disconnect from a REB")
    public void disconnect() throws REBException
    {
        gbl.close();
    }


   /**
    *  Acquires an image.
    *
    *  @param  name  The image name
    */
    @Command(name="acquire", description="Acquire an image")
    public void trigger(@Argument(name="name", description="Image name")
                        String name) throws REBException
    {
        gbl.acquireImage(name);
    }

}
