package org.lsst.ccs.drivers.reb.sim;

import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Keeps track of which ID is registered to which (simulated) device, and
 * allocation of handles for active connections. Note that this implementation
 * delegates creation of new AddressSpaces to the client factory. Once created
 * devices are not shutdown, since new connections could be established to the
 * same device later.
 *
 * @author tonyj
 */
class HandleAndIdManager {

    private static final BitSet HANDLES_IN_USE = new BitSet();

    static {
        HANDLES_IN_USE.set(0); // Never return handle 0
    }
    private static final Map<Integer, AddressSpace> ID_TO_REGISTER_SET = new HashMap<>();
    private static final Map<Long, AddressSpace> HANDLE_TO_REGISTER_SET = Collections.synchronizedMap(new HashMap<>());
    private final ClientFactorySimulation factory;

    HandleAndIdManager(ClientFactorySimulation factory) {
        this.factory = factory;
    }
    
    //Temporary placeholders for quantities to be passed from the Global
    //Client to the Image client.
    //This code is aweful and should be replace with a better solution.
    private int rebType;
    private static int[] registers;
           
    
    public void setRegisterList(int rebType, int[] registers) {
        this.rebType = rebType;
        this.registers = registers;
    }
    
    public int getRebType() {
        return rebType;
    }
    
    public int[] getRegisters() {
        return registers;
    }

    /**
     * Find the address space for the given id, If one already exists then return it,
     * otherwise delegate to the factory to create it.
     * @param id The id to find.
     * @return The found or created AddressSpace.
     */
    AddressSpace getAddressSpaceForId(int id) {

        synchronized (ID_TO_REGISTER_SET) {
            AddressSpace rs = ID_TO_REGISTER_SET.get(id);
            if (rs == null) {
                rs = factory.createREBSimulation(id);
                ID_TO_REGISTER_SET.put(id, rs);
            }
            return rs;
        }
    }

    AddressSpace getAddressSpaceForHandle(long handle) {
        synchronized (HANDLE_TO_REGISTER_SET) {
            return HANDLE_TO_REGISTER_SET.get(handle);
        }
    }
    
    long allocateHandle(AddressSpace rs) {
        synchronized (HANDLES_IN_USE) {
            long handle = HANDLES_IN_USE.nextClearBit(0);
            HANDLES_IN_USE.set((int) handle);
            HANDLE_TO_REGISTER_SET.put(handle, rs);
            return handle;
        }
    }

    void freeHandle(long handle) {
        synchronized (HANDLES_IN_USE) {
            HANDLES_IN_USE.clear((int) handle);
            HANDLE_TO_REGISTER_SET.remove(handle);
        }
    }


}
