package org.lsst.ccs.drivers.reb;

/**
 ***********************************
 *
 *  REB sequencer access routines.
 *
 *  @author Owen Saxton
 *
 ***********************************
 */
public class Sequencer {

   /**
    *  Public constants.
    */
    public static final int
        REG_SEQ_LINES        = 0x100000,
        REG_SEQ_TIMES        = 0x200000,
        REG_SEQ_STACK        = 0x300000,
        REG_SEQ_PROGRAM      = 0x300000,
        REG_SEQ_STEP         = 0x310000,
        REG_SEQ_STOP         = 0x320000,
        REG_SEQ_SCAN_CNTRL   = 0x330000,
        REG_SEQ_SCAN_RESET_2 = 0x340000,
        REG_SEQ_SCAN_RESET   = 0x330001,
        REG_SEQ_START_ADDR   = 0x340000,
        REG_SEQ_EXEC_REFS    = 0x350000,
        REG_SEQ_EXEC_REPS    = 0x360000,
        REG_SEQ_JUMP_REFS    = 0x370000,
        REG_SEQ_JUMP_REPS    = 0x380000,
        REG_SEQ_ERROR_VALUE  = 0x390000,
        REG_SEQ_ERROR_RESET  = 0x390001,
        REG_SEQ_SSIZE        = 0x400000,
        REG_SEQ_NSLICE       = 0x400005,
        REG_SEQ_SOURCE       = 0x400006,
        REG_SEQ_STRIPE       = 0x400007,
        REG_SEQ_BEB          = 0x400007,
        REG_SEQ_STEP_0       = 0x500000,
        REG_SEQ_STOP_0       = 0x600000,

        SEQ_MAX_FUNC_0       = 4,
        SEQ_MAX_FUNC         = 16,
        SEQ_MAX_SLICE        = 16,
        SEQ_MAX_PROGRAM      = 1024,
        SEQ_MAX_STACK        = 4096,
        SEQ_MAX_PARAMETER    = 16,

        SEQ_SRC_FPGA_PTN     = 0,
        SEQ_SRC_BEB_PTN      = 0x1beb,
        SEQ_SRC_BEB_ADC      = 0x0beb,
        SEQ_SRC_ADC          = 0,
        SEQ_SRC_PATTERN      = 1,

        SEQ_OPC_EXECUTE      = 1,
        SEQ_OPC_EXECUTE_FP   = 2,
        SEQ_OPC_EXECUTE_RP   = 3,
        SEQ_OPC_EXECUTE_FRP  = 4,
        SEQ_OPC_JUMP_1       = 2,
        SEQ_OPC_JUMP         = 5,
        SEQ_OPC_JUMP_AP      = 6,
        SEQ_OPC_JUMP_RP      = 7,
        SEQ_OPC_JUMP_ARP     = 8,
        SEQ_OPC_END_SUBR_1   = 3,
        SEQ_OPC_END_SUBR     = 14,
        SEQ_OPC_END_PROG_1   = 4,
        SEQ_OPC_END_PROG     = 15,

        SEQ_ARG_IND_FUNC     = 0x01,
        SEQ_ARG_IND_JUMP     = 0x01,
        SEQ_ARG_IND_COUNT    = 0x02,
        SEQ_ARG_IND_MASK     = 0x03,

        SEQ_ERR_M_CODE       = 0xf0000000,
        SEQ_ERR_M_ADDR       = 0x0fffffff,

        SEQ_PRG_V_OPCODE     = 28,
        SEQ_PRG_V_FUNC       = 24,
        SEQ_PRG_M_FUNC       = SEQ_MAX_FUNC - 1,
        SEQ_PRG_M_EXCCNT     = 0xffffff,
        SEQ_PRG_M_LOOP       = 0x800000,
        SEQ_PRG_V_SUBADD_1   = 18,
        SEQ_PRG_V_SUBADD     = 16,
        SEQ_PRG_M_SUBADD     = SEQ_MAX_PROGRAM - 1,
        SEQ_PRG_M_SUBCNT_1   = 0x1ffff,
        SEQ_PRG_M_SUBCNT     = 0xffff,
        SEQ_PRG_M_PARM       = SEQ_MAX_PARAMETER - 1,
        SEQ_STK_V_FUNC       = 28,
        SEQ_STK_M_FUNC       = SEQ_MAX_FUNC_0 - 1,
        SEQ_STK_V_LOOP       = 27,
        SEQ_STK_M_COUNT      = 0x7ffffff,
        MAX_SUBR_LEVEL       = 16;

   /**
    *  Fields.
    */
    protected int[][] lines = new int[SEQ_MAX_FUNC][SEQ_MAX_SLICE];
    protected int[][] times = new int[SEQ_MAX_FUNC][SEQ_MAX_SLICE];
    protected int[] prog  = new int[SEQ_MAX_PROGRAM];
    protected int[] stack = new int[SEQ_MAX_STACK];
    protected int[] execRefs = new int[SEQ_MAX_PARAMETER];
    protected int[] execReps = new int[SEQ_MAX_PARAMETER];
    protected int[] jumpRefs = new int[SEQ_MAX_PARAMETER];
    protected int[] jumpReps = new int[SEQ_MAX_PARAMETER];
    protected int stackSize;
    protected int startAddr;
    BaseSet bss;


   /**
    *  Constructor.
    *
    *  @param  bss  The associated base set object
    */
    public Sequencer(BaseSet bss)
    {
        this.bss = bss;
    }


   /**
    *  Enables the sequencer.
    *
    *  @throws  REBException 
    */
    public void enable() throws REBException
    {
        bss.enable(BaseSet.RSET_SEQUENCER);
    }


   /**
    *  Disables the sequencer.
    *
    *  @throws  REBException 
    */
    public void disable() throws REBException
    {
        bss.disable(BaseSet.RSET_SEQUENCER);
    }


   /**
    *  Checks whether the sequencer is enabled.
    *
    *  @return  Whether the sequencer is enabled
    *
    *  @throws  REBException 
    */
    public boolean isEnabled() throws REBException
    {
        return bss.isEnabled(BaseSet.RSET_SEQUENCER);
    }


   /**
    *  Waits for the sequencer to be done (become idle).
    *
    *  @param  timeout  The maximum time to wait (ms)
    *
    *  @throws  REBException 
    */
    public void waitDone(int timeout) throws REBException
    {
        bss.waitDone(BaseSet.RSET_SEQUENCER, timeout);
    }


   /**
    *  Gets the sequencer trigger time.
    *
    *  @return  The Unix millisecond time of the sequencer trigger
    *
    *  @throws  REBException 
    */
    public long getTriggerTime() throws REBException
    {
        return bss.getTriggerTime(BaseSet.RSET_SEQUENCER);
    }


   /**
    *  Writes an output lines value.
    *
    *  @param  func   The function number
    *
    *  @param  slice  The time slice number
    *
    *  @param  value  The output lines value
    *
    *  @throws  REBException 
    */
    public void writeLines(int func, int slice, int value)
        throws REBException
    {
        checkFunc(func);
        checkSlice(slice);
        bss.write(REG_SEQ_LINES | (func<< 4) | slice, value);
        lines[func][slice] = value;
    }


   /**
    *  Writes a set of output lines values.
    *
    *  @param  func    The function number
    *
    *  @param  values  The array of output lines values
    *
    *  @throws  REBException 
    */
    public void writeLines(int func, int[] values) throws REBException
    {
        writeLines(func, values, 0, values.length);
    }


   /**
    *  Writes a set of output lines values.
    *
    *  @param  func    The function number
    *
    *  @param  values  The array of output lines values
    *
    *  @param  offset  The offset in values of the first value to write
    *
    *  @param  count   The number of values to write
    *
    *  @throws  REBException 
    */
    public void writeLines(int func, int[] values, int offset, int count)
        throws REBException
    {
        checkFunc(func);
        checkSlice(count - 1);
        bss.write(REG_SEQ_LINES | (func << 4), values, offset, count);
        System.arraycopy(values, offset, lines[func], 0, count);
    }


   /**
    *  Writes a time slice value.
    *
    *  @param  func   The function number
    *
    *  @param  slice  The time slice number
    *
    *  @param  value  The time slice value
    *
    *  @throws  REBException 
    */
    public void writeTimes(int func, int slice, int value)
        throws REBException
    {
        checkFunc(func);
        checkSlice(slice);
        bss.write(REG_SEQ_TIMES | (func << 4) | slice, value);
        times[func][slice] = value;
    }


   /**
    *  Writes a set of time slice values.
    *
    *  @param  func    The function number
    *
    *  @param  values  The array of time slice values
    *
    *  @throws  REBException 
    */
    public void writeTimes(int func, int[] values) throws REBException
    {
        writeTimes(func, values, 0, values.length);
    }


   /**
    *  Writes a set of time slice values.
    *
    *  @param  func    The function number
    *
    *  @param  values  The array of time slice values
    *
    *  @param  offset  The offset in values of the first value to write
    *
    *  @param  count   The number of values to write
    *
    *  @throws  REBException 
    */
    public void writeTimes(int func, int[] values, int offset, int count)
        throws REBException
    {
        checkFunc(func);
        checkSlice(count - 1);
        bss.write(REG_SEQ_TIMES | (func << 4), values, offset, count);
        System.arraycopy(values, offset, times[func], 0, count);
    }


   /**
    *  Writes a stack value.
    *
    *  @param  addr   The stack address
    *
    *  @param  value  The value to write
    *
    *  @throws  REBException 
    */
    public void writeStack(int addr, int value) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        checkStackAddr(addr);
        bss.write(REG_SEQ_STACK + addr, value);
        stack[addr] = value;
    }


   /**
    *  Writes a set of stack values.
    *
    *  @param  addr    The first stack address
    *
    *  @param  values  The array of values to write
    *
    *  @throws  REBException 
    */
    public void writeStack(int addr, int[] values) throws REBException
    {
        writeStack(addr, values, 0, values.length);
    }


   /**
    *  Writes a set of stack values.
    *
    *  @param  addr    The first stack address
    *
    *  @param  values  The array of stack values to write
    *
    *  @param  offset  The offset in values of the first value to write
    *
    *  @param  count   The number of values to write
    *
    *  @throws  REBException 
    */
    public void writeStack(int addr, int[] values, int offset, int count)
        throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        checkStackAddr(addr, count);
        bss.write(REG_SEQ_STACK + addr, values, offset, count);
        System.arraycopy(values, offset, stack, addr, count);
    }


   /**
    *  Writes a complete set of stack values, setting the stack size.
    *
    *  @param  values  The array of stack values to write
    *
    *  @throws  REBException 
    */
    public void writeStack(int[] values) throws REBException
    {
        writeStack(values, 0, values.length);
    }


   /**
    *  Writes a complete set of stack values, setting the stack size.
    *
    *  @param  values  The array of stack values to write
    *
    *  @param  offset  The offset in values of the first item to write
    *
    *  @param  count   The number of values to write
    *
    *  @throws  REBException 
    */
    public void writeStack(int[] values, int offset, int count)
        throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        checkStackAddr(0, count);
        bss.write(REG_SEQ_STACK, values, offset, count);
        bss.write(REG_SEQ_SSIZE, count);
        System.arraycopy(values, offset, stack, 0, count);
        stackSize = count;
    }


   /**
    *  Writes the stack size.
    *
    *  @param  value  The value to write to the stack size register
    *
    *  @throws  REBException 
    */
    public void writeStackSize(int value) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        bss.write(REG_SEQ_SSIZE, value);
        stackSize = value;
    }


   /**
    *  Writes a program word.
    *
    *  @param  addr   The program address
    *
    *  @param  value  The value to write
    *
    *  @throws  REBException 
    */
    public void writeProgram(int addr, int value) throws REBException
    {
        bss.checkNotVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        checkProgAddr(addr);
        bss.write(REG_SEQ_PROGRAM + addr, value);
        prog[addr] = value;
    }


   /**
    *  Writes a set of program words.
    *
    *  @param  addr    The first program address
    *
    *  @param  values  The array of values to write
    *
    *  @throws  REBException 
    */
    public void writeProgram(int addr, int[] values) throws REBException
    {
        writeProgram(addr, values, 0, values.length);
    }


   /**
    *  Writes a set of program words.
    *
    *  @param  addr    The first program address
    *
    *  @param  values  The array of program words to write
    *
    *  @param  offset  The offset in values of the first value to write
    *
    *  @param  count   The number of values to write
    *
    *  @throws  REBException 
    */
    public void writeProgram(int addr, int[] values, int offset, int count)
        throws REBException
    {
        bss.checkNotVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        checkProgAddr(addr, count);
        bss.write(REG_SEQ_PROGRAM + addr, values, offset, count);
        System.arraycopy(values, offset, prog, addr, count);
    }


   /**
    *  Writes an execute program word.
    *
    *  @param  addr   The program address
    *
    *  @param  func   The ID of the function to execute
    *
    *  @param  count  The number of times to execute the function
    *
    *  @throws  REBException 
    */
    public void writeProgExec(int addr, int func, int count)
        throws REBException
    {
        if (bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3)) {
            writeProgExec(addr, 0, func, count);
        }
        else {
            checkFunc(func);
            writeProgram(addr, count | (func << SEQ_PRG_V_FUNC) 
                                 | (SEQ_OPC_EXECUTE << SEQ_PRG_V_OPCODE));
        }
    }


   /**
    *  Writes an execute program word.
    *
    *  @param  addr     The program address
    *
    *  @param  argType  The mask of argument types  
    *
    *  @param  func     The ID of the function to execute
    *
    *  @param  count    The number of times to execute the function, or -1
    *                   for an infinite loop
    *
    *  @throws  REBException 
    */
    public void writeProgExec(int addr, int argType, int func, int count)
        throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        argType &= SEQ_ARG_IND_MASK;
        if ((argType & SEQ_ARG_IND_FUNC) == 0) {
            checkFunc(func);
        }
        else {
            checkParamAddr(func);
        }
        if ((argType & SEQ_ARG_IND_COUNT) == 0) {
            checkExecCount(count);
        }
        else {
            checkParamAddr(count);
        }
        writeProgram(addr, ((SEQ_OPC_EXECUTE + argType) << SEQ_PRG_V_OPCODE)
                             | (func << SEQ_PRG_V_FUNC) | count);
    }


   /**
    *  Writes a jump program word.
    *
    *  @param  addr     The program address
    *
    *  @param  subaddr  The address of the subroutine to jump to
    *
    *  @param  count    The number of times to run the subroutine
    *
    *  @throws  REBException 
    */
    public void writeProgJump(int addr, int subaddr, int count)
        throws REBException
    {
        if (bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3)) {
            writeProgJump(addr, 0, subaddr, count);
        }
        else {
            checkProgAddr(subaddr);
            checkJumpCount(count);
            writeProgram(addr, (SEQ_OPC_JUMP_1 << SEQ_PRG_V_OPCODE)
                                 | (subaddr << SEQ_PRG_V_SUBADD_1) | count);
        }
    }


   /**
    *  Writes a jump program word.
    *
    *  @param  addr     The program address
    *
    *  @param  argType  The mask of argument types
    *
    *  @param  subaddr  The address of the subroutine to jump to
    *
    *  @param  count    The number of times to run the subroutine
    *
    *  @throws  REBException 
    */
    public void writeProgJump(int addr, int argType, int subaddr, int count)
        throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        argType &= SEQ_ARG_IND_MASK;
        if ((argType & SEQ_ARG_IND_FUNC) == 0) {
            checkProgAddr(subaddr);
        }
        else {
            checkParamAddr(subaddr);
        }
        if ((argType & SEQ_ARG_IND_COUNT) == 0) {
            checkJumpCount(count);
        }
        else {
            checkParamAddr(count);
        }
        writeProgram(addr, ((SEQ_OPC_JUMP + argType) << SEQ_PRG_V_OPCODE)
                             | (subaddr << SEQ_PRG_V_SUBADD) | count);
    }


   /**
    *  Writes an end subroutine word.
    *
    *  @param  addr  The program address
    *
    *  @throws  REBException 
    */
    public void writeProgEndSubr(int addr) throws REBException
    {
        int opcode = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3)
                       ? SEQ_OPC_END_SUBR : SEQ_OPC_END_SUBR_1;
        writeProgram(addr, opcode << SEQ_PRG_V_OPCODE);
    }


   /**
    *  Writes an end program word.
    *
    *  @param  addr  The program address
    *
    *  @throws  REBException 
    */
    public void writeProgEnd(int addr) throws REBException
    {
        int opcode = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3)
                       ? SEQ_OPC_END_PROG : SEQ_OPC_END_PROG_1;
        writeProgram(addr, opcode << SEQ_PRG_V_OPCODE);
    }


   /**
    *  Writes an execute function code parameter.
    *
    *  @param  addr  The parameter address
    *
    *  @param  func  The ID of the function to execute
    *
    *  @throws  REBException 
    */
    public void writeExecFunc(int addr, int func) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        checkFunc(func);
        bss.write(REG_SEQ_EXEC_REFS + addr, func);
        execRefs[addr] = func;
    }


   /**
    *  Reads an execute function code parameter.
    *
    *  @param  addr  The parameter address
    *
    *  @throws  REBException 
    *
    *  @return  The function pointer value
    */
    public int readExecFunc(int addr) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        return bss.read(REG_SEQ_EXEC_REFS + addr);
    }


   /**
    *  Writes an execute repeat count parameter.
    *
    *  @param  addr   The parameter address
    *
    *  @param  count  The number of times to execute the function, or -1
    *                 for an infinite loop
    *
    *  @throws  REBException 
    */
    public void writeExecCount(int addr, int count) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        checkExecCount(count);
        bss.write(REG_SEQ_EXEC_REPS + addr, count);
        execReps[addr] = count;
    }


   /**
    *  Reads an execute repeat count parameter.
    *
    *  @param  addr  The parameter address
    *
    *  @throws  REBException 
    *
    *  @return  The function repeat count
    */
    public int readExecCount(int addr) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        return bss.read(REG_SEQ_EXEC_REPS + addr);
    }


   /**
    *  Writes a jump subroutine address parameter.
    *
    *  @param  addr  The parameter address
    *
    *  @param  subr  The address of the subroutine to execute
    *
    *  @throws  REBException 
    */
    public void writeJumpSubr(int addr, int subr) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        checkProgAddr(subr);
        bss.write(REG_SEQ_JUMP_REFS + addr, subr);
        jumpRefs[addr] = subr;
    }


   /**
    *  Reads a jump subroutine address parameter
    *
    *  @param  addr  The parameter address
    *
    *  @throws  REBException 
    *
    *  @return  The subroutine pointer value
    */
    public int readJumpSubr(int addr) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        return bss.read(REG_SEQ_JUMP_REFS + addr);
    }


   /**
    *  Writes a jump repeat count parameter.
    *
    *  @param  addr   The parameter address
    *
    *  @param  count  The number of times to execute the subroutine
    *
    *  @throws  REBException 
    */
    public void writeJumpCount(int addr, int count) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        checkJumpCount(count);
        bss.write(REG_SEQ_JUMP_REPS + addr, count);
        jumpReps[addr] = count;
    }


   /**
    *  Reads a jump repeat count parameter
    *
    *  @param  addr  The parameter address
    *
    *  @throws  REBException 
    *
    *  @return  The subroutine repeat count
    */
    public int readJumpCount(int addr) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkParamAddr(addr);
        return bss.read(REG_SEQ_JUMP_REPS + addr);
    }


   /**
    *  Writes the start address register.
    *
    *  @param  value  The value to write
    *
    *  @throws  REBException 
    */
    public void writeStartAddr(int value) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        checkProgAddr(value);
        bss.write(REG_SEQ_START_ADDR, value);
        startAddr = 0;
    }


   /**
    *  Reads the start address register.
    *
    *  @return  The start address value read
    *
    *  @throws  REBException 
    */
    public int readStartAddr() throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3);
        return bss.read(REG_SEQ_START_ADDR);
    }


   /**
    *  Writes the slice count.
    *
    *  @param  value  The slice count value to write
    *
    *  @throws  REBException 
    */
    public void writeSliceCount(int value) throws REBException
    {
        bss.write(REG_SEQ_NSLICE, value);
    }


   /**
    *  Reads the slice count.
    *
    *  @return  The slice count value read
    *
    *  @throws  REBException 
    */
    public int readSliceCount() throws REBException
    {
        return bss.read(REG_SEQ_NSLICE);
    }


   /**
    *  Writes the data source register.
    *
    *  @param  value  The value to write
    *
    *  @throws  REBException 
    */
    public void writeDataSource(int value) throws REBException
    {
        bss.write(REG_SEQ_SOURCE, value);
    }


   /**
    *  Reads the data source register.
    *
    *  @return  The slice count value read
    *
    *  @throws  REBException 
    */
    public int readDataSource() throws REBException
    {
        return bss.read(REG_SEQ_SOURCE);
    }


   /**
    *  Writes the stripe selection register.
    *
    *  @param  value  The value to write
    *
    *  @throws  REBException 
    */
    public void writeStripeSelect(int value) throws REBException
    {
        bss.checkNotVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        bss.write(REG_SEQ_STRIPE, value);
    }


   /**
    *  Reads the stripe selection register.
    *
    *  @return  The slice count value read
    *
    *  @throws  REBException 
    */
    public int readStripeSelect() throws REBException
    {
        bss.checkNotVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        return bss.read(REG_SEQ_STRIPE);
    }


   /**
    *  Writes the BEB selector register.
    *
    *  @param  value  The value to write
    *
    *  @throws  REBException 
    */
    public void writeBebSelect(int value) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0);
        bss.write(REG_SEQ_BEB, value);
    }


   /**
    *  Sends the step command.
    *
    *  @throws  REBException 
    */
    public void sendStep() throws REBException
    {
        int stepAddr = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0)
                         ? REG_SEQ_STEP_0 : REG_SEQ_STEP;
        bss.write(stepAddr, 0);
    }


   /**
    *  Sends the stop command.
    *
    *  @throws  REBException 
    */
    public void sendStop() throws REBException
    {
        int stopAddr = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0)
                         ? REG_SEQ_STOP_0 : REG_SEQ_STOP;
        bss.write(stopAddr, 0);
    }


   /**
    *  Enables or disables scan mode.
    *
    *  @param  enable  True to enable scan mode; false to disable it
    *
    *  @throws  REBException 
    */
    public void enableScan(boolean enable) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_2, BaseSet.VERSION_3);
        bss.write(REG_SEQ_SCAN_CNTRL, enable ? 1 : 0);
    }


   /**
    *  Gets whether scan mode is enabled.
    *
    *  @return  Whether or not scan mode is enabled
    *
    *  @throws  REBException 
    */
    public boolean isScanEnabled() throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_2, BaseSet.VERSION_3);
        return (bss.read(REG_SEQ_SCAN_CNTRL) & 1) != 0;
    }


   /**
    *  Resets scan mode.
    *
    *  @throws  REBException 
    */
    public void resetScan() throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_2, BaseSet.VERSION_3);
        int addr = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_2)
                     ? REG_SEQ_SCAN_RESET_2 : REG_SEQ_SCAN_RESET;
        bss.write(addr, 0);
    }


   /**
    *  Gets sequencer error address.
    *
    *  @return  The program address of the sequencer error, or -1 if none
    *
    *  @throws  REBException 
    */
    public int getErrorAddr() throws REBException
    {
        if (bss.getVersion(BaseSet.OPTN_SEQUENCER, true) < BaseSet.VERSION_3) return -1;
        int value = bss.read(REG_SEQ_ERROR_VALUE);
        return (value & SEQ_ERR_M_CODE) != 0 ? value & SEQ_ERR_M_ADDR : -1;
    }


   /**
    *  Resets sequencer error.
    *
    *  @throws  REBException 
    */
    public void resetError() throws REBException
    {
        if (bss.getVersion(BaseSet.OPTN_SEQUENCER, true) < BaseSet.VERSION_3) return;
        bss.write(REG_SEQ_ERROR_RESET, 0);
    }


   /**
    *  Clears the sequencer program cached copy.
    *
    *  @throws  REBException 
    */
    public void clearCache() throws REBException
    {
        for (int j = 0; j < lines.length; j++) {
            for (int k = 0; k < lines[j].length; k++) {
                lines[j][k] = 0;
                times[j][k] = 0;
            }
        }
        if (bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0)) {
            for (int j = 0; j < stack.length; j++) {
                stack[j] = 0;
            }
            stackSize = 0;
        }
        else {
            for (int j = 0; j < prog.length; j++) {
                prog[j] = 0;
            }
        }
        if (bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3)) {
            for (int j = 0; j < execRefs.length; j++) {
                execRefs[j] = 0;
                execReps[j] = 0;
                jumpRefs[j] = 0;
                jumpReps[j] = 0;
            }
            startAddr = 0;
        }
    }


   /**
    *  Gets the slice count of a sequencer program cached copy.
    *
    *  @return  The number of slices generated by the whole program
    *
    *  @throws  REBException 
    */
    public int getCacheSliceCount() throws REBException
    {
        int maxFunc = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0)
                        ? SEQ_MAX_FUNC_0 : SEQ_MAX_FUNC;
        int[] funcSlice = new int[maxFunc];
        int[] funcSend = new int[maxFunc];
        //funcSlice[0] = 0;
        for (int j = 1; j < maxFunc; j++) {
            int nSlice = 0, send = 0;
            boolean prevAdc = false, prevSoi = false, prevEoi = false;
            for (int k = 0; k < SEQ_MAX_SLICE && times[j][k] != 0; k++) {
                int line = lines[j][k];
                if ((line & 0x2000) != 0) {
                    send = 1;
                }
                if ((line & 0x4000) != 0) {
                    send = -1;
                }
                boolean adc = (line & 0x1000) != 0;
                if (adc & !prevAdc) {
                    nSlice++;
                }
                prevAdc = adc;
            }
            funcSend[j] = send;
            funcSlice[j] = nSlice;
        }
        if (bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0)) {
            int nSlice = 0;
            for (int j = 0; j < stackSize; j++) {
                int func = (stack[j] >> SEQ_STK_V_FUNC) & SEQ_STK_M_FUNC;
                int count = stack[j] & SEQ_STK_M_COUNT;
                if (funcSlice[func] != 0
                      && (stack[j] & (1 << SEQ_STK_V_LOOP)) != 0) {
                    throw new REBException("Data-producing infinite loop");
                }
                nSlice += count * funcSlice[func];
            }
            return nSlice;
        }
        else {
            return getProgSlices(startAddr, 0, funcSlice, funcSend);
        }
    }


   /**
    *  Gets the number of slices generated by a sequencer program routine.
    *
    *  @param  addr       The program address
    *
    *  @param  level      The subroutine nesting level
    *
    *  @param  funcSlice  The array of slice counts for each defined function
    *
    *  @return  The number of slices generated by the routine
    *
    *  @throws  REBException 
    */
    private int getProgSlices(int addr, int level, int[] funcSlice,
                              int[] funcSend) throws REBException
    {
        int nSlice = 0;
        int version = bss.getVersion(BaseSet.OPTN_SEQUENCER);
        int opcExecFP = version == BaseSet.VERSION_3 ? SEQ_OPC_EXECUTE_FP : -1;
        int opcExecRP = version == BaseSet.VERSION_3 ? SEQ_OPC_EXECUTE_RP : -1;
        int opcExecFRP = version == BaseSet.VERSION_3 ? SEQ_OPC_EXECUTE_FRP : -1;
        int opcJump = version == BaseSet.VERSION_3 ? SEQ_OPC_JUMP : SEQ_OPC_JUMP_1;
        int opcJumpAP = version == BaseSet.VERSION_3 ? SEQ_OPC_JUMP_AP : -1;
        int opcJumpRP = version == BaseSet.VERSION_3 ? SEQ_OPC_JUMP_RP : -1;
        int opcJumpARP = version == BaseSet.VERSION_3 ? SEQ_OPC_JUMP_ARP : -1;
        int opcEndS = version == BaseSet.VERSION_3 ? SEQ_OPC_END_SUBR : SEQ_OPC_END_SUBR_1;
        int opcEndP = version == BaseSet.VERSION_3 ? SEQ_OPC_END_PROG : SEQ_OPC_END_PROG_1;
        int subaddV = version == BaseSet.VERSION_3 ? SEQ_PRG_V_SUBADD : SEQ_PRG_V_SUBADD_1;

        for (; addr < SEQ_MAX_PROGRAM; addr++) {
            int cmnd = prog[addr], opcode = cmnd >>> SEQ_PRG_V_OPCODE;
            if (opcode == SEQ_OPC_EXECUTE || opcode == opcExecFP
                  || opcode == opcExecRP || opcode == opcExecFRP) {
                int func, count;
                if (opcode == SEQ_OPC_EXECUTE || opcode == opcExecRP) {
                    func = (cmnd >> SEQ_PRG_V_FUNC) & SEQ_PRG_M_FUNC;
                }
                else {
                    func = execRefs[(cmnd >> SEQ_PRG_V_FUNC) & SEQ_PRG_M_PARM];
                }
                if (opcode == SEQ_OPC_EXECUTE || opcode == opcExecFP) {
                    count = cmnd & SEQ_PRG_M_EXCCNT;
                }
                else {
                    count = execReps[cmnd & SEQ_PRG_M_PARM];
                }
                funcSend[0] = funcSend[func] == 0 ? funcSend[0] : funcSend[func];
                int fSlice = funcSend[0] > 0 ? funcSlice[func] : 0;
                if (fSlice != 0 && (count & SEQ_PRG_M_LOOP) != 0) {
                    throw new REBException("Data-producing infinite loop");
                }
                nSlice += count * fSlice;
            }
            else if (opcode == opcJump || opcode == opcJumpAP
                       || opcode == opcJumpRP || opcode == opcJumpARP) {
                if (level >= MAX_SUBR_LEVEL) {
                    throw new REBException("Maximum subroutine level exceeded");
                }
                int sAddr, count;
                if (opcode == opcJump || opcode == opcJumpRP) {
                    sAddr = (cmnd >> subaddV) & SEQ_PRG_M_SUBADD;
                }
                else {
                    sAddr = jumpRefs[(cmnd >> subaddV) & SEQ_PRG_M_PARM];
                }
                if (opcode == opcJump || opcode == opcJumpAP) {
                    count = cmnd & SEQ_PRG_M_SUBCNT;
                }
                else {
                    count = jumpReps[cmnd & SEQ_PRG_M_PARM];
                }
                int slice = getProgSlices(sAddr, level + 1, funcSlice, funcSend);
                nSlice += (slice >= 0 ? count * slice : -1 ^ slice);
                if (slice < 0) {
                    return nSlice ^ (level == 0 ? 0 : -1);
                }
            }
            else if (opcode == opcEndS) {
                if (level == 0) {
                    throw new REBException("Subroutine trailer at top level");
                }
                return nSlice;
            }
            else if (opcode == opcEndP) {
                return nSlice ^ (level == 0 ? 0 : -1);
            }
            else {
                throw new REBException("Invalid sequencer opcode ("
                                         + (cmnd >>> SEQ_PRG_V_OPCODE) + ")");
            }
        }

        if (level == 0) {
            throw new REBException("Missing end of program");
        }
        else {
            throw new REBException("Missing subroutine trailer");
        }
    }


   /**
    *  Gets the version number.
    *
    *  @return  The version number
    *
    *  @throws  REBException if firmware version is unknown
    */
    public int getVersion() throws REBException
    {
        return bss.getVersion(BaseSet.OPTN_SEQUENCER, true);
    }


   /**
    *  Checks a function number.
    *
    *  @param  func  The function ID
    *
    *  @throws  REBException if function ID is invalid
    */
    private void checkFunc(int func) throws REBException
    {
        int maxFunc = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_0)
                        ? SEQ_MAX_FUNC_0 : SEQ_MAX_FUNC;
        if (func < 0 || func >= maxFunc) {
            throw new REBException("Invalid function number");
        }
    }


   /**
    *  Checks a time slice number.
    *
    *  @param  slice  The time slice number
    *
    *  @throws  REBException if slice number is invalid
    */
    private void checkSlice(int slice) throws REBException
    {
        if (slice < 0 || slice >= SEQ_MAX_SLICE) {
            throw new REBException("Invalid time slice number");
        }
    }


   /**
    *  Checks a stack address.
    *
    *  @param  addr  The stack address
    *
    *  @throws  REBException if stack address is invalid
    */
    private void checkStackAddr(int addr) throws REBException
    {
        if (addr < 0 || addr >= SEQ_MAX_STACK) {
            throw new REBException("Invalid stack address");
        }
    }


   /**
    *  Checks a range of stack addresses.
    *
    *  @param  addr   The first stack address
    *
    *  @param  count  The number of stack addresses
    *
    *  @throws  REBException if stack address range is invalid
    */
    private void checkStackAddr(int addr, int count) throws REBException
    {
        if (addr < 0 || count <= 0 || addr + count > SEQ_MAX_STACK) {
            throw new REBException("Invalid stack address");
        }
    }


   /**
    *  Checks a program address.
    *
    *  @param  addr  The program address
    *
    *  @throws  REBException if program address id invalid
    */
    private void checkProgAddr(int addr) throws REBException
    {
        if (addr < 0 || addr >= SEQ_MAX_PROGRAM) {
            throw new REBException("Invalid program address");
        }
    }


   /**
    *  Checks a range of program addresses.
    *
    *  @param  addr   The first program address
    *
    *  @param  count  The number of program addresses
    *
    *  @throws  REBException if program address range is invalid
    */
    private void checkProgAddr(int addr, int count) throws REBException
    {
        if (addr < 0 || count <= 0 || addr + count > SEQ_MAX_PROGRAM) {
            throw new REBException("Invalid program address");
        }
    }


   /**
    *  Checks a parameter address.
    *
    *  @param  addr   The parameter address
    *
    *  @throws  REBException if parameter address is invalid
    */
    private void checkParamAddr(int addr) throws REBException
    {
        if (addr < 0 || addr >= SEQ_MAX_PARAMETER) {
            throw new REBException("Invalid parameter address");
        }
    }


   /**
    *  Gets an execute function repetition count value.
    *
    *  @param  count    The number of times to execute the function
    *
    *  @throws  REBException if count is invalid
    */
    private void checkExecCount(int count) throws REBException
    {
        if ((count & ~SEQ_PRG_M_EXCCNT) != 0) {
            throw new REBException("Invalid repetition count");
        }
    }


   /**
    *  Checks a jump repetition count value.
    *
    *  @param  count  The number of times to perform the jump
    *
    *  @throws  REBException if the count is invalid
    */
    private void checkJumpCount(int count) throws REBException
    {
        int mask = bss.isVersion(BaseSet.OPTN_SEQUENCER, BaseSet.VERSION_3)
                     ? SEQ_PRG_M_SUBCNT : SEQ_PRG_M_SUBCNT_1;
        if ((count & ~mask) != 0) {
            throw new REBException("Invalid repetition count");
        }
    }

}
