package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 **********************************************
 *
 *  Temperature ADCs reading routines.
 *
 *  There are five supported board versions:
 *
 *    VERSION_0: REB0
 *    VERSION_1: REB1
 *    VERSION_2: WREB
 *    VERSION_3: REB3+
 *    VERSION_4: GREB
 *
 *  @author Owen Saxton
 *
 **********************************************
 */
public class TempAdcs {

   /**
    *  Public constants
    */
    public final static int
        REG_TEMP_ADCS    = 0x600010,
/*      ADC_DREB_T1      = 0,
        ADC_DREB_T2      = 1,
        ADC_REB_T1       = 2,
        ADC_REB_T2       = 3,
        ADC_REB_T3       = 4,
        ADC_REB_T4       = 5,
        ADC_REB_T5       = 6,
        ADC_REB_T6       = 7,
        ADC_REB_T7       = 8,
        ADC_REB_T8       = 9,
        ADC_REB_T9       = 10, */
        NUM_TEMP_REGS_S  = 10,
        NUM_TEMP_REGS_S0 = 11,
        NUM_TEMP_REGS_W  = 6,
        NUM_TEMP_REGS_G  = 10,
        NUM_TEMP_REGS_M  = NUM_TEMP_REGS_S0,
        ERROR_MASK       = 0x00010000;

    public final static double
        TEMP_SCALE       = 1.0 / 128.0,
        TEMP_ERROR       = -8888.0;

   /**
    *  Private data
    */
    private static final Map<Integer, Integer> numRegsMap = new HashMap<>();
    static {
        numRegsMap.put(BaseSet.VERSION_0, NUM_TEMP_REGS_S0);
        numRegsMap.put(BaseSet.VERSION_1, NUM_TEMP_REGS_S0);
        numRegsMap.put(BaseSet.VERSION_2, NUM_TEMP_REGS_W);
        numRegsMap.put(BaseSet.VERSION_3, NUM_TEMP_REGS_S);
        numRegsMap.put(BaseSet.VERSION_4, NUM_TEMP_REGS_G);
    }
    private final BaseSet bss;


   /**
    *  Constructor.
    *
    *  @param  bss  The associated base set object
    */
    public TempAdcs(BaseSet bss)
    {
        this.bss = bss;
    }


   /**
    *  Enables the temperature ADC reading.
    *
    *  @throws  REBException 
    */
    public void enable() throws REBException
    {
        bss.getVersion(BaseSet.OPTN_BOARD_TEMP);
        bss.enable(BaseSet.RSET_TEMP_ADCS);
    }


   /**
    *  Waits for the data to be available.
    *
    *  @throws  REBException 
    */
    public void waitDone() throws REBException
    {
        bss.getVersion(BaseSet.OPTN_BOARD_TEMP);
        bss.waitDone(BaseSet.RSET_TEMP_ADCS);
    }


   /**
    *  Gets the time of the read enable.
    *
    *  @return  The Unix millisecond time of the read enable
    *
    *  @throws  REBException 
    */
    public long getTriggerTime() throws REBException
    {
        bss.checkNotVersion(BaseSet.OPTN_BOARD_TEMP, BaseSet.VERSION_0);
        return bss.getTriggerTime(BaseSet.RSET_TEMP_ADCS);
    }


   /**
    *  Reads one temperature value.
    *
    *  @param  adc  The number of the ADC to read
    *
    *  @return  The ADC value (Celsius)
    *
    *  @throws  REBException 
    */
    public double readAdc(int adc) throws REBException
    {
        if (adc < 0 || adc >= getNumRegs()) {
            throw new REBException("Invalid temperature ADC number");
        }
        enable();
        waitDone();
        int value = bss.read(REG_TEMP_ADCS + adc);

        return (value & ERROR_MASK) != 0 ? TEMP_ERROR : TEMP_SCALE * (short)value;
    }


   /**
    *  Reads all the temperature values.
    *
    *  @return  The array of ADC values (Celsius)
    *
    *  @throws  REBException 
    */
    public double[] readAdcs() throws REBException
    {
        return readAdcs(0, getNumRegs());
    }


   /**
    *  Reads a range of temperature values.
    *
    *  @param  first  The number of the first ADC to read
    *
    *  @param  count  The number of ADCs to read
    *
    *  @return  The array of ADC values (Celsius)
    *
    *  @throws  REBException 
    */
    public double[] readAdcs(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > getNumRegs()) {
            throw new REBException("Invalid temperature ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[count];
        double[] data = new double[count];
        bss.read(REG_TEMP_ADCS + first, rawData);
        for (int j = 0; j < count; j++) {
            int value = rawData[j];
            data[j] = (value & ERROR_MASK) != 0
                        ? TEMP_ERROR : TEMP_SCALE * (short)value;
        }

        return data;
    }


   /**
    *  Gets the number of ADC registers.
    *
    *  @return  The number of ADC registers
    *
    *  @throws  REBException 
    */
    public int getNumRegs() throws REBException
    {
        return numRegsMap.get(bss.getVersion(BaseSet.OPTN_BOARD_TEMP));
    }

}
