package org.lsst.ccs.drivers.reb;

import java.nio.ByteBuffer;

/**
 ***************************
 *
 *  Image data
 *
 *  @author Owen Saxton
 *
 ***************************
 */
public class Image extends ImageMetadata {

   /*
    *  Private data.
    */
    private long mdata;               // Reference to the C++ metadata object
    private ImageClient.Impl client;  // The image client used
    private ByteBuffer data;          // The image data


   /*
    *  Constructors.
    */
    public Image() {
        super();
    }

    public Image(String name, long timestamp, int length, int address,
                 long sciId, int sensorType, int stripes, int[] registers,
                 int sciVersion, String clientVersion, String serverVersion,
                 long event, int[] members, long mdata, ImageClient.Impl client) {
        super(name, timestamp, length, address, sciId, sensorType, stripes,
              registers, sciVersion, clientVersion, serverVersion, event, members);
        this.mdata = mdata;
        this.client = client;
    }

    public Image(String name, long timestamp, int length, int address,
                 long sciId, int sensorType, int stripes, int[] registers,
                 int sciVersion, String clientVersion, String serverVersion,
                 long event, int[] members, ByteBuffer data) {
        super(name, timestamp, length, address, sciId, sensorType, stripes,
              registers, sciVersion, clientVersion, serverVersion, event, members);
        this.data = data;
    }

    @Deprecated
    public Image(long tag, int length, int format, int schema, int version,
                 int address, int cluster, int element, long mdata,
                 ImageClient.Impl client) {
        super(tag, length, format, schema, version, address, cluster, element);
        this.mdata = mdata;
        this.client = client;
    }

    @Deprecated
    public Image(long tag, int length, int format, int schema, int version,
                 int address, int cluster, int element, ByteBuffer data) {
        super(tag, length, format, schema, version, address, cluster, element);
        this.data = data;
    }


   /**
    *  Gets the image pixel data.
    *
    *  @return  The image's pixel data if it has been successfully read, or
    *           null if it hasn't been. The data returned is a read-only
    *           wrapper around the image data to assure the user cannot
    *           change the state of the image.
    */
    public ByteBuffer getData() {
        return (data != null && data.limit() > 0) ? data.asReadOnlyBuffer() : null;
    }


   /**
    *  Gets the image metadata.
    *
    *  @return  The image's metadata
    */
    public ImageMetadata getMetadata() {
        return new ImageMetadata(name, timestamp, length, address, sciId,
                                 sensorType, stripes, registers, sciVersion,
                                 clientVersion, serverVersion, event, members);
    }


   /**
    *  Deletes the metadata reference.
    */
    public void deleteMetadataRef() {
        if (mdata != 0) {
            client.deleteImageMetadataRef(this);
        }
    }


   /**
    *  Performs finalization.
    *
    *  Allocated non-java memory (in this case the C++ image metadata object)
    *  has to be freed.
    *
    *  @throws Throwable
    */
    @Override
    public void finalize() throws Throwable {
        super.finalize();
        deleteMetadataRef();
    }

    private static final long serialVersionUID = -7388976703992252134L;
}
