package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 ********************************************
 *
 *  Routines for setting the on-board DACs.
 *
 *  @author Owen Saxton
 *
 ********************************************
 */
public class BoardDacs {

   /**
    *  Public constants
    */
    public static final int
        VERSION_UNSUPP   = BaseSet.VERSION_UNSUPP,
        VERSION_REB0_0   = BaseSet.VERSION_0,
        VERSION_REB0_1   = BaseSet.VERSION_1,
        VERSION_REB3     = BaseSet.VERSION_3,
        VERSION_REB5     = BaseSet.VERSION_6,
        VERSION_WREB1    = BaseSet.VERSION_2,
        VERSION_WREB2    = BaseSet.VERSION_4,
        VERSION_GREB1    = BaseSet.VERSION_5,
        VERSION_GREB2    = BaseSet.VERSION_7,

        REG_BOARD_DACS   = 0x400000,
        REG_BOARD_DACS_0 = 0x500001,
        OFF_CLOCK_RAILS  = 0,
        OFF_CS_GATE      = 0x10,
        OFF_HEATER       = 0X10,
        OFF_CABAC_ALT    = 0x100,
        OFF_BIAS_GATE    = 0X100,
        INC_STRIPS       = 0x10,
        DACS_SET         = 0,
        DACS_LOAD        = 1,
        DAC_HAS_STRIP    = 0x10,
        DAC_CLOCK_RAILS  = 0,
        DAC_CS_GATE      = 1,
        DAC_CABAC_ALT    = 2,
        DAC_HEATER       = 3,
        DAC_BIAS_GATE    = DAC_HAS_STRIP,
        //NUM_STRIPS       = Asic.NUM_STRIPS,

        CHAN_SCLK_L      = 0,
        CHAN_SCLK_L_SH   = 1,
        CHAN_SCLK_H      = 2,
        CHAN_SCLK_H_SH   = 3,
        CHAN_RG_L        = 4,
        CHAN_RG_L_SH     = 5,
        CHAN_RG_H        = 6,
        CHAN_RG_H_SH     = 7,
        CHAN_PCLK_L      = 8,
        CHAN_PCLK_L_SH   = 9,
        CHAN_PCLK_H      = 10,
        CHAN_PCLK_H_SH   = 11,
        CHAN_OD          = 12,
        CHAN_GD          = 13,
        CHAN_RD          = 14,
        CHAN_OG          = 15,
        CHAN_OG_SH       = 16,
        CHAN_CSGATE      = 17,
        CHAN_HEATER      = 18,
        CHAN_CSGATE_1    = CHAN_CSGATE,
        CHAN_CSGATE_2    = 19,
        CHAN_CSGATE_3    = 20,
        CHAN_HEATER_1    = CHAN_HEATER,
        CHAN_HEATER_2    = 21;
        //CHAN_OD_1        = 25,
        //CHAN_OD_2        = 26,
        //CHAN_OD_3        = 27,
        //CHAN_OD_CTRL     = CHAN_OD;
        //CHAN_FSB_CTRL    = 17,
    
   /**
    *  Maps.
    */
    private static final Map<Integer, Integer> dacMapReb0 = new HashMap<>();
    static {
        dacMapReb0.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMapReb0.put(DAC_CS_GATE,     OFF_CS_GATE);
    }

    private static final Map<Integer, Integer> chanMapReb0 = new HashMap<>();
    static {
        chanMapReb0.put(CHAN_SCLK_L,   (DAC_CLOCK_RAILS << 16) | 0);
        chanMapReb0.put(CHAN_SCLK_H,   (DAC_CLOCK_RAILS << 16) | 1);
        chanMapReb0.put(CHAN_RG_L,     (DAC_CLOCK_RAILS << 16) | 2);
        chanMapReb0.put(CHAN_RG_H,     (DAC_CLOCK_RAILS << 16) | 3);
        chanMapReb0.put(CHAN_PCLK_L,   (DAC_CLOCK_RAILS << 16) | 4);
        chanMapReb0.put(CHAN_PCLK_H,   (DAC_CLOCK_RAILS << 16) | 5);
        chanMapReb0.put(CHAN_HEATER_1, (DAC_CLOCK_RAILS << 16) | 6);
        chanMapReb0.put(CHAN_HEATER_2, (DAC_CLOCK_RAILS << 16) | 7);
        chanMapReb0.put(CHAN_CSGATE_1, (DAC_CS_GATE << 16) | 0);
        chanMapReb0.put(CHAN_CSGATE_2, (DAC_CS_GATE << 16) | 1);
        chanMapReb0.put(CHAN_CSGATE_3, (DAC_CS_GATE << 16) | 2);
    }

    private static final Map<Integer, Integer> dacMapReb3 = new HashMap<>();
    static {
        dacMapReb3.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMapReb3.put(DAC_HEATER,      OFF_HEATER);
        dacMapReb3.put(DAC_BIAS_GATE,   OFF_BIAS_GATE);
    }

    private static final Map<Integer, Integer> chanMapReb3 = new HashMap<>();
    static {
        chanMapReb3.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMapReb3.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMapReb3.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMapReb3.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMapReb3.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMapReb3.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMapReb3.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMapReb3.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMapReb3.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMapReb3.put(CHAN_HEATER,    (DAC_HEATER << 16) | 0);
        chanMapReb3.put(CHAN_GD,        (DAC_BIAS_GATE << 16) | 0);
        chanMapReb3.put(CHAN_OD,        (DAC_BIAS_GATE << 16) | 1);
        chanMapReb3.put(CHAN_OG_SH,     (DAC_BIAS_GATE << 16) | 2);
        chanMapReb3.put(CHAN_OG,        (DAC_BIAS_GATE << 16) | 3);
        chanMapReb3.put(CHAN_RD,        (DAC_BIAS_GATE << 16) | 4);
        chanMapReb3.put(CHAN_CSGATE,    (DAC_BIAS_GATE << 16) | 5);
    }

    private static final Map<Integer, Integer> dacMapWreb1 = new HashMap<>();
    static {
        dacMapWreb1.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMapWreb1.put(DAC_CS_GATE,     OFF_CS_GATE);
        dacMapWreb1.put(DAC_CABAC_ALT,   OFF_CABAC_ALT);
    }

    private static final Map<Integer, Integer> chanMapWreb1 = new HashMap<>();
    static {
        chanMapWreb1.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMapWreb1.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMapWreb1.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMapWreb1.put(CHAN_SCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMapWreb1.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMapWreb1.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMapWreb1.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x06);
        chanMapWreb1.put(CHAN_RG_H_SH,   (DAC_CLOCK_RAILS << 16) | 0x07);
        chanMapWreb1.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMapWreb1.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMapWreb1.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMapWreb1.put(CHAN_PCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x13);
        chanMapWreb1.put(CHAN_CSGATE,    (DAC_CS_GATE << 16) | 0);
        //chanMapWreb1.put(CHAN_CSGATE_1,  (DAC_CS_GATE << 16) | 0);
        //chanMapWreb1.put(CHAN_CSGATE_2,  (DAC_CS_GATE << 16) | 1);
        //chanMapWreb1.put(CHAN_CSGATE_3,  (DAC_CS_GATE << 16) | 2);
        //chanMapWreb1.put(CHAN_FSB_CTRL,  (DAC_CS_GATE << 16) | 3);
        chanMapWreb1.put(CHAN_HEATER_1,  (DAC_CS_GATE << 16) | 4);
        chanMapWreb1.put(CHAN_HEATER_2,  (DAC_CS_GATE << 16) | 5);
        chanMapWreb1.put(CHAN_OD,        (DAC_CS_GATE << 16) | 6);
        chanMapWreb1.put(CHAN_GD,        (DAC_CABAC_ALT << 16) | 0);
        chanMapWreb1.put(CHAN_RD,        (DAC_CABAC_ALT << 16) | 1);
        chanMapWreb1.put(CHAN_OG,        (DAC_CABAC_ALT << 16) | 2);
        chanMapWreb1.put(CHAN_OG_SH,     (DAC_CABAC_ALT << 16) | 3);
    }

    private static final Map<Integer, Integer> chanMapWreb2 = new HashMap<>();
    static {
        chanMapWreb2.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMapWreb2.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMapWreb2.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMapWreb2.put(CHAN_SCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMapWreb2.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMapWreb2.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMapWreb2.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x06);
        chanMapWreb2.put(CHAN_RG_H_SH,   (DAC_CLOCK_RAILS << 16) | 0x07);
        chanMapWreb2.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMapWreb2.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMapWreb2.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMapWreb2.put(CHAN_PCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x13);
        chanMapWreb2.put(CHAN_CSGATE,    (DAC_CS_GATE << 16) | 2);
        //chanMapWreb2.put(CHAN_CSGATE_1,  (DAC_CS_GATE << 16) | 0);
        //chanMapWreb2.put(CHAN_CSGATE_2,  (DAC_CS_GATE << 16) | 1);
        //chanMapWreb2.put(CHAN_CSGATE_3,  (DAC_CS_GATE << 16) | 2);
        chanMapWreb2.put(CHAN_HEATER,    (DAC_CS_GATE << 16) | 4);
        chanMapWreb2.put(CHAN_OD,        (DAC_CS_GATE << 16) | 5);
        //chanMapWreb2.put(CHAN_OD_1,      (DAC_CS_GATE << 16) | 5);
        //chanMapWreb2.put(CHAN_OD_2,      (DAC_CS_GATE << 16) | 6);
        //chanMapWreb2.put(CHAN_OD_3,      (DAC_CS_GATE << 16) | 7);
        chanMapWreb2.put(CHAN_GD,        (DAC_CABAC_ALT << 16) | 0);
        chanMapWreb2.put(CHAN_RD,        (DAC_CABAC_ALT << 16) | 1);
        chanMapWreb2.put(CHAN_OG,        (DAC_CABAC_ALT << 16) | 2);
        chanMapWreb2.put(CHAN_OG_SH,     (DAC_CABAC_ALT << 16) | 3);
    }

    private static final Set<Integer> wrebStripSet = new HashSet<>();
    static {
        wrebStripSet.add(CHAN_GD);
        wrebStripSet.add(CHAN_RD);
        wrebStripSet.add(CHAN_OG);
        wrebStripSet.add(CHAN_OG_SH);
        wrebStripSet.add(CHAN_OD);
        wrebStripSet.add(CHAN_CSGATE);
    }

    private static final Map<Integer, Integer> dacMapGreb1 = new HashMap<>();
    static {
        dacMapGreb1.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMapGreb1.put(DAC_BIAS_GATE,   OFF_BIAS_GATE);
    }

    private static final Map<Integer, Integer> chanMapGreb1 = new HashMap<>();
    static {
        chanMapGreb1.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMapGreb1.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMapGreb1.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMapGreb1.put(CHAN_SCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMapGreb1.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMapGreb1.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMapGreb1.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x06);
        chanMapGreb1.put(CHAN_RG_H_SH,   (DAC_CLOCK_RAILS << 16) | 0x07);
        chanMapGreb1.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMapGreb1.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMapGreb1.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMapGreb1.put(CHAN_PCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x13);
        chanMapGreb1.put(CHAN_HEATER,    (DAC_CLOCK_RAILS << 16) | 0x14);
        chanMapGreb1.put(CHAN_GD,        (DAC_BIAS_GATE << 16) | 0);
        chanMapGreb1.put(CHAN_RD,        (DAC_BIAS_GATE << 16) | 1);
        chanMapGreb1.put(CHAN_OG,        (DAC_BIAS_GATE << 16) | 2);
        chanMapGreb1.put(CHAN_OG_SH,     (DAC_BIAS_GATE << 16) | 3);
        chanMapGreb1.put(CHAN_CSGATE,    (DAC_BIAS_GATE << 16) | 4);
        chanMapGreb1.put(CHAN_OD,        (DAC_BIAS_GATE << 16) | 5);
    }

    private static final Map<Integer, Map> chanMapMap = new HashMap<>();
    static {
        chanMapMap.put(VERSION_REB0_0, chanMapReb0);
        chanMapMap.put(VERSION_REB0_1, chanMapReb0);
        chanMapMap.put(VERSION_REB3,   chanMapReb3);
        chanMapMap.put(VERSION_REB5,   chanMapReb3);
        chanMapMap.put(VERSION_WREB1,  chanMapWreb1);
        chanMapMap.put(VERSION_WREB2,  chanMapWreb2);
        chanMapMap.put(VERSION_GREB1,  chanMapGreb1);
        chanMapMap.put(VERSION_GREB2,  chanMapGreb1);
    }

    private static final Map<Integer, Map> dacMapMap = new HashMap<>();
    static {
        dacMapMap.put(VERSION_REB0_0, dacMapReb0);
        dacMapMap.put(VERSION_REB0_1, dacMapReb0);
        dacMapMap.put(VERSION_REB3,   dacMapReb3);
        dacMapMap.put(VERSION_REB5,   dacMapReb3);
        dacMapMap.put(VERSION_WREB1,  dacMapWreb1);
        dacMapMap.put(VERSION_WREB2,  dacMapWreb1);
        dacMapMap.put(VERSION_GREB1,  dacMapGreb1);
        dacMapMap.put(VERSION_GREB2,  dacMapGreb1);
    }

    BaseSet bss;


   /**
    *  Constructor.
    *
    *  @param  bss  The associated base set object
    */
    public BoardDacs(BaseSet bss)
    {
        this.bss = bss;
    }


   /**
    *  Gets the DAC version.
    *
    *  @return  The version number: VERSION_0 & VERSION_1 for REB0; VERSION_2
    *           for WREB1; VERSION_3 for REB3/4; VERSION_4 for WREB2;
    *           VERSION_5 for GREB/WREB3; VERSION_6 for REB5
    */
    public int getVersion()
    {
        try {
            return bss.getVersion(BaseSet.OPTN_BOARD_DACS);
        }
        catch (REBException e) {
            return BaseSet.VERSION_UNSUPP;
        }
    }


   /**
    *  Gets the number of strips.
    *
    *  @return  The number of strips
    *
    *  @throws  REBException if the firmware version is unknown
    */
    public int getNumStrips() throws REBException
    {
        return bss.getNumStrips();
    }


   /**
    *  Loads all DAC values.
    *
    *  All the values previously set in the DACs are loaded into their output
    *  registers
    *
    *  @throws  REBException 
    */
    public void loadAll() throws REBException
    {
        loadGlobal();
        loadStrip();
    }


   /**
    *  Loads all global DAC values.
    *
    *  All the values previously set in the global DACs are loaded into their
    *  output registers
    *
    *  @throws  REBException 
    */
    public void loadGlobal() throws REBException
    {
        for (int dac : getDacMap(getVersion()).keySet()) {
            if ((dac & DAC_HAS_STRIP) == 0) {
                load(dac);
            }
        }
    }


   /**
    *  Loads all strip-associated DAC values.
    *
    *  All the values previously set in the strip-associated DACs are loaded
    *  into their output registers
    *
    *  @throws  REBException 
    */
    public void loadStrip() throws REBException
    {
        for (int strip = 0; strip < getNumStrips(); strip++) {
            loadStrip(strip);
        }
    }


   /**
    *  Loads all DAC values for a strip.
    *
    *  All the values previously set for a particular strip are loaded into
    *  their output registers
    *
    *  @param  strip  The strip number
    *
    *  @throws  REBException 
    */
    public void loadStrip(int strip) throws REBException
    {
        int version = getVersion();
        if (version == VERSION_WREB1 || version == VERSION_WREB2) { // Fake it
            load(DAC_CS_GATE);
            load(DAC_CABAC_ALT);
        }
        else {
            for (int dac : getDacMap(version).keySet()) {
                if ((dac & DAC_HAS_STRIP) != 0) {
                    load(strip, dac);
                }
            }
        }
    }


   /**
    *  Loads one DAC's values.
    *
    *  All the values previously set in a DAC are loaded into its output
    *  registers.  This is a no-op for VERSION_REB0_0
    *
    *
    *  @param  dac  The DAC number
    *
    *  @throws  REBException 
    */
    public void load(int dac) throws REBException
    {
        if ((dac & DAC_HAS_STRIP) == 0) {
            int version = getVersion();
            if (version == VERSION_REB0_0) return;
            bss.write(getBase(version, dac, false) + DACS_LOAD, 0);
        }
        else {
            for (int strip = 0; strip < getNumStrips(); strip++) {
                load(strip, dac);
            }
        }
    }


   /**
    *  Loads the values for one DAC and strip.
    *
    *  All the values previously set in a DAC are loaded into its output
    *  registers.
    *
    *  @param  strip  The strip number.
    *
    *  @param  dac    The DAC number
    *
    *  @throws  REBException 
    */
    public void load(int strip, int dac) throws REBException
    {
        int version = getVersion();
        bss.write(getBase(version, dac, true) + getStripOffset(version, strip, true) + DACS_LOAD, 0);
    }


   /**
    *  Sets a DAC value.
    *
    *  The supplied value is set into its DAC, but not loaded into the DAC's
    *  output register until the load method is called
    *
    *  @param  chan   The overall channel number
    *
    *  @param  value  The value to be set
    *
    *  @throws  REBException 
    */
    public void set(int chan, int value) throws REBException
    {
        set(chan, value, false);
    }


   /**
    *  Sets a DAC value.
    *
    *  The supplied value is set into its DAC, and optionally loaded into
    *  the DAC's output register
    *
    *  @param  chan   The overall channel number
    *
    *  @param  value  The value to be set (0 - 4095)
    *
    *  @param  load   Whether to load the DAC's output register
    *
    *  @throws  REBException 
    */
    public void set(int chan, int value, boolean load) throws REBException
    {
        int version = getVersion();
        int desc = getDescriptor(version, chan, false);
        int dac = desc >> 16;
        int addr = getBase(version, dac, false) + DACS_SET;
        if (dac == DAC_HEATER) {
            bss.write(addr, 0x380001);  // Use internal 5V reference
            bss.write(addr, ((desc & 0xffff) << 16) | ((value & 0xfff) << 4));
        }
        else {
            bss.write(addr, ((desc & 0xffff) << 12) | (value & 0x0fff));
        }
        if (load) {
            load(dac);
        }
    }


   /**
    *  Sets a DAC value into a strip.
    *
    *  The supplied value is set into the its DAC, but not loaded into the
    *  DAC's output register until the load method is called
    *
    *  @param  strip  The strip number.
    *
    *  @param  chan   The overall channel number
    *
    *  @param  value  The value to be set
    *
    *  @throws  REBException 
    */
    public void set(int strip, int chan, int value) throws REBException
    {
        set(strip, chan, value, false);
    }


   /**
    *  Sets a DAC value into a strip.
    *
    *  The supplied value is set into the its DAC, and optionally loaded
    *  into the DAC's output register
    *
    *  @param  strip  The strip number.
    *
    *  @param  chan   The overall channel number
    *
    *  @param  value  The value to be set
    *
    *  @param  load   Whether to load the DAC's output register
    *
    *  @throws  REBException 
    */
    public void set(int strip, int chan, int value, boolean load) throws REBException
    {
        int version = getVersion();
        if (version == VERSION_WREB1 || version == VERSION_WREB2) { // Fake it
            int desc = getWrebDescriptor(version, strip, chan);
            int dac = desc >> 16;
            bss.write(getBase(version, dac, false) + DACS_SET,
                      ((desc & 0xffff) << 12) | (value & 0x0fff));
            if (load) {
                load(dac);
            }
        }
        else {
            int desc = getDescriptor(version, chan, true);
            int dac = desc >> 16;
            bss.write(getBase(version, dac, true) + getStripOffset(version, strip, false) + DACS_SET,
                      ((desc & 0xffff) << 12) | (value & 0x0fff) | (version == VERSION_GREB2 ? (strip << 16) : 0));
            if (load) {
                load(strip, dac);
            }
        }
    }


   /**
    *  Gets a DAC's base register address.
    *
    *  @param  version   The board DACs system version
    *
    *  @param  dac       The DAC number
    *
    *  @param  hasStrip  Whether DAC is associated with a strip
    *
    *  @return  The base address of the DAC
    *
    *  @throws  REBException 
    */
    private int getBase(int version, int dac, boolean hasStrip)
        throws REBException
    {
        Integer base = getDacMap(version).get(dac);
        if (base == null || ((dac & DAC_HAS_STRIP) != 0) != hasStrip) {
            throw new REBException("Invalid DAC number (" + dac + ")");
        }
        return (version == VERSION_REB0_0 ? REG_BOARD_DACS_0 : REG_BOARD_DACS) + base;
    }


   /**
    *  Gets a strip's register address offset.
    *
    *  @param  version  The board DACs system version
    *
    *  @param  strip    The strip number
    *
    *  @param  isLoad   True if for load operation
    *
    *  @return  The strip address offset
    *
    *  @throws  REBException 
    */
    private int getStripOffset(int version, int strip, boolean isLoad) throws REBException
    {
        if (strip < 0 || strip >= getNumStrips()) {
            throw new REBException("Invalid strip number (" + strip + ")");
        }
        return version == VERSION_GREB1 ? isLoad ? strip : 0 : version == VERSION_GREB2 ? 0 : INC_STRIPS * strip;
    }


   /**
    *  Gets the applicable DAC map.
    *
    *  @param  version   The option version number
    *
    *  @return  The DAC map
    */
    private Map<Integer, Integer> getDacMap(int version)
    {
        return dacMapMap.get(version);
    }


   /**
    *  Gets the appropriate channel map.
    *
    *  @param  version   The option version number
    *
    *  @return  The channel map
    */
    private Map<Integer, Integer> getChanMap(int version)
    {
        return chanMapMap.get(version);
    }


   /**
    *  Gets a channel descriptor.
    *
    *  @param  version   The option version number
    *
    *  @param  chan      The channel number
    *
    *  @param  hasStrip  Whether channel is associated with a strip
    *
    *  @return  The channel descriptor
    *
    *  @throws  REBException 
    */
    private int getDescriptor(int version, int chan, boolean hasStrip)
        throws REBException
    {
        Map<Integer, Integer> chanMap = getChanMap(version);
        Integer desc = chanMap.get(chan);
        if (desc == null || (((desc >> 16) & DAC_HAS_STRIP) != 0) != hasStrip) {
            throw new REBException("Invalid channel number (" + chan + ")");
        }
        return desc;
    }


   /**
    *  Gets a channel descriptor for a WREB "strip".
    *
    *  @param  version   The DAC version number
    *
    *  @param  strip     The strip number
    *
    *  @param  chan      The channel number
    *
    *  @return  The channel descriptor
    *
    *  @throws  REBException 
    */
    private int getWrebDescriptor(int version, int strip, int chan)
        throws REBException
    {
        if (strip != 0) {
            throw new REBException("Invalid strip number (" + strip + ")");
        }
        if (!wrebStripSet.contains(chan)) {
            throw new REBException("Invalid channel number (" + chan + ")");
        }

        return getChanMap(version).get(chan);
    }

}
