package org.lsst.ccs.drivers.reb;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 **************************************************************
 *
 *  Program to test the Java image client interface routines
 *
 *  @author Owen Saxton
 *
 **************************************************************
 */
public class TestImage implements ImageClient.Listener {

   /*
    *  Private fields
    */
    private final static String CUP = "\033[A", NL = "\n";
    private final ImageClient imc = new ImageClient();
    private final BaseSet reg = new BaseSet();
    private int rebId = -1, imgLength = -1, imgOffset = 0;
    private boolean listening, showTime, reuseImage = true;
    private Image image = new Image();


   /**
    *  Enumeration for on/off.
    */
    public enum OnOff {

        ON, OFF;

    }


   /**
    *  Enumeration for hardware type.
    */
    public enum HdwType {

        DAQ0(BaseSet.HDW_TYPE_DAQ0),
        DAQ1(BaseSet.HDW_TYPE_DAQ1),
        PCI(BaseSet.HDW_TYPE_PCI);

        int value;

        HdwType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    *  Connects to a REB.
    *
    *  @param  id  The REB ID
    *
    *  @throws  REBException
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="id", description="REB ID")
                        int id) throws REBException
    {
        imc.open(id);
        reg.open(id);
        rebId = id;
    }


   /**
    *  Connects to a REB.
    *
    *  @param  id       The REB ID
    *
    *  @param  ifcName  The network interface name
    *
    *  @throws  REBException
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="id", description="REB ID")
                        int id,
                        @Argument(name="ifcname",
                                  description="Network interface name")
                        String ifcName) throws REBException
    {
        imc.open(id, ifcName);
        reg.open(id, ifcName);
        rebId = id;
    }


   /**
    *  Connects to a REB.
    *
    *  @param  type     The hardware type (DAQ0, DAQ1, PCI)
    *
    *  @param  id       The REB ID
    *
    *  @param  ifcName  The network interface name or PCI device name
    *
    *  @throws  REBException
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="type", description="Hardware type")
                        HdwType type,
                        @Argument(name="id", description="REB ID")
                        int id,
                        @Argument(name="ifcname",
                                  description="Network interface name")
                        String ifcName) throws REBException
    {
        imc.open(type.getValue(), id, ifcName);
        reg.open(type.getValue(), id, ifcName);
        rebId = id;
    }


   /**
    *  Disconnects from a REB.
    *
    *  @throws  REBException
    */
    @Command(name="disconnect", description="Disconnect from a REB")
    public void disconnect() throws REBException
    {
        rebId = -1;
        imc.close();
        reg.close();
    }


   /**
    *  Turns image data buffer re-use on or off.
    *
    *  @param  action  The buffer re-use action: ON or OFF
    */
    @Command(name="reuse", description="Enable or disable image buffer re-use")
    public void reuse(@Argument(name="action", description="Action to take")
                      OnOff action)
    {
        reuseImage = (action == OnOff.ON);
    }


   /**
    *  Turns image acquisition time display on or off.
    *
    *  @param  action  The display action: ON or OFF
    */
    @Command(name="time", description="Enable or disable time display")
    public void time(@Argument(name="action", description="Action to take")
                     OnOff action)
    {
        showTime = (action == OnOff.ON);
    }


   /**
    *  Turns the listener on or off.
    *
    *  @param  action  The listener action: ON or OFF
    *
    *  @return  "Cursor up" string if success; error message otherwise
    */
    @Command(name="listen", description="Enable or disable the listener")
    public String listen(@Argument(name="action", description="Action to take")
                         OnOff action)
    {
        String result = checkConnect();
        if (result != null) return result;
        if (action.ordinal() == 0) {
            listening = true;
            imc.setListener(this, reuseImage ? image : null);
        }
        else {
            listening = false;
            imc.clearListener();
        }

        return CUP;
    }


   /**
    *  Sets the internal image display region.
    *
    *  @param  length  The length of the display region
    */
    @Command(name="region", description="Set the image display region")
    public void region(@Argument(name="length",
                                 description="The region length (pixels)")
                       int length)
    {
        imgLength = length;
    }


   /**
    *  Sets the internal image display region.
    *
    *  @param  length  The length of the display region
    *
    *  @param  offset  The offset to the start of the region
    */
    @Command(name="region", description="Set the image display region")
    public void region(@Argument(name="length",
                                 description="The region length (pixels)")
                       int length,
                       @Argument(name="offset",
                                 description="The offset to the region (pixels)")
                       int offset)
    {
        imgLength = length;
        imgOffset = offset;
    }


   /**
    *  Displays parameters.
    *
    *  @return  The result string
    */
    @Command(name="show", description="Display parameters")
    public String show()
    {
        return String.format(  "REB ID       = %-8s  Image length = %s" +
                             "\nImage offset = %-8s  Listener on  = %s" +
                             "\nShow time    = %-8s  Reuse image  = %s",
                             rebId, imgLength, imgOffset, listening, showTime,
                             reuseImage);
    }


   /**
    *  Waits for an image to arrive.
    *
    *  @return  The result string
    *
    *  @throws  REBException
    */
    @Command(name="wait", description="Wait for an image to arrive")
    public String awaitImage() throws REBException
    {
        String result = checkConnect();
        if (result != null) return result;
        result = checkListen();
        if (result != null) return result;
        image = imc.awaitImage(reuseImage ? image : null);
        if (showTime) {
            long cTime = reg.getTime();
            long tTime = reg.getTriggerTime(BaseSet.RSET_SEQUENCER);
            result = "Elapsed time = " + (cTime - tTime) + " msec";
        }
        String text = formatMetadata(image, "Got metadata", false);

        return (result == null) ? text : result + NL + text;
    }


   /**
    *  Reads an image.
    *
    *  @return  The result string
    *
    *  @throws  REBException
    */
    @Command(name="read", description="Read an image")
    public String read() throws REBException
    {
        String result = checkConnect();
        if (result != null) return result;
        result = checkListen();
        if (result != null) return result;
        long start = System.currentTimeMillis();
        boolean imgOkay = imc.readImage(image);
        if (showTime) {
            result = "Elapsed time = " + (System.currentTimeMillis() - start)
                       + " msec";
        }
        if (!imgOkay) {
            String text = "Error reading image data";
            result = (result == null) ? text : result + NL + text;
        }

        return (result == null) ? CUP : result;
    }


   /**
    *  Displays image data.
    *
    *  @param  length  The number of pixels to display
    *
    *  @return  The result string
    */
    @Command(name="display", description="Display image data")
    public String display(@Argument(name="length",
                                    description="The display region length (pixels)")
                       int length)
    {
        String text = formatImage(image, 0, length);
        return (text == null) ? CUP : text;
    }


   /**
    *  Displays image data.
    *
    *  @param  length  The number of pixels to display
    *
    *  @param  offset  The offset to the first pixel
    *
    *  @return  The result string
    */
    @Command(name="display", description="Display image data")
    public String display(@Argument(name="length",
                                    description="The display region length (pixels)")
                          int length,
                          @Argument(name="offset",
                                    description="The offset to the display region (pixels)")
                          int offset)
    {
        String text = formatImage(image, offset, length);
        return (text == null) ? CUP : text;
    }


   /**
    *  Displays image metadata.
    *
    *  @return  The result string
    */
    @Command(name="metadata", description="Display image metadata")
    public String metadata()
    {
        return formatMetadata(image, "Metadata", true);
    }


   /**
    *  Resets the front end system.
    *
    *  @throws  REBException
    */
    @Command(name="reset", description="Reset the front end")
    public void reset() throws REBException
    {
        imc.resetFrontEnd();
    }


   /**
    *  Handles each received image.
    *
    *  @param  img   The received image
    */
    @Override
    public void processImage(Image img)
    {
        image = img;
        if (showTime) {
            try {
                long cTime = reg.getTime();
                long tTime = reg.getTriggerTime(BaseSet.RSET_SEQUENCER);
                System.out.println("Elapsed time = " + (cTime - tTime)
                                     + " msec");
            }
            catch (REBException e) {            
            }
        }
        System.out.println(formatMetadata(image, "Got image", true));
        if (image.getData() != null) {
            String text = formatImage(image, imgOffset, imgLength);
            if (text != null) {
                System.out.println(text);
            }
        }
    }


   /**
    *  Formats the metadata for an image.
    *
    *  @param  img        The image whose metadata is to be displayed
    *
    *  @param  caption    A caption to be displayed before the metadata
    *
    *  @param  showValid  If true, display whether or not the image contains
    *                     valid data
    *
    *  @return  A string containing the formatted metadata
    */
    private String formatMetadata(Image img, String caption, boolean showValid)
    {
        StringBuilder blanks = new StringBuilder(caption.length());
        for (int j = 0; j < caption.length(); j++) {
            blanks.append(" ");
        }
        return String.format("%s: tag = %016x, length = %s, format = %s," + NL
                               + "%s  schema = %08x,  version = %08x, "
                               + "address = %s," + NL
                               + "%s  cluster = %s, element = %s%s",
                             caption, image.getTag(), image.getLength(),
                             image.getFormat(), blanks, image.getSchema(),
                             image.getVersion(), image.getAddress(),
                             blanks, image.getCluster(), image.getElement(),
                             showValid ? image.getData() != null ? ", valid"
                                       : ", invalid" : "");
    }


   /**
    *  Formats the contents of an image.
    *
    *  @param  img     The image to be displayed
    *
    *  @param  offset  The offset to the first pixel to display
    *
    *  @param  count   The number of pixels to display.  If zero, display up
    *                  to the stored length.  If negative, display nothing.
    *
    *  @return  The forrmatted image contents
    */
    private String formatImage(Image img, int offset, int count)
    {
        if (count < 0) {
            return null;
        }
        ByteBuffer buff = img.getData();
        if (buff == null) {
            return "Image contains no valid data";
        }

        offset = (offset < 0) ? 0 : offset;
        int iCount = img.getLength();
        if (count == 0 || count + offset > iCount) {
            count = iCount - offset;
        }
        if (count <= 0) {
            return null;
        }

        StringBuilder text = new StringBuilder();
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.position(4 * offset);
        for (int j = 0; j < count; j++) {
            if ((j & 0x03) == 0) {
                if (j != 0) {
                    text.append(NL);
                }
                text.append(String.format("%08x:", offset + j));
            }
            text.append(String.format(" %08x", buff.getInt()));
        }

        return text.toString();
    }


   /**
    *  Checks that a REB is connected.
    */
    private String checkConnect()
    {
        return (imc != null) ? null : "No REB connected";
    }


   /**
    *  Checks that the listener is not active.
    */
    private String checkListen()
    {
        return !listening ? null : "Listener is active";
    }

}
