package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 ********************************
 *
 *  ASIC ADC reading routines.
 *
 *  @author Owen Saxton
 *
 ********************************
 */
public class SlowAdcs {

   /**
    *  Public constants
    */
    public static final int
        VERSION_WREB1     = BaseSet.VERSION_0,
        VERSION_WREB2     = BaseSet.VERSION_2,
        VERSION_REB3      = BaseSet.VERSION_1,
        VERSION_REB4      = BaseSet.VERSION_3,
        VERSION_GREB      = BaseSet.VERSION_4,
        VERSION_REB5      = BaseSet.VERSION_5,

        REG_TEMP_SLCT     = 0x600100,
        REG_TEMP_READ     = 0x600101,
        REG_TEMP_MUX      = 0x600200,
        TEMP_CABAC_TOP    = 0,
        TEMP_CABAC_BOTTOM = 2,
        TEMP_ASPIC_TOP    = 1,
        TEMP_ASPIC_BOTTOM = 3,
        NUM_TEMPS         = 4,
        MUX_T_B_SLOW      = 0,
        MUX_T_FAST_B_SLOW = 1,
        MUX_T_SLOW_B_FAST = 2,
        MUX_T_B_FAST      = 3,
        NUM_MUX_SET       = 4,

        REG_ASPIC_START   = 0x600100,
        REG_ASPIC_READ    = 0x601000,
        REG_MUX_CONFIG    = 0x600101,
        REG_MUX_READ      = 0x601010,
        //NUM_ASPIC_TEMPS   = Asic.NUM_SIDES * Asic.NUM_STRIPS,
        NUM_CURR_CHANS    = 8,

        REG_MON_START     = 0xc00000,
        REG_MON_CONFIG    = 0xc00001,
        REG_MON_READ      = 0xc00010,
        OFF_MON_ADCS      = 0x100,
        //NUM_MON_ADCS      = 2,
        OFF_MON_OD        = 0,
        OFF_MON_GD        = 1,
        OFF_MON_OG        = 2,
        OFF_MON_RD        = 3,
        OFF_MON_CKP       = 4,
        OFF_MON_CKS       = 5,
        OFF_MON_RG        = 6,
        OFF_MON_VREF25    = 7,
        OFF_MON_ODI_G     = 7,
        OFF_MON_SCK_L     = 0,
        OFF_MON_SCK_U     = 1,
        OFF_MON_RG_L      = 2,
        OFF_MON_RG_U      = 3,
        OFF_MON_CKP_SH    = 4,
        OFF_MON_V_DPHI    = 4,
        OFF_MON_ODI_W     = 5,
        OFF_MON_CK_SH     = 5,
        OFF_MON_V_CLKL    = 6,

        CHAN_OD_0         = 0,
        CHAN_OG_0         = 1,
        CHAN_RD_0         = 2,
        CHAN_GD_0         = 3,
        CHAN_OD_1         = 4,
        CHAN_OG_1         = 5,
        CHAN_RD_1         = 6,
        CHAN_GD_1         = 7,
        CHAN_OD_2         = 8,
        CHAN_OG_2         = 9,
        CHAN_RD_2         = 10,
        CHAN_GD_2         = 11,
        CHAN_VREF5_0      = 12,
        CHAN_VREF5_1      = 13,
        CHAN_VREF5_2      = 14,
        CHAN_VREF25_1     = 15,
        CHAN_OD           = CHAN_OD_0,
        CHAN_OG           = CHAN_OG_0,
        CHAN_RD           = CHAN_RD_0,
        CHAN_GD           = CHAN_GD_0,
        CHAN_VREF25       = 16,
        CHAN_CKP          = 17,
        CHAN_CKS          = 18,
        CHAN_RG           = 19,
        CHAN_SCK_L        = 20,
        CHAN_SCK_U        = 21,
        CHAN_RG_L         = 22,
        CHAN_RG_U         = 23,
        CHAN_CKP_SH       = 24,
        CHAN_ODI          = 25,
        CHAN_CKP_L        = 26,
        CHAN_CKP_U        = 27,
        CHAN_VP12         = 28,
        CHAN_VN12         = 29,
        CHAN_V_DPHI       = 30,
        CHAN_V_CLKL       = 31,
        CHAN_CKP_0        = CHAN_CKP,
        CHAN_CKP_1        = 32,
        CHAN_CKS_0        = CHAN_CKS,
        CHAN_CKS_1        = 33,
        CHAN_RG_0         = CHAN_RG,
        CHAN_RG_1         = 34,
        CHAN_ODI_0        = CHAN_ODI,
        CHAN_ODI_1        = 35,

        RANGE_PM10        = 1,
        RANGE_PM5         = 2,
        RANGE_PM2_5       = 3,
        RANGE_0_10        = 5,
        RANGE_0_5         = 6,

        ADC_0             = 0,
        ADC_1             = 1,
        ADC_2             = 2,

        SAM_0             = 0,
        SAM_1             = 1,
        SAM_2             = 2,
        SAM_3             = 3,
        SAM_4             = 4,
        SAM_5             = 5,
        SAM_6             = 6,
        SAM_7             = 7,

        BIAS_0            = 0,
        BIAS_1            = 1,
        BIAS_2            = 2,
        BIAS_3            = 3,
        BIAS_4            = 4,
        BIAS_5            = 5,
        BIAS_6            = 6,
        BIAS_7            = 7;

    public static final double
        VOLT_16_SCALE_WR   = 1.5 / 65536,
        VOLT_16_SCALE_GR   = 8.192 / 65536,
        VOLT_12_SCALE      = 5.0 / 4096,
        VOLT_12_SCALE_HI   = VOLT_12_SCALE * 11,
        VOLT_12_SCALE_R4   = 10.0 * 2.0 / 4095, //+-5V ADC input range & 5V ref instead of 2.5V
        VOLT_12_SCALE_M4   = VOLT_12_SCALE_R4 * 3,
        VOLT_12_SCALE_H4   = VOLT_12_SCALE_R4 * 11,
        VOLT_12_SCALE_R5   = 10.0 / 4095, //+-5V ADC input range
        VOLT_12_SCALE_M5   = VOLT_12_SCALE_R5 * 3,
        VOLT_12_SCALE_H5   = VOLT_12_SCALE_R5 * 11,
        VOLT_16_SCALE      = 10.0 / 65536,
        VOLT_16_SCALE_MD   = VOLT_16_SCALE * 5,
        VOLT_16_SCALE_HI   = VOLT_16_SCALE * 11,
        VOLT_16_SCALE_AG   = VOLT_16_SCALE * 7.024,
        CURR_16_SCALE_OD   = VOLT_16_SCALE / 60,
        CURR_12_SCALE      = VOLT_12_SCALE / 1000,
        CURR_12_SCALE_R4   = CURR_12_SCALE * 2,
        TEMP_OFFSET        = 2.477 / 0.0056,     // From Claire Juramy
        TEMP_CONVN         = -1.0 / 0.0056,      //         "
        TEMP_16_SCALE_WR   = VOLT_16_SCALE_WR * TEMP_CONVN, // * 2,
        TEMP_16_SCALE_GR   = VOLT_16_SCALE_GR * TEMP_CONVN,
        TEMP_12_SCALE      = VOLT_12_SCALE * TEMP_CONVN,
        TEMP_12_SCALE_R4   = TEMP_12_SCALE * 2;

    private static final int[]
        voltsCfgReb3 = {0x3c, 0x5c, 0x7c, 0x1c, 0x9c, 0xdc, 0xfc, 0x1d,
                        0xbc, 0x3d, 0x5d, 0x9d, 0xbd, 0xdd, 0x7d, 0xfd};

    private static final double[]
        voltsScaleReb3 = {VOLT_12_SCALE_HI, VOLT_12_SCALE_HI, VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE,    VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE,    VOLT_12_SCALE,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE_HI, VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE};

    private static final Map<Integer, Integer> chanMapReb3 = new HashMap<>();
    static {
        chanMapReb3.put(CHAN_OD_0,     0);
        chanMapReb3.put(CHAN_OG_0,     1);
        chanMapReb3.put(CHAN_RD_0,     2);
        chanMapReb3.put(CHAN_GD_0,     3);
        chanMapReb3.put(CHAN_VREF5_0,  4);
        chanMapReb3.put(CHAN_OD_1,     5);
        chanMapReb3.put(CHAN_OG_1,     6);
        chanMapReb3.put(CHAN_RD_1,     7);
        chanMapReb3.put(CHAN_GD_1,     8);
        chanMapReb3.put(CHAN_VREF5_1,  9);
        chanMapReb3.put(CHAN_VREF25_1, 10);
        chanMapReb3.put(CHAN_OD_2,     11);
        chanMapReb3.put(CHAN_OG_2,     12);
        chanMapReb3.put(CHAN_RD_2,     13);
        chanMapReb3.put(CHAN_GD_2,     14);
        chanMapReb3.put(CHAN_VREF5_2,  15);
    }

    private static final int[][]
        voltsCfgReb4 = {{BIAS_1, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_2, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_3, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_0, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_4, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_6, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_7, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_0, SAM_5, ADC_1, RANGE_PM5},
                        {BIAS_5, SAM_4, ADC_1, RANGE_PM5},
                        {BIAS_1, SAM_5, ADC_1, RANGE_PM5},
                        {BIAS_2, SAM_5, ADC_1, RANGE_PM5},
                        {BIAS_4, SAM_5, ADC_1, RANGE_PM5},
                        {BIAS_5, SAM_5, ADC_1, RANGE_PM5},
                        {BIAS_6, SAM_5, ADC_1, RANGE_PM5},
                        {BIAS_3, SAM_5, ADC_1, RANGE_PM5},
                        {BIAS_7, SAM_5, ADC_1, RANGE_PM5},
                        {0, SAM_0, ADC_2, RANGE_PM5},
                        {0, SAM_1, ADC_2, RANGE_PM5},
                        {0, SAM_2, ADC_2, RANGE_PM5},
                        {0, SAM_3, ADC_2, RANGE_PM5},
                        {0, SAM_4, ADC_2, RANGE_PM5},
                        {0, SAM_5, ADC_2, RANGE_PM5},
                        {0, SAM_6, ADC_2, RANGE_PM5},
                        {0, SAM_7, ADC_2, RANGE_PM5}};

    private static final double[]
        voltsScaleReb4 = {VOLT_12_SCALE_H4, VOLT_12_SCALE_M4, VOLT_12_SCALE_H4,
                          VOLT_12_SCALE_H4, VOLT_12_SCALE_R4, VOLT_12_SCALE_H4,
                          VOLT_12_SCALE_M4, VOLT_12_SCALE_H4, VOLT_12_SCALE_H4,
                          VOLT_12_SCALE_R4, VOLT_12_SCALE_R4, VOLT_12_SCALE_H4,
                          VOLT_12_SCALE_M4, VOLT_12_SCALE_H4, VOLT_12_SCALE_H4,
                          VOLT_12_SCALE_R4, VOLT_12_SCALE_M4, VOLT_12_SCALE_M4,
                          VOLT_12_SCALE_M4, VOLT_12_SCALE_M4, VOLT_12_SCALE_M4,
                          VOLT_12_SCALE_M4, VOLT_12_SCALE_M4, VOLT_12_SCALE_M4};

    private static final Map<Integer, Integer> chanMapReb4 = new HashMap<>();
    static {
        chanMapReb4.put(CHAN_OD_0,     0);
        chanMapReb4.put(CHAN_OG_0,     1);
        chanMapReb4.put(CHAN_RD_0,     2);
        chanMapReb4.put(CHAN_GD_0,     3);
        chanMapReb4.put(CHAN_VREF5_0,  4);
        chanMapReb4.put(CHAN_OD_1,     5);
        chanMapReb4.put(CHAN_OG_1,     6);
        chanMapReb4.put(CHAN_RD_1,     7);
        chanMapReb4.put(CHAN_GD_1,     8);
        chanMapReb4.put(CHAN_VREF5_1,  9);
        chanMapReb4.put(CHAN_VREF25_1, 10);
        chanMapReb4.put(CHAN_OD_2,     11);
        chanMapReb4.put(CHAN_OG_2,     12);
        chanMapReb4.put(CHAN_RD_2,     13);
        chanMapReb4.put(CHAN_GD_2,     14);
        chanMapReb4.put(CHAN_VREF5_2,  15);
        chanMapReb4.put(CHAN_CKP_U,    16);
        chanMapReb4.put(CHAN_CKP_L,    17);
        chanMapReb4.put(CHAN_SCK_U,    18);
        chanMapReb4.put(CHAN_SCK_L,    19);
        chanMapReb4.put(CHAN_RG_U,     20);
        chanMapReb4.put(CHAN_RG_L,     21);
        chanMapReb4.put(CHAN_VP12,     22);
        chanMapReb4.put(CHAN_VN12,     23);
    }

    private static final double[]
        voltsScaleReb5 = {VOLT_12_SCALE_H5, VOLT_12_SCALE_M5, VOLT_12_SCALE_H5,
                          VOLT_12_SCALE_H5, VOLT_12_SCALE_R5, VOLT_12_SCALE_H5,
                          VOLT_12_SCALE_M5, VOLT_12_SCALE_H5, VOLT_12_SCALE_H5,
                          VOLT_12_SCALE_R5, VOLT_12_SCALE_R5, VOLT_12_SCALE_H5,
                          VOLT_12_SCALE_M5, VOLT_12_SCALE_H5, VOLT_12_SCALE_H5,
                          VOLT_12_SCALE_R5, VOLT_12_SCALE_M5, VOLT_12_SCALE_M5,
                          VOLT_12_SCALE_M5, VOLT_12_SCALE_M5, VOLT_12_SCALE_M5,
                          VOLT_12_SCALE_M5, VOLT_12_SCALE_M5, VOLT_12_SCALE_M5};

    private static final double[][]
        voltsScaleWreb2 = {{VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                            VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                            VOLT_16_SCALE_HI, VOLT_16_SCALE},
                           {VOLT_16_SCALE_MD, VOLT_16_SCALE_MD, VOLT_16_SCALE_MD,
                            VOLT_16_SCALE_MD, VOLT_16_SCALE_MD, CURR_16_SCALE_OD}};

    private static final Map<Integer, Integer> chanMapWreb2 = new HashMap<>();
    static {
        chanMapWreb2.put(CHAN_OD,     OFF_MON_OD);
        chanMapWreb2.put(CHAN_GD,     OFF_MON_GD);
        chanMapWreb2.put(CHAN_OG,     OFF_MON_OG);
        chanMapWreb2.put(CHAN_RD,     OFF_MON_RD);
        chanMapWreb2.put(CHAN_CKP,    OFF_MON_CKP);
        chanMapWreb2.put(CHAN_CKS,    OFF_MON_CKS);
        chanMapWreb2.put(CHAN_RG,     OFF_MON_RG);
        chanMapWreb2.put(CHAN_VREF25, OFF_MON_VREF25);
        chanMapWreb2.put(CHAN_SCK_L,  0x10 | OFF_MON_SCK_L);
        chanMapWreb2.put(CHAN_SCK_U,  0x10 | OFF_MON_SCK_U);
        chanMapWreb2.put(CHAN_RG_L,   0x10 | OFF_MON_RG_L);
        chanMapWreb2.put(CHAN_RG_U,   0x10 | OFF_MON_RG_U);
        chanMapWreb2.put(CHAN_CKP_SH, 0x10 | OFF_MON_CKP_SH);
        chanMapWreb2.put(CHAN_ODI,    0x10 | OFF_MON_ODI_W);
    }

    private static final double[][]
        voltsScaleGreb = {{VOLT_16_SCALE_AG, VOLT_16_SCALE_AG, VOLT_16_SCALE_AG,
                           VOLT_16_SCALE_AG, VOLT_16_SCALE_AG, VOLT_16_SCALE_AG,
                           VOLT_16_SCALE_AG},
                          {VOLT_16_SCALE_AG, VOLT_16_SCALE_AG, VOLT_16_SCALE_AG,
                           VOLT_16_SCALE_AG, VOLT_16_SCALE_AG, VOLT_16_SCALE_AG,
                           VOLT_16_SCALE_AG, CURR_16_SCALE_OD},
                          {VOLT_16_SCALE_AG, VOLT_16_SCALE_AG, VOLT_16_SCALE_AG,
                           VOLT_16_SCALE_AG, VOLT_16_SCALE_AG, VOLT_16_SCALE_AG,
                           VOLT_16_SCALE_AG, CURR_16_SCALE_OD},
};

    private static final Map<Integer, Integer> chanMapGreb = new HashMap<>();
    static {
        chanMapGreb.put(CHAN_SCK_L,  OFF_MON_SCK_L);
        chanMapGreb.put(CHAN_SCK_U,  OFF_MON_SCK_U);
        chanMapGreb.put(CHAN_RG_L,   OFF_MON_RG_L);
        chanMapGreb.put(CHAN_RG_U,   OFF_MON_RG_U);
        chanMapGreb.put(CHAN_CKP_SH, OFF_MON_CK_SH);
        chanMapGreb.put(CHAN_V_DPHI, OFF_MON_V_DPHI);
        chanMapGreb.put(CHAN_V_CLKL, OFF_MON_V_CLKL);
        chanMapGreb.put(CHAN_OD_0,   0x10 | OFF_MON_OD);
        chanMapGreb.put(CHAN_GD_0,   0x10 | OFF_MON_GD);
        chanMapGreb.put(CHAN_OG_0,   0x10 | OFF_MON_OG);
        chanMapGreb.put(CHAN_RD_0,   0x10 | OFF_MON_RD);
        chanMapGreb.put(CHAN_CKP_0,  0x10 | OFF_MON_CKP);
        chanMapGreb.put(CHAN_CKS_0,  0x10 | OFF_MON_CKS);
        chanMapGreb.put(CHAN_RG_0,   0x10 | OFF_MON_RG);
        chanMapGreb.put(CHAN_ODI_0,  0x10 | OFF_MON_ODI_G);
        chanMapGreb.put(CHAN_OD_1,   0x20 | OFF_MON_OD);
        chanMapGreb.put(CHAN_GD_1,   0x20 | OFF_MON_GD);
        chanMapGreb.put(CHAN_OG_1,   0x20 | OFF_MON_OG);
        chanMapGreb.put(CHAN_RD_1,   0x20 | OFF_MON_RD);
        chanMapGreb.put(CHAN_CKP_1,  0x20 | OFF_MON_CKP);
        chanMapGreb.put(CHAN_CKS_1,  0x20 | OFF_MON_CKS);
        chanMapGreb.put(CHAN_RG_1,   0x20 | OFF_MON_RG);
        chanMapGreb.put(CHAN_ODI_1,  0x20 | OFF_MON_ODI_G);
    }

    private static final Map<Integer, Map> chanMapMap = new HashMap<>();
    static {
        chanMapMap.put(VERSION_REB3,  chanMapReb3);
        chanMapMap.put(VERSION_REB4,  chanMapReb4);
        chanMapMap.put(VERSION_REB5,  chanMapReb4);
        chanMapMap.put(VERSION_WREB2, chanMapWreb2);
        chanMapMap.put(VERSION_GREB,  chanMapGreb);
    }

    private static final Map<Integer, double[][]> voltsScaleMap = new HashMap<>();
    static {
        voltsScaleMap.put(VERSION_WREB2, voltsScaleWreb2);
        voltsScaleMap.put(VERSION_GREB,  voltsScaleGreb);
    }

    long aspicTime;
    BaseSet bss;


   /**
    *  Constructor.
    *
    *  @param  bss  The associated base set object
    */
    public SlowAdcs(BaseSet bss)
    {
        this.bss = bss;
    }


   /**
    *  Reads the REB3/4/5 slow ADC directly.
    *
    *  @param  config  The ADC/MUX configuration value to use
    *
    *  @return  The ADC value (channel + counts)
    *
    *  @throws  REBException 
    */
    public synchronized int readDirect(int config) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4, VERSION_REB5);
        bss.write(REG_MUX_CONFIG, config);
        return bss.read(REG_MUX_READ) & 0xffff;
    }


   /**
    *  Reads a range of REB3/4, WREB1/2 or GREB ASPIC temperature values.
    *
    *  @param  first  The number of the first temperature to read
    *
    *  @param  count  The number of temperatures to read
    *
    *  @return  The array of values (Celsius)
    *
    *  @throws  REBException 
    */
    public synchronized double[] readAspicTemps(int first, int count) throws REBException
    {
        int version = bss.getVersion(BaseSet.OPTN_SLOW_ADCS);
        if (first < 0 || count < 0 || first + count > getNumAspicTemps()) {
            throw new REBException("Invalid ASPIC temperature range");
        }
        double[] data = new double[count];
        if (version == VERSION_WREB1 || version == VERSION_WREB2) {
            for (int j = 0; j < count; j++) {
                data[j] = readWrebAspicTemp(first + j);
            }
        }
        else if (version == VERSION_GREB) {
            for (int j = 0; j < count; j++) {
                int index = first + j;
                data[j] = readGrebAspicTemp(index / 2, index & 1);
            }
        }
        else {
            double scale = (version == VERSION_REB4) ? TEMP_12_SCALE_R4 : TEMP_12_SCALE;
            bss.write(REG_ASPIC_START, 0);
            for (int j = 0; j < count; j++) {
                int value = bss.read(REG_ASPIC_READ + first + j);
                data[j] = TEMP_OFFSET + scale * (value & 0xfff);
            }
        }
        return data;
    }


   /**
    *  Reads a REB3/4, WREB2 or GREB ASPIC temperature directly.
    *
    *  @param  strip  The strip number
    *
    *  @param  side   The side: 0 (top) or 1 (bottom)
    *
    *  @return  The temperature
    *
    *  @throws  REBException 
    */
    public synchronized double readAspicTemp(int strip, int side) throws REBException
    {
        int version = bss.getVersion(BaseSet.OPTN_SLOW_ADCS);
        checkStrip(strip, version);
        checkSide(side);
        if (version == VERSION_WREB1 || version == VERSION_WREB2) {
            return readWrebAspicTemp(side);
        }
        else if (version == VERSION_GREB) {
            return readGrebAspicTemp(strip, side);
        }
        else {
            double scale;
            if (version == VERSION_REB3) {
                bss.write(REG_MUX_CONFIG, 4 * strip + side + 2); // Improve this!
                scale = TEMP_12_SCALE;
            }
            else {
                int samChan = 4 * strip + side + 2;
                setMuxReb4(0, samChan & 7, samChan >> 3, RANGE_0_5);
                scale = (version == VERSION_REB4) ? TEMP_12_SCALE_R4 : TEMP_12_SCALE;
            }
            return TEMP_OFFSET + (bss.read(REG_MUX_READ) & 0xfff) * scale;
        }
    }


   /**
    *  Reads a WREB ASPIC temperature.
    *
    *  @param  side   The side: 0 (top) or 1 (bottom)
    *
    *  @return  The temperature
    *
    *  @throws  REBException 
    */
    private double readWrebAspicTemp(int side) throws REBException
    {
        long wait = aspicTime + 150 - System.currentTimeMillis();
        if (wait > 0) {
            try { Thread.sleep(wait); }
            catch (InterruptedException e) {}
        }
        bss.write(REG_TEMP_SLCT, TEMP_ASPIC_TOP + 2 * side);
        try { Thread.sleep(150); }
        catch (InterruptedException e) {}
        aspicTime = System.currentTimeMillis();
        int value = ((bss.read(REG_TEMP_READ) ^ 0x00800000) << 8) >> 14;
        //System.out.println("ASPIC temp: write time = " + (time1 - time0)
        //                     + " ms, read time = "
        //                     + (System.currentTimeMillis() - time1) + " ms");
        //return TEMP_OFFSET + value * TEMP_16_SCALE;
        return TEMP_OFFSET + (value + 0x20000) * TEMP_16_SCALE_WR;
    }


   /**
    *  Reads a GREB ASPIC temperature.
    *
    *  @param  strip  The strip number
    *
    *  @param  side   The side: 0 (top) or 1 (bottom)
    *
    *  @return  The temperature
    *
    *  @throws  REBException 
    */
    private double readGrebAspicTemp(int strip, int side) throws REBException
    {
        bss.write(REG_TEMP_SLCT, strip);
        try { Thread.sleep(10); }
        catch (InterruptedException e) {}
        //int[] values = new int[4];
        //bss.read(REG_TEMP_READ, values);
        //System.out.format("Strip=%s, side=%s, values=%08x,%08x,%08x,%08x\n",
        //                  strip, side, values[0], values[1], values[2], values[3]);
        //return TEMP_OFFSET + (values[side] >> 16) * TEMP_16_SCALE_GR;
        int value = bss.read(REG_TEMP_READ + side);
        //System.out.format("Strip=%s, side=%s, value=%08x\n", strip, side, value);
        return TEMP_OFFSET + (value >> 16) * TEMP_16_SCALE_GR;
    }


   /**
    *  Reads a REB3 bias or WREB2/GREB/REB4 clock/bias voltage.
    *
    *  This assumes that, for the WREB, the ADCs have already been read into
    *  REB registers.
    *
    *  @param  chan   The overall channel number
    *
    *  @return  The voltage
    *
    *  @throws  REBException 
    */
    public synchronized double readVoltage(int chan) throws REBException
    {
        int version = bss.getVersion(BaseSet.OPTN_SLOW_ADCS);
        Map chanMap = chanMapMap.get(version);
        if (chanMap == null) {
            throw new REBException("Unsupported firmware version");
        }
        Integer desc = (Integer)chanMap.get(chan);
        if (desc == null) {
            throw new REBException(("Invalid voltage channel: " + chan));
        }
        if (version == VERSION_REB3) {
            bss.write(REG_MUX_CONFIG, voltsCfgReb3[desc]);
            return (bss.read(REG_MUX_READ) & 0xfff) * voltsScaleReb3[desc];
        }
        else if (version == VERSION_REB4 || version == VERSION_REB5) {
            int[] cfg = voltsCfgReb4[desc];
            setMuxReb4(cfg[0], cfg[1], cfg[2], cfg[3]);
            int offset = (cfg[3] >= RANGE_0_10) ? 0 : 0x800;
            double[] voltsScale = (version == VERSION_REB4) ? voltsScaleReb4 : voltsScaleReb5;
            return ((bss.read(REG_MUX_READ) & 0xfff) - offset) * voltsScale[desc];
        }
        else {
            int adc = desc >> 4, ch = desc & 0x0f;
            int rawValue = bss.read(REG_MON_READ + adc * OFF_MON_ADCS + ch) & 0xffff;
            return (rawValue - 0x8000) * voltsScaleMap.get(version)[adc][ch];
        }
    }


   /**
    *  Reads a REB3 bias or WREB2/GREB/REB4 clock/bias voltage.
    *
    *  This fetches the ADC data into REB registers first if necessary.
    *
    *  @param  chan   The overall channel number
    *
    *  @return  The voltage
    *
    *  @throws  REBException 
    */
    public synchronized double readVoltageNow(int chan) throws REBException
    {
        int version = bss.getVersion(BaseSet.OPTN_SLOW_ADCS);
        if (version == VERSION_WREB2 || version == VERSION_GREB) {
            Integer desc = (Integer)chanMapMap.get(version).get(chan);
            if (desc == null) {
                throw new REBException(("Invalid voltage channel: " + chan));
            }
            fetchVoltages(1 << (desc >> 4));
        }
        return readVoltage(chan);
    }


   /**
    *  Reads a REB3/4 CCD current.
    *
    *  @param  strip  The strip number
    *
    *  @param  side   The side: 0 (top) or 1 (bottom)
    *
    *  @param  chan   The channel number
    *
    *  @return  The current (amps)
    *
    *  @throws  REBException 
    */
    public synchronized double readCurrent(int strip, int side, int chan) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4, VERSION_REB5);
        int version = bss.getVersion(BaseSet.OPTN_SLOW_ADCS);
        checkStrip(strip, version);
        checkSide(side);
        if (chan < 0 || chan >= NUM_CURR_CHANS) {
            throw new REBException(("Invalid current channel: " + chan));
        }
        double scale;
        if (version == VERSION_REB3) {
            bss.write(REG_MUX_CONFIG, (4 * strip + side) | (chan << 5) | 0x10);
            scale = CURR_12_SCALE;
        }
        else {
            int samChan = 4 * strip + side;
            setMuxReb4(chan, samChan & 7, samChan >> 3, RANGE_0_5);
            scale = (version == VERSION_REB4) ? CURR_12_SCALE_R4 : CURR_12_SCALE;
        }
        return (bss.read(REG_MUX_READ) & 0xfff) * scale;
    }


   /**
    *  Reads a REB3/4 CCD current.
    *
    *  @param  chan   The overall channel number: 16*strip + 8*side + chan
    *
    *  @return  The current (amps)
    *
    *  @throws  REBException 
    */
    public double readCurrent(int chan) throws REBException
    {
        return readCurrent(chan >> 4, (chan >> 3) & 1, chan & 7);
    }


   /**
    *  Fetches the WREB2/GREB monitoring ADC values.
    *
    *  This causes the values from the specified ADCs to be read into
    *  registers to be retrieved by subsequent readVoltage calls
    *
    *  @param  mask  The mask of ADC numbers to be fetched
    *
    *  @throws  REBException 
    */
    public void fetchVoltages(int mask) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SLOW_ADCS, VERSION_WREB2, VERSION_GREB);
        if (bss.getVersion(BaseSet.OPTN_SLOW_ADCS) == VERSION_WREB2) {
            for (int j = 0; j <= bss.getNumStrips(); j++) {
                if ((mask & (1 << j)) != 0) {
                    bss.write(REG_MON_START + j * OFF_MON_ADCS, 0);
                }
            }
        }
        else {
            bss.write(REG_MON_START, 0);
        }
    }


   /**
    *  Fetches the WREB2/GREB monitoring ADC values.
    *
    *  This causes the values from all ADCs to be read into registers to be
    *  retrieved by subsequent readVoltage calls
    *
    *  @throws  REBException 
    */
    public void fetchVoltages() throws REBException
    {
        fetchVoltages((1 << (bss.getNumStrips() + 1)) - 1);
    }


   /**
    *  Configures a WREB2/GREB monitoring ADC.
    *
    *  @param  adc    The ADC number (0 or 1)
    *
    *  @param  value  The value to be sent to the ADC
    *
    *  @throws  REBException 
    */
    public void configure(int adc, int value) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SLOW_ADCS, VERSION_WREB2, VERSION_GREB);
        checkAdc(adc);
        if (bss.getVersion(BaseSet.OPTN_SLOW_ADCS) == VERSION_WREB2) {
            bss.write(REG_MON_CONFIG + adc * OFF_MON_ADCS, value & 0x0f);
        }
        else {
            bss.write(REG_MON_CONFIG, (adc << 4) | (value & 0x0f));
        }
    }


   /**
    *  Reads the WREB1/2 slow ADC.
    *
    *  @param  chan  The number of the channel to read
    *
    *  @return  The converted ADC value (volts)
    *
    *  @throws  REBException 
    */
    public synchronized double readSlow(int chan) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SLOW_ADCS, VERSION_WREB1, VERSION_WREB2);
        if (chan < 0 || chan >= NUM_TEMPS) {
            throw new REBException("Invalid slow ADC channel number");
        }
        bss.write(REG_TEMP_SLCT, chan);
        int value = ((bss.read(REG_TEMP_READ) ^ 0x00800000) << 8) >> 14;

        return VOLT_16_SCALE_WR * value;
    }


   /**
    *  Sets the WREB1 ASIC MUX.
    *
    *  @param  set  The value to set
    *
    *  @throws  REBException 
    */
    public void setMux(int set) throws REBException
    {
        bss.checkVersion(BaseSet.OPTN_SLOW_ADCS, VERSION_WREB1);
        if (set < 0 || set >= NUM_MUX_SET) {
            throw new REBException("Invalid MUX setting value: " + set);
        }
        bss.write(REG_TEMP_MUX, set);
    }


   /**
    *  Gets the number of ASPIC temperatures.
    *
    *  @return  The number of ASPIC temperatures
    *
    *  @throws  REBException 
    */
    public int getNumAspicTemps() throws REBException
    {
        return Asic.NUM_SIDES * bss.getNumStrips();
    }


   /**
    *  Gets the number of CCD currents.
    *
    *  @return  The number of CCD currents
    *
    *  @throws  REBException 
    */
    public int getNumCcdCurrents() throws REBException
    {
        return bss.isVersion(BaseSet.OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4, VERSION_REB5)
                 ? Asic.NUM_SIDES * NUM_CURR_CHANS * bss.getNumStrips(): 0;
    }


   /**
    *  Tests a channel number for validity.
    *
    *  @param  chan  The channel number
    *
    *  @return  Whether channel number is valid
    */
    public boolean testChannel(int chan)
    {
        try {
            Map chanMap = chanMapMap.get(bss.getVersion(BaseSet.OPTN_SLOW_ADCS));
            return chanMap != null && chanMap.get(chan) != null;
        }
        catch (REBException e) {
            return false;
        }
    }


   /**
    *  Sets the REB4 MUX value.
    *
    *  @param  bias   The bias MUX channel
    *
    *  @param  sam    The SAM MUX channel
    *
    *  @param  adc    The ADC channel
    *
    *  @param  range  The encoded range
    *
    *  @throws  REBException 
    */
    private void setMuxReb4(int bias, int sam, int adc, int range)
        throws REBException
    {
        bss.write(REG_MUX_CONFIG, (sam << 19) | (bias << 16) | (4 << 9) | (adc << 5) | (range << 1));
    }


   /**
    *  Checks a strip number.
    *
    *  @param  strip    The strip number
    *
    *  @param  version  The firmware version number
    *
    *  @throws  REBException 
    */
    private void checkStrip(int strip, int version) throws REBException
    {
        if (strip < 0 || strip >= bss.getNumStrips()) {
            throw new REBException("Invalid strip number: " + strip);
        }
    }


   /**
    *  Checks a side number.
    *
    *  @param  side  The side number
    *
    *  @throws  REBException 
    */
    private static void checkSide(int side) throws REBException
    {
        if (side < 0 || side >= Asic.NUM_SIDES) {
            throw new REBException(("Invalid board side: " + side));
        }
    }


   /**
    *  Checks an ADC number.
    *
    *  @param  adc  The ADC number
    *
    *  @throws  REBException 
    */
    private void checkAdc(int adc) throws REBException
    {
        if (adc < 0 || adc > bss.getNumStrips()) {
            throw new REBException(("Invalid ADC number: " + adc));
        }
    }

}
