package org.lsst.ccs.drivers.reb.sim;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.drivers.reb.Sequencer;
import static org.lsst.ccs.drivers.reb.Sequencer.MAX_SUBR_LEVEL;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_MAX_FUNC;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_MAX_PARAMETER;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_MAX_PROGRAM;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_MAX_SLICE;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_END_PROG;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_END_SUBR;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_EXECUTE;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_EXECUTE_FP;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_EXECUTE_FRP;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_EXECUTE_RP;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_JUMP;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_JUMP_AP;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_JUMP_ARP;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_OPC_JUMP_RP;
import static org.lsst.ccs.drivers.reb.Sequencer.SEQ_PRG_V_OPCODE;
import org.lsst.ccs.drivers.reb.sim.RegisterSet.ReadOnlyRegister;
import org.lsst.ccs.drivers.reb.sim.RegisterSet.Register;
import org.lsst.ccs.drivers.reb.sim.RegisterSet.WriteOnlyRegister;

/**
 * Simulate the REB3 sequencer. The simulation allows programs to be loaded and
 * run. Support is included for start, step and stop functions. Listeners can be
 * attached to listen for start/stop transitions, function calls, and to listen
 * to output transitions (waveforms). Most of the functionality of the sequencer
 * is controlled by manipulating its registers using the AddressSpace object
 * passed to the constructor.
 *
 * @author tonyj
 * @see
 * <a href="https://confluence.slac.stanford.edu/display/LSSTCAM/REB+3+registers+sets">Sequencer
 * User Guide</a>
 */
public class SequencerSimulation {

    private final int[] lines = new int[SEQ_MAX_FUNC * SEQ_MAX_SLICE];
    private final int[] times = new int[SEQ_MAX_FUNC * SEQ_MAX_SLICE];
    private final int[] program = new int[SEQ_MAX_PROGRAM];
    private final int[] execRefs = new int[SEQ_MAX_PARAMETER];
    private final int[] execReps = new int[SEQ_MAX_PARAMETER];
    private final int[] jumpRefs = new int[SEQ_MAX_PARAMETER];
    private final int[] jumpReps = new int[SEQ_MAX_PARAMETER];
    private int programCounter;
    private int stackDepth;
    private int startAddress;
    private int errorFlag;
    private int stripes = 7;
    private int output = 0;
    private boolean patternMode = false;
    private boolean scanMode = false;
    // Special simulation features
    private int maxSubroutineRepetitions = 0;
    private int maxFunctionRepetitions = 0;
    private int renormalization = 0;

    public enum State {
        STOPPED(false), RUNNING(true), LOOPING(true), ERROR(false);
        private final boolean running;

        State(boolean running) {
            this.running = running;
        }

        public boolean isRunning() {
            return running;
        }
    };

    private enum LoopExit {
        STEP, STOP
    }
    private volatile State state = State.STOPPED;
    private final List<StateListener> stateListeners = new CopyOnWriteArrayList<>();
    private final List<FunctionListener> functionListeners = new CopyOnWriteArrayList<>();
    private final List<SubroutineListener> subroutineListeners = new CopyOnWriteArrayList<>();
    private final List<WaveformListener> waveformListeners = new CopyOnWriteArrayList<>();
    private final SynchronousQueue<LoopExit> loopSemaphore = new SynchronousQueue();
    private final ExecutorService executor = Executors.newSingleThreadExecutor();

    /**
     * Create a sequencer simulator.
     *
     * @param addressSpace The address space to which the sequencer will add its
     * registers.
     */
    public SequencerSimulation(AddressSpace addressSpace) {
        addressSpace.add(new RegisterSet.ArrayRegisterSet(Sequencer.REG_SEQ_LINES, lines));
        addressSpace.add(new RegisterSet.ArrayRegisterSet(Sequencer.REG_SEQ_TIMES, times));
        addressSpace.add(new RegisterSet.ArrayRegisterSet(Sequencer.REG_SEQ_PROGRAM, program));
        addressSpace.add(new RegisterSet.ArrayRegisterSet(Sequencer.REG_SEQ_EXEC_REFS, execRefs));
        addressSpace.add(new RegisterSet.ArrayRegisterSet(Sequencer.REG_SEQ_EXEC_REPS, execReps));
        addressSpace.add(new RegisterSet.ArrayRegisterSet(Sequencer.REG_SEQ_JUMP_REFS, jumpRefs));
        addressSpace.add(new RegisterSet.ArrayRegisterSet(Sequencer.REG_SEQ_JUMP_REPS, jumpReps));

        addressSpace.add(new StartAddressRegister(Sequencer.REG_SEQ_START_ADDR));
        addressSpace.add(new ErrorRegister(Sequencer.REG_SEQ_ERROR_VALUE));
        addressSpace.add(new ErrorResetRegister(Sequencer.REG_SEQ_ERROR_RESET));
        addressSpace.add(new StripeRegister(Sequencer.REG_SEQ_STRIPE));
        addressSpace.add(new PatternRegister(Sequencer.REG_SEQ_SOURCE));
        addressSpace.add(new ScanRegister(Sequencer.REG_SEQ_SCAN_CNTRL));
        addressSpace.add(new ScanResetRegister(Sequencer.REG_SEQ_SCAN_RESET));
        addressSpace.add(new StopRegister(Sequencer.REG_SEQ_STOP));
        addressSpace.add(new StepRegister(Sequencer.REG_SEQ_STEP));
    }

    private class StartAddressRegister extends Register {

        StartAddressRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            if (startAddress < 0 || startAddress >= program.length) {
                throw new SimulationException("Illegal start address %08x", value);
            }
            startAddress = value;
        }

        @Override
        public int read() {
            return startAddress;
        }

    }

    private class ErrorRegister extends ReadOnlyRegister {

        ErrorRegister(int address) {
            super(address);
        }

        @Override
        public int read() {
            return errorFlag;
        }
    }

    private class ErrorResetRegister extends WriteOnlyRegister {

        ErrorResetRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            errorFlag = 0;
            stateChanged(State.STOPPED);
        }

    }

    private class StopRegister extends WriteOnlyRegister {

        StopRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            try {
                stop();
            } catch (InterruptedException ex) {
                throw new SimulationException("Interupted while stopping", ex);
            }
        }

    }

    private class StepRegister extends WriteOnlyRegister {

        StepRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            try {
                step();
            } catch (InterruptedException ex) {
                throw new SimulationException("Interupted while stepping", ex);
            }
        }

    }

    private class StripeRegister extends Register {

        StripeRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            stripes = value;
        }

        @Override
        public int read() {
            return stripes;
        }

    }

    private class PatternRegister extends Register {

        PatternRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            patternMode = (value & 1) != 0;
        }

        @Override
        public int read() {
            return patternMode ? 1 : 0;
        }

    }

    private class ScanRegister extends Register {

        ScanRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            scanMode = (value & 1) != 0;
        }

        @Override
        public int read() {
            return scanMode ? 1 : 0;
        }

    }

    private class ScanResetRegister extends WriteOnlyRegister {

        ScanResetRegister(int address) {
            super(address);
        }

        @Override
        public void write(int value) {
            //FIXME: Provide implementation
        }

    }

    /**
     * Start the sequencer. This function returns immediately, and execution of
     * the sequencer continues on a separate thread.
     *
     * @return A future which can be used to obtain the final state of the
     * sequencer when it finishes due to reaching the end of a program, or due
     * to an error.
     */
    public Future<FinalState> start() {
        if (state != State.STOPPED) {
            throw new SimulationException("Attempt to start sequencer when sequencer not cleanly stopped");
        }

        Callable<FinalState> callable = () -> {
            try {
                programCounter = startAddress;
                stackDepth = 0;
                stateChanged(State.RUNNING);
                runSequencer();
            } catch (EndOfProgramException x) {
                stateChanged(State.STOPPED);
                return new FinalState(state, programCounter, null);
            } catch (InterruptedException | SimulationException ex) {
                errorFlag = programCounter + 0x10000000;
                stateChanged(State.ERROR);
                return new FinalState(state, programCounter, ex);
            } finally {
                if (output != lines[0]) {
                    waveformListeners.stream().forEach((w) -> {
                        w.transition(output, lines[0], 0);
                    });
                    output = lines[0];
                }
            }
            // This should never happen
            return null;
        };
        return executor.submit(callable);
    }

    private void stop() throws InterruptedException {
        if (state != State.LOOPING) {
            throw new SimulationException("Sequencer received STOP command when not in infinite loop");
        }
        loopSemaphore.put(LoopExit.STOP);
    }

    private void step() throws InterruptedException {
        if (state != State.LOOPING) {
            throw new SimulationException("Sequencer received STEP command when not in infinite loop");
        }
        loopSemaphore.put(LoopExit.STEP);
    }

    /**
     * Shutdown the simulator, stopping any programs currently running.
     */
    public void shutdown() {
        executor.shutdownNow();
    }

    private void stateChanged(State newState) {
        State oldState = state;
        state = newState;
        stateListeners.stream().forEach((l) -> {
            l.stateChanged(oldState, newState);
        });
    }

    /**
     * Get the current state of the simulator.
     *
     * @return
     */
    public State getState() {
        return state;
    }

    private void runSequencer() throws EndOfProgramException, InterruptedException {
        for (;;) {
            if (executor.isShutdown()) {
                throw new InterruptedException("Sequencer is being shutdown");
            }
            int code = program[programCounter];
            int opCode = code >>> SEQ_PRG_V_OPCODE;
            switch (opCode) {
                case SEQ_OPC_EXECUTE:
                    int function = (code >> 24) & 0xf;
                    boolean infiniteLoop = (code >> 23 & 1) != 0;
                    int repetitions = code & 0x7fffff;
                    callFunction(function, infiniteLoop, repetitions);
                    break;
                case SEQ_OPC_EXECUTE_FP:
                    int functionPointer = (code >> 24) & 0xf;
                    function = dereferenceFunctionPointer(functionPointer);
                    infiniteLoop = (code >> 23 & 1) != 0;
                    repetitions = code & 0x7fffff;
                    callFunction(function, infiniteLoop, repetitions);
                    break;
                case SEQ_OPC_EXECUTE_RP:
                    function = (code >> 24) & 0xf;
                    int repetitionsPointer = code & 0xf;
                    infiniteLoop = dereferenceInfiniteLoop(repetitionsPointer);
                    repetitions = dereferenceRepetitions(repetitionsPointer);
                    callFunction(function, infiniteLoop, repetitions);
                    break;
                case SEQ_OPC_EXECUTE_FRP:
                    functionPointer = (code >> 24) & 0xf;
                    repetitionsPointer = code & 0xf;
                    function = dereferenceFunctionPointer(functionPointer);
                    infiniteLoop = dereferenceInfiniteLoop(repetitionsPointer);
                    repetitions = dereferenceRepetitions(repetitionsPointer);
                    callFunction(function, infiniteLoop, repetitions);
                    break;
                case SEQ_OPC_JUMP:
                    int jumpAddress = (code >> 16) & 0x3ff;
                    repetitions = code & 0xffff;
                    callSubroutine(jumpAddress, repetitions);
                    break;
                case SEQ_OPC_JUMP_AP:
                    int jumpPointer = (code >> 16) & 0xf;
                    jumpAddress = dereferenceJumpPointer(jumpPointer);
                    repetitions = code & 0xffff;
                    callSubroutine(jumpAddress, repetitions);
                    break;
                case SEQ_OPC_JUMP_RP:
                    jumpAddress = (code >> 16) & 0x3ff;
                    repetitionsPointer = code & 0xf;
                    repetitions = dereferenceJumpRepetitions(repetitionsPointer);
                    callSubroutine(jumpAddress, repetitions);
                    break;
                case SEQ_OPC_JUMP_ARP:
                    jumpPointer = (code >> 16) & 0xf;
                    repetitionsPointer = code & 0xf;
                    repetitions = dereferenceJumpRepetitions(repetitionsPointer);
                    jumpAddress = dereferenceJumpPointer(jumpPointer);
                    callSubroutine(jumpAddress, repetitions);
                    break;
                case SEQ_OPC_END_SUBR:
                    return;
                case SEQ_OPC_END_PROG:
                    endProgram();
                default:
                    throw new SimulationException("Illegal opcode %08x at address %08x", opCode, programCounter);
            }
            programCounter++;
        }

    }

    private void callFunction(int function, boolean infiniteLoop, int repetitions) throws InterruptedException, EndOfProgramException {
        // Per conversation with Stefano, repetitions==0 is tested before infiniteLoop
        if (repetitions == 0) {
            return;
        }
        functionListeners.stream().forEach((f) -> {
            f.functionCalled(function, infiniteLoop, repetitions);
        });
        // If there are no waveform listeners, then no point actually executing the function
        // since it has no effect.
        if (!waveformListeners.isEmpty()) {
            int actualRepetitions = repetitions;
            if (maxFunctionRepetitions > 0 && actualRepetitions > maxFunctionRepetitions) {
                actualRepetitions = maxFunctionRepetitions;
            }
            if (infiniteLoop) {
                if (renormalization > 0) {
                    actualRepetitions = renormalization;
                } else {
                    actualRepetitions = 0;
                }
            }
            if (infiniteLoop) {
                stateChanged(State.LOOPING);
            }
            for (int i = 0; actualRepetitions == 0 || i < actualRepetitions; i++) {
                if (executor.isShutdown()) {
                    throw new InterruptedException("Sequencer is being shutdown");
                }
                if (infiniteLoop) {
                    LoopExit exit = loopSemaphore.poll();
                    if (exit == LoopExit.STEP) {
                        stateChanged(State.RUNNING);
                        break;
                    } else if (exit == LoopExit.STOP) {
                        throw new EndOfProgramException();
                    }
                }

                for (int slice = function * SEQ_MAX_SLICE; slice < (function + 1) * SEQ_MAX_SLICE; slice++) {
                    int time = times[slice];
                    if (time == 0) {
                        break;
                    }
                    int line = lines[slice];
                    waveformListeners.stream().forEach((w) -> {
                        w.transition(output, line, time * Sequencer.CLOCK_PERIOD_1);
                    });
                    output = line;
                }
            }
        } else if (infiniteLoop) {
            stateChanged(State.LOOPING);
            // Somewhat arbitrary, but if renormalization is set, but no waveformListener, timeout after renormalization microseconds
            LoopExit exit = (renormalization>0) ? loopSemaphore.poll(renormalization,TimeUnit.MICROSECONDS) : loopSemaphore.take();
            if (exit == null || exit == LoopExit.STEP) {
                stateChanged(State.RUNNING);
            } else if (exit == LoopExit.STOP) {
                throw new EndOfProgramException();
            }
        }

    }

    private void callSubroutine(int jumpAddress, int repetitions) throws EndOfProgramException, InterruptedException {
        // We do not actually use the reb3 stack, rather we use the actual java call stack 
        // to implement subroutine calls. It is unclear if there are any downsides to this.
        if (stackDepth >= MAX_SUBR_LEVEL) {
            throw new SimulationException("Maximum stack depth exceeded");
        }
        if (repetitions == 0) {
            return;
        }
        subroutineListeners.stream().forEach((f) -> {
            f.subroutineCalled(jumpAddress, repetitions);
        });
        int actualRepetitions = repetitions;
        if (maxSubroutineRepetitions > 0 && actualRepetitions > maxSubroutineRepetitions) {
            actualRepetitions = maxSubroutineRepetitions;
        }
        stackDepth++;
        int saveProgramCounter = programCounter;
        for (int i = 0; i < actualRepetitions; i++) {
            programCounter = jumpAddress;
            runSequencer();
        }
        stackDepth--;
        subroutineListeners.stream().forEach((f) -> {
            f.subroutineReturned(jumpAddress);
        });
        programCounter = saveProgramCounter;
    }

    private int dereferenceFunctionPointer(int functionPointer) {
        return execRefs[functionPointer];
    }

    private boolean dereferenceInfiniteLoop(int repetitionsPointer) {
        return (execReps[repetitionsPointer] & 0x800000) != 0;
    }

    private int dereferenceRepetitions(int repetitionsPointer) {
        return execReps[repetitionsPointer] & 0x7fffff;
    }

    private int dereferenceJumpPointer(int jumpPointer) {
        return jumpRefs[jumpPointer];
    }

    private int dereferenceJumpRepetitions(int jumpPointer) {
        return jumpReps[jumpPointer];
    }

    private void endProgram() throws EndOfProgramException {
        throw new EndOfProgramException();
    }

    private static class EndOfProgramException extends Exception {

        public EndOfProgramException() {
        }
    }

    /**
     * The current maximum subroutine repetition count.
     *
     * @return The current subroutine repetition limit, or <code>0</code> if
     * unlimited.
     * @see #setMaxSubroutineRepetitions(int)
     */
    public int getMaxSubroutineRepetitions() {
        return maxSubroutineRepetitions;
    }

    /**
     * Allows the maximum repetition count for subroutines to be arbitrarily
     * limited. Any subroutine call which specifies a larger repetition count
     * will actually only repeat this number of times.
     *
     * @param maxSubroutineRepetitions The maximum number of repetitions, or
     * <code>0</code> to remove any limit.
     */
    public void setMaxSubroutineRepetitions(int maxSubroutineRepetitions) {
        this.maxSubroutineRepetitions = maxSubroutineRepetitions;
    }

    /**
     * The current maximum function repetition count.
     *
     * @return The current function repetition limit, or <code>0</code> if
     * unlimited.
     * @see #setMaxFunctionRepetitions(int)
     */
    public int getMaxFunctionRepetitions() {
        return maxFunctionRepetitions;
    }

    /**
     * Allows the maximum repetition count for functions to be arbitrarily
     * limited. Any function call which specifies a larger repetition count will
     * actually only repeat this number of times.
     *
     * @param maxFunctionRepetitions The maximum number of repetitions, or
     * <code>0</code> to remove any limit.
     */
    public void setMaxFunctionRepetitions(int maxFunctionRepetitions) {
        this.maxFunctionRepetitions = maxFunctionRepetitions;
    }

    /**
     * Get the number of times infinite loops will actually repeat.
     *
     * @return The repeat count, or <code>0</code> to indicate unlimited.
     * @see #setRenormalization(int)
     */
    public int getRenormalization() {
        return renormalization;
    }

    /**
     * Limits how many times infinite loops will actually repeat.
     *
     * @param renormalization The number of repetitions used for infinite loops,
     * or <code>0</code> to indicate no limit.
     */
    public void setRenormalization(int renormalization) {
        this.renormalization = renormalization;
    }

    /**
     * Add a state change listener. Note that the state change listener will be
     * called on the sequencer execution thread.
     *
     * @param l The state change listener
     */
    public void addStateListener(StateListener l) {
        stateListeners.add(l);
    }

    /**
     * Remove a state change listener.
     *
     * @param l The state change listener
     */
    public void removeStateListener(StateListener l) {
        stateListeners.remove(l);
    }

    /**
     * Add a subroutine listener. Note that the subroutine listener will be
     * called on the sequencer execution thread.
     *
     * @param l The subroutine listener
     */
    public void addSubroutineListener(SubroutineListener l) {
        subroutineListeners.add(l);
    }

    /**
     * Remove a subroutine listener.
     *
     * @param l The subroutine listener.
     */
    public void removeSubroutineListener(SubroutineListener l) {
        subroutineListeners.remove(l);
    }

    /**
     * Add a function listener. Note that the function listener will be called
     * on the sequencer execution thread.
     *
     * @param l The function listener
     */
    public void addFunctionListener(FunctionListener l) {
        functionListeners.add(l);
    }

    /**
     * Remove a function listener.
     *
     * @param l The function listener.
     */
    public void removeFunctionListener(FunctionListener l) {
        functionListeners.remove(l);
    }

    /**
     * Add a waveform listener to listen to changes of the output waveforms.
     * Note that registering a waveform listener causes the simulator to start
     * executing individual function repetitions, and thus slows the simulator
     * down considerably. Note that infinite loops do not currently result in a
     * unlimited number of calls to the waveform listener.
     *
     * @param l The waveform listener.
     */
    public void addWaveformListener(WaveformListener l) {
        waveformListeners.add(l);
    }

    /**
     * Remove a waveform listener.
     *
     * @param l The waveform listener.
     */
    public void removeWaveformListener(WaveformListener l) {
        waveformListeners.remove(l);
    }

    /**
     * A state listener is called any time the sequencer state changes.
     */
    public interface StateListener {

        /**
         * Called whenever the state of the sequencer changes
         *
         * @param oldState The state prior to the transitions
         * @param newState The state after the transition
         */
        public void stateChanged(State oldState, State newState);
    }

    /**
     * A subroutine listener is called any time a subroutine starts or ends
     *
     */
    public interface SubroutineListener {

        public void subroutineCalled(int address, int repetitions);

        public void subroutineReturned(int address);
    }

    /**
     * A function listener is called any time a function is called.
     */
    public interface FunctionListener {

        /**
         * Called whenever a new function is entered. Note that if
         * repetitions==0 then the function is skipped (even if infiniteLoop is
         * true) so in that case this method is not invoked.
         *
         * @param function The index of the function
         * @param infiniteLoop <code>true</code> if the function will loop until
         * stop/step
         * @param repetitions The number of repetitions of this function
         */
        public void functionCalled(int function, boolean infiniteLoop, int repetitions);
    }

    /**
     * A waveform listener is called any time an output transitions
     */
    public interface WaveformListener {

        /**
         * Called each time a function causes a change in the output waveform.
         *
         * @param oldState The state of the outputs before the transition
         * @param newState The status of the outputs after the transition
         * @param nanos The number of nanoSeconds the outputs will remain in
         * this state.
         */
        public void transition(int oldState, int newState, int nanos);
    }

    /**
     * An object of this class can be obtained from the future returned by the
     * start method. It details the state of the sequencer when execution
     * finished, either as the result of an error, or because an end program
     * instruction us reached.
     */
    public class FinalState {

        private final State stateAtEnd;
        private final Exception exception;
        private final int programCounter;

        private FinalState(State stateAtEnd, int programCounter, Exception exception) {
            this.stateAtEnd = stateAtEnd;
            this.exception = exception;
            this.programCounter = programCounter;
        }

        /**
         * Sequencer state at the end of execution.
         *
         * @return Either ERROR or STOPPED.
         */
        public State getStateAtEnd() {
            return stateAtEnd;
        }

        /**
         * Give any exception which caused sequencer to stop.
         *
         * @return The exception, or <code>null</code> if the program stopped
         * normally.
         */
        public Exception getException() {
            return exception;
        }

        /**
         * Get the program counter at the end of execution.
         *
         * @return The program counter.
         */
        public int getProgramCounter() {
            return programCounter;
        }

    }
}
