package org.lsst.ccs.drivers.reb;


/**
 ******************************************************************************
 **
 **  Routines for reading the RTD temperature sensors.
 **
 **  These are read using an Analog Devices AD7794 chip which requires a
 **  certain amount of setup.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TempRtds extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_RTD_COMMAND = 0x700000,
        REG_RTD_REPLY   = 0x700001,
        REG_RTD_RESET   = 0x700002,
        CMND_READ       = 0x80000,
        NUM_RTD_TEMPS   = 4;
 
    public final static int
        STATUS_REG     = 0,        // Status register (8-bit)
        MODE_REG       = 1,        // Mode register (16-bit)
        CONFIG_REG     = 2,        // Configuration register (16-bit)
        DATA_REG       = 3,        // Data register (24- or 16-bit)
        ID_REG         = 4,        // ID register (8-bit)
        IO_REG         = 5,        // IO register (8-bit)
        OFFSET_REG     = 6,        // Offset register (24- or 16-bit)
        SCALE_REG      = 7,        // (Full-)Scale register (24- or 16-bit)

        COM_READ       = 0x40,     // Comm reg: read operation
        COM_ADDRESS_M  = 0x38,     // Comm reg: address mask
        COM_ADDRESS_V  = 3,        // Comm reg: address offset
        COM_CONT_READ  = 0x04,     // Comm reg: continuous read

        STS_READY      = 0x80,     // Status: (not) ready
        STS_ERROR      = 0x40,     // Status: voltage/temp data ready
        STS_NOXREF     = 0x20,     // Status: no external reference
        STS_AD7794     = 0x08,     // Status: AD7794 chip, not AD7795
        STS_CHANNEL_M  = 0x07,     // Status: channel mask
        STS_CHANNEL_V  = 0,        // Status: channel offset

        MOD_MODE_M     = 0xe000,   // Mode: mode select mask
        MOD_MODE_V     = 13,       // Mode: mode select offset
        MODE_CONT      = 0,        // Mode select: continuous conversion
        MODE_SINGLE    = 1,        // Mode select: single conversion
        MODE_IDLE      = 2,        // Mode select: idle
        MODE_PWR_DOWN  = 3,        // Mode select: power-down
        MODE_INT_Z_CAL = 4,        // Mode select: internal zero calibration
        MODE_INT_F_CAL = 5,        // Mode select: int. full-scale calibration
        MODE_SYS_Z_CAL = 6,        // Mode select: system zero calibration
        MODE_SYS_F_CAL = 7,        // Mode select: sys. full-scale calibration
        MOD_PSW        = 0x1000,   // Mode: power switch
        MOD_AMP_CM     = 0x0200,   // Mode: amplifier common mode
        MOD_CLOCK_M    = 0x00c0,   // Mode: clock source mask
        MOD_CLOCK_V    = 6,        // Mode: clock source offset
        CLOCK_INT      = 0,        // Clock source: internal 64 KHz
        CLOCK_INT_AVL  = 1,        // Clock source: int 64 KHz, avail on pin
        CLOCK_EXT      = 2,        // Clock source: external 64 KHz
        CLOCK_EXT_HALF = 3,        // Clock source: ext 64 KHz, halved
        MOD_CHOP_DIS   = 0x0010,   // Mode: chop disable
        MOD_FRS_M      = 0x000f,   // Mode: filter rate select mask
        MOD_FRS_V      = 0,        // Mode: filter rate select offset
        FRS_RATE_470   = 1,        // Filter rate: 470 Hz
        FRS_RATE_242   = 2,        // Filter rate: 242 Hz
        FRS_RATE_123   = 3,        // Filter rate: 123 Hz
        FRS_RATE_62    = 4,        // Filter rate: 62 Hz
        FRS_RATE_50    = 5,        // Filter rate: 50 Hz
        FRS_RATE_39    = 6,        // Filter rate: 39 Hz
        FRS_RATE_33    = 7,        // Filter rate: 33.2 Hz
        FRS_RATE_19    = 8,        // Filter rate: 19.6 Hz
        FRS_RATE_17    = 9,        // Filter rate: 16.7 Hz (80 dB rej'n)
        FRS_RATE_16    = 10,       // Filter rate: 16.7 Hz (65 dB rej'n)
        FRS_RATE_12    = 11,       // Filter rate: 12.5 Hz
        FRS_RATE_10    = 12,       // Filter rate: 10 Hz
        FRS_RATE_8     = 13,       // Filter rate: 8.33 Hz
        FRS_RATE_6     = 14,       // Filter rate: 6.25 Hz
        FRS_RATE_4     = 15,       // Filter rate: 4.17 Hz
        MOD_STANDARD   = FRS_RATE_16 << MOD_FRS_V,

        CFG_VBIAS_M    = 0xc000,   // Config: bias voltage mask
        CFG_VBIAS_V    = 14,       // Config: bias voltage offset
        VBIAS_OFF      = 0,        // Bias voltage: disabled
        VBIAS_AIN1     = 1,        // Bias voltage: connected to AIN1(-)
        VBIAS_AIN2     = 2,        // Bias voltage: connected to AIN2(-)
        VBIAS_AIN3     = 3,        // Bias voltage: connected to AIN3(-)
        CFG_BURNOUT    = 0x2000,   // Config: burnout current enabled
        CFG_UNIPOLAR   = 0x1000,   // Config: unipolar operation
        CFG_BOOST      = 0x0800,   // Config: boost bias voltage current
        CFG_GAIN_M     = 0x0700,   // Config: gain select mask
        CFG_GAIN_V     = 8,        // Config: gain select offset
        GAIN_1         = 0,        // ADC gain: 1
        GAIN_2         = 1,        // ADC gain: 2
        GAIN_4         = 2,        // ADC gain: 4
        GAIN_8         = 3,        // ADC gain: 8
        GAIN_16        = 4,        // ADC gain: 16
        GAIN_32        = 5,        // ADC gain: 32
        GAIN_64        = 6,        // ADC gain: 64
        GAIN_128       = 7,        // ADC gain: 128
        RANGE_2500     = GAIN_1,   // ADC range: 2.5 V
        RANGE_1250     = GAIN_2,   // ADC range: 1.25 V
        RANGE_625      = GAIN_4,   // ADC range: 0.625 V
        RANGE_312      = GAIN_8,   // ADC range: 0.3125 V
        RANGE_156      = GAIN_16,  // ADC range: 0.15625 V
        RANGE_78       = GAIN_32,  // ADC range: 0.078125 V
        RANGE_39       = GAIN_64,  // ADC range: 0.0390625 V
        RANGE_19       = GAIN_128, // ADC range: 0.01953125 V
        CFG_REFSEL_M   = 0x00c0,   // Config: reference select mask
        CFG_REFSEL_V   = 6,        // Config: reference select offset
        REFSEL_EXT1    = 0,        // Ref select: external on REFIN1
        REFSEL_EXT2    = 1,        // Ref select: external on REFIN2
        REFSEL_INT     = 2,        // Ref select: internal
        CFG_REF_DETECT = 0x0020,   // Config: reference detect enable
        CFG_BUFFERED   = 0x0010,   // Config: buffered mode
        CFG_CHANNEL_M  = 0x000f,   // Config: channel select mask
        CFG_CHANNEL_V  = 0,        // Config: channel select offset
        CHAN_AIN1      = 0,        // Channel: AIN1
        CHAN_AIN2      = 1,        // Channel: AIN2
        CHAN_AIN3      = 2,        // Channel: AIN3
        CHAN_AIN4      = 3,        // Channel: AIN4
        CHAN_AIN5      = 4,        // Channel: AIN5
        CHAN_AIN6      = 5,        // Channel: AIN6
        CHAN_TEMP      = 6,        // Channel: temperature sensor
        CHAN_VDD       = 7,        // Channel: VDD monitor
        CHAN_AIN1A     = 8,        // Channel: AIN1 (again)
        CFG_STANDARD   = (GAIN_128 << CFG_GAIN_V) | CFG_BUFFERED,

        IO_DIOENA      = 0x40,     // IO: digital IO enable
        IO_IO2DAT      = 0x20,     // IO: pin P2 data
        IO_IO1DAT      = 0x10,     // IO: pin P1 data
        IO_IEXCDIR_M   = 0x0c,     // IO: current sources direction mask
        IO_IEXCDIR_V   = 2,        // IO: current sources direction offset
        IEXCDIR_12     = 0,        // Sources direction: IOUT1 & IOUT2
        IEXCDIR_21     = 1,        // Sources direction: IOUT2 & IOUT1
        IEXCDIR_11     = 2,        // Sources direction: both to IOUT1
        IEXCDIR_22     = 3,        // Sources direction: both to IOUT2
        IO_IEXCENA_M   = 0x03,     // IO: current sources enable mask
        IO_IEXCENA_V   = 0,        // IO: current sources enable offset
        IEXCENA_OFF    = 0,        // Sources enable: disabled
        IEXCENA_10     = 1,        // Sources enable: 10 uA
        IEXCENA_210    = 2,        // Sources enable: 210 uA
        IEXCENA_1000   = 3,        // Sources enable: 1 mA
        IO_STANDARD    = 0,

        OPTN_CONFIG    = 0x01,     // Data read option: configure conv. mode
        OPTN_SINGLE    = 0x02,     // Data read option: single mode, not cont.
        OPTN_IMMED     = 0x04,     // Data read option: don't wait for ready

        DIO_ENABLE     = 0x01,     // DIO value: enable digital output
        DIO_P1DATA     = 0x02,     // DIO value: pin P1 on
        DIO_P2DATA     = 0x04,     // DIO value: pin P2 on

        CAL_INT_ZERO   = 0,        // Calibration type: internal zero
        CAL_INT_FULL   = 1,        // Calibration type: internal full-scale
        CAL_SYS_ZERO   = 2,        // Calibration type: system zero
        CAL_SYS_FULL   = 3;        // Calibration type: system full-scale

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int[]
        REG_SIZE = {1, 2, 2, 3, 1, 1, 3, 3};   // Array of register sizes

    private final static double
        HALF_RANGE      = 8388608.0,
        REF_VOLTAGE     = 1.17,
        INT_TEMP_SCALE  = 100.0,
        INT_VOLT_SCALE  = 6.0;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public TempRtds()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public TempRtds(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Resets the chip.
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void reset() throws REBException
    {
        write(REG_RTD_RESET, 0);
    }


   /**
    ***************************************************************************
    **
    **  Performs standard chip setup
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setupStandard() throws REBException
    {
        writeRegister(MODE_REG, MOD_STANDARD);
        writeRegister(CONFIG_REG, CFG_STANDARD);
        writeRegister(IO_REG, IO_STANDARD);
    }


   /**
    ***************************************************************************
    **
    **  Gets the chip status
    **
    **  @return  The status value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getStatus() throws REBException
    {
        return readRegister(STATUS_REG);
    }


   /**
    ***************************************************************************
    **
    **  Sets the channel number
    **
    **  @param  chan  The channel number to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setChannel(int chan) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_CHANNEL_M, chan << CFG_CHANNEL_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the channel number
    **
    **  @return  The channel number
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getChannel() throws REBException
    {
        return (readRegister(CONFIG_REG) & CFG_CHANNEL_M) >> CFG_CHANNEL_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the conversion mode
    **
    **  @param  mode  The mode value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setConvMode(int mode) throws REBException
    {
        updateRegister(MODE_REG, MOD_MODE_M, mode << MOD_MODE_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the conversion mode
    **
    **  @return  The mode value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getConvMode() throws REBException
    {
        return (readRegister(MODE_REG) & MOD_MODE_M) >> MOD_MODE_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the clock source
    **
    **  @param  source  The source value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setClockSource(int source) throws REBException
    {
        updateRegister(MODE_REG, MOD_CLOCK_M, source << MOD_CLOCK_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the clock source
    **
    **  @return  The source value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getClockSource() throws REBException
    {
        return (readRegister(MODE_REG) & MOD_CLOCK_M) >> MOD_CLOCK_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the power switch state
    **
    **  @param  on  Whether or not to set power switch on
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setPowerSwitch(boolean on) throws REBException
    {
        updateRegister(MODE_REG, MOD_PSW, on ? MOD_PSW : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the power switch state
    **
    **  @return  Whether or not power switch is on
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public boolean isPowerSwitch() throws REBException
    {
        return (readRegister(MODE_REG) & MOD_PSW) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the conversion rate
    **
    **  @param  rate  The rate value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setConvRate(int rate) throws REBException
    {
        updateRegister(MODE_REG, MOD_FRS_M, rate << MOD_FRS_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the conversion rate
    **
    **  @return  The conversion rate
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getConvRate() throws REBException
    {
        return (readRegister(MODE_REG) & MOD_FRS_M) >> MOD_FRS_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the unipolar mode
    **
    **  @param  enable  Whether or not to set unipolar mode
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setUnipolar(boolean enable) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_UNIPOLAR, enable ? CFG_UNIPOLAR : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the unipolar mode
    **
    **  @return  Whether or not unipolar mode is set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public boolean isUnipolar() throws REBException
    {
        return (readRegister(CONFIG_REG) & CFG_UNIPOLAR) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the buffered mode
    **
    **  @param  enable  Whether or not to set buffered mode
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setBuffered(boolean enable) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_BUFFERED, enable ? CFG_BUFFERED : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the buffered mode
    **
    **  @return  Whether or not buffered mode is set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public boolean isBuffered() throws REBException
    {
        return (readRegister(CONFIG_REG) & CFG_BUFFERED) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the reference detect mode
    **
    **  @param  enable  Whether or not to set reference detect mode
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setRefDetect(boolean enable) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_REF_DETECT, enable ? CFG_REF_DETECT : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the reference detect mode
    **
    **  @return  Whether or not reference detect mode is set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public boolean isRefDetect() throws REBException
    {
        return (readRegister(CONFIG_REG) & CFG_REF_DETECT) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the reference selection
    **
    **  @param  mode  The reference selection value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setRefSelect(int mode) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_REFSEL_M, mode << CFG_REFSEL_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the reference selection
    **
    **  @return  The reference selection value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getRefSelect() throws REBException
    {
        return (readRegister(CONFIG_REG) & CFG_REFSEL_M) >> CFG_REFSEL_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the ADC gain
    **
    **  @param  gain  The gain value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setGain(int gain) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_GAIN_M, gain << CFG_GAIN_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the ADC gain
    **
    **  @return  The gain value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getGain() throws REBException
    {
        return (readRegister(CONFIG_REG) & CFG_GAIN_M) >> CFG_GAIN_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the DIO values
    **
    **  @param  value  The DIO values to set, the OR of the following:
    **                   DIO_ENABLE: enable digital output
    **                   DIO_P1DATA: set 1 on pin P1
    **                   DIO_P2DATA: set 1 on pin P2
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setDio(int value) throws REBException
    {
        int mask = ((value & DIO_ENABLE) != 0 ? IO_DIOENA : 0)
                     | ((value & DIO_P1DATA) != 0 ? IO_IO1DAT : 0)
                     | ((value & DIO_P2DATA) != 0 ? IO_IO2DAT : 0);
        updateRegister(IO_REG, IO_DIOENA | IO_IO1DAT | IO_IO2DAT, mask);
    }


   /**
    ***************************************************************************
    **
    **  Gets the DIO values
    **
    **  @return  The DIO values currently set, encoded as follows:
    **             DIO_ENABLE: digital output enabled
    **             DIO_P1DATA: 1 on pin P1
    **             DIO_P2DATA: 1 on pin P2
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getDio() throws REBException
    {
        int mask = readRegister(IO_REG) & (IO_DIOENA | IO_IO1DAT | IO_IO2DAT);
        return ((mask & IO_DIOENA) != 0 ? DIO_ENABLE : 0)
                 | ((mask &IO_IO1DAT) != 0 ? DIO_P1DATA : 0)
                 | ((mask &IO_IO2DAT) != 0 ? DIO_P2DATA : 0);
    }


   /**
    ***************************************************************************
    **
    **  Sets the current source direction
    **
    **  @param  dirn  The direction value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setSourceDirection(int dirn) throws REBException
    {
        updateRegister(IO_REG, IO_IEXCDIR_M, dirn << IO_IEXCDIR_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the current source direction
    **
    **  @return  The direction value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getSourceDirection() throws REBException
    {
        return (readRegister(IO_REG) & IO_IEXCDIR_M) >> IO_IEXCDIR_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the current source enable value
    **
    **  @param  enab  The enable value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void setSourceEnable(int enab) throws REBException
    {
        updateRegister(IO_REG, IO_IEXCENA_M, enab << IO_IEXCENA_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the current source enable value
    **
    **  @return  The enable value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int getSourceEnable() throws REBException
    {
        return (readRegister(IO_REG) & IO_IEXCENA_M) >> IO_IEXCENA_V;
    }


   /**
    ***************************************************************************
    **
    **  Reads an ADC channel
    **
    **  @param  chan   The ADC channel to read (0 - 7)
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_CONFIG: First configures the conversion mode.
    **
    **                 OPTN_SINGLE: Configures for a single conversion,
    **                              otherwise continuous.
    **
    **                 OPTN_IMMED:  Reads the data register immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read value (raw counts)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int readAdc(int chan, int optns) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_CHANNEL_M, chan << CFG_CHANNEL_V);
        if ((optns & OPTN_CONFIG) != 0) {
            int mode = (optns & OPTN_SINGLE) != 0 ? MODE_SINGLE : MODE_CONT;
            updateRegister(MODE_REG, MOD_MODE_M, mode << MOD_MODE_V);
        }
        if ((optns & OPTN_IMMED) == 0) {
            waitReady();
        }

        return readRegister(DATA_REG);
    }


   /**
    ***************************************************************************
    **
    **  Reads the internal temperature sensor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_CONFIG: First configures the conversion mode.
    **
    **                 OPTN_SINGLE: Configures for a single conversion,
    **                              otherwise continuous.
    **
    **                 OPTN_IMMED:  Reads the data register immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read temperature (Celsius)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public double readTemperature(int optns) throws REBException
    {
        return (readAdc(CHAN_TEMP, optns) / HALF_RANGE - 1.0) * REF_VOLTAGE
                 * INT_TEMP_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads the internal voltage monitor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_CONFIG: First configures the conversion mode.
    **
    **                 OPTN_SINGLE: Configures for a single conversion,
    **                              otherwise continuous.
    **
    **                 OPTN_IMMED:  Reads the data register immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read voltage
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public double readVdd(int optns) throws REBException
    {
        return (readAdc(CHAN_VDD, optns) / HALF_RANGE - 1.0) * REF_VOLTAGE
                 * INT_VOLT_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Calibrates an ADC channel
    **
    **  @param  chan  The ADC channel to calibrate (0 - 7)
    **
    **  @param  type  The type of calibration to perform:
    **
    **                  CAL_INT_ZERO: internal zero-point
    **                  CAL_INT_FULL: internal full-scale
    **                  CAL_SYS_ZERO: system zero-point
    **                  CAL_SYS_FULL: system full-scale
    **
    **  @return  The new calibration value (raw counts)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int calibrate(int chan, int type) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_CHANNEL_M, chan << CFG_CHANNEL_V);
        updateRegister(MODE_REG, MOD_MODE_M,
                       (type | MODE_INT_Z_CAL) << MOD_MODE_V);
        waitReady();

        return readRegister((type & CAL_INT_FULL) == 0 ? OFFSET_REG
                                                       : SCALE_REG);
    }


   /**
    ***************************************************************************
    **
    **  Gets the calibration data for an ADC channel.
    **
    **  @param  chan  The ADC channel (0 - 7)
    **
    **  @return  The calibration offset and scale values (raw counts)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int[] getCalibration(int chan) throws REBException
    {
        updateRegister(CONFIG_REG, CFG_CHANNEL_M, chan << CFG_CHANNEL_V);
        int[] value = {readRegister(OFFSET_REG), readRegister(SCALE_REG)};

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads a register.
    **
    **  @param  regnum   The register to read
    **
    **  @return  The value read
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public int readRegister(int regnum) throws REBException
    {
        int reg = regnum & 0x07;
        write(REG_RTD_COMMAND, CMND_READ | (reg << 16));
        return read(REG_RTD_REPLY) & ((1 << (8 * REG_SIZE[reg])) - 1);
    }


   /**
    ***************************************************************************
    **
    **  Writes a register
    **
    **  @param  regnum  The register to write
    **
    **  @param  value   The value to write
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void writeRegister(int regnum, int value) throws REBException
    {
        int reg = regnum & 0x07;
        int val = (REG_SIZE[reg] == 1) ? (value & 0xff) << 8 : value & 0xffff;
        write(REG_RTD_COMMAND, reg << 16 | val);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register
    **
    **  @param  regnum  The register to update
    **
    **  @param  mask    The mask of bits to update
    **
    **  @param  value   The value to write, under the mask
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void updateRegister(int regnum, int mask, int value)
        throws REBException
    {
        int reg = regnum & 0x07;
        write(REG_RTD_COMMAND, CMND_READ | (reg << 16));
        int val = (read(REG_RTD_REPLY) & ~mask) | (value & mask);
        val = (REG_SIZE[reg] == 1) ? (val & 0xff) << 8 : val & 0xffff;
        write(REG_RTD_COMMAND, reg << 16 | val);
    }


    /*
    ***************************************************************************
    **
    **  Overwriting read and write routines for debugging.
    **
    ***************************************************************************
    */
    @Override
    public int read(int addr) throws REBException
    {
        if (debug) {
            System.out.format("%s read  %08x:", System.currentTimeMillis(), addr);
        }
        int value = super.read(addr);
        if (debug) {
            System.out.format(" %08x\n", value);
        }
        sleep(readDelay);
        return value;
    }

    @Override
    public void write(int addr, int value) throws REBException
    {
        if (debug) {
            System.out.format("%s write %08x: %08x\n", System.currentTimeMillis(),
                              addr, value);
        }
        super.write(addr, value);
        sleep(writeDelay);
    }

    private boolean debug;
    private int readDelay, writeDelay;

    public void setDebug(boolean on)
    {
        debug = on;
    }

    public void setDelay(int rDelay, int wDelay)
    {
        readDelay = rDelay;
        writeDelay = wDelay;
    }


   /**
    ***************************************************************************
    **
    **  Gets the array of register sizes
    **
    **  @return  The register size array
    **
    ***************************************************************************
    */
    public int[] getRegSizes()
    {
        return REG_SIZE;
    }


   /**
    ***************************************************************************
    **
    **  Waits for the chip to become ready after a conversion
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    private void waitReady() throws REBException
    {
        long start = System.currentTimeMillis();
        boolean ready = false;
        while (!(ready = (getStatus() & STS_READY) == 0)
                 && System.currentTimeMillis() - start <= 5000) {
            try {
                Thread.sleep(5);
            }
            catch (InterruptedException e) {
            }
        }
        if (!ready) {
            throw new REBException("Conversion timed out");
        }
    }


   /**
    ***************************************************************************
    **
    **  Sleeps for a period.
    **
    ***************************************************************************
    */
    private void sleep(int msec) throws REBException
    {
        try {
            Thread.sleep(msec);
        }
        catch (InterruptedException e) {
        }
    }

}
