package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 ******************************************************************************
 **
 **  ASIC ADC reading routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class SlowAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        VERSION_WGREB1    = VERSION_0,
        VERSION_WGREB2    = VERSION_2,
        VERSION_REB3      = VERSION_1,
        VERSION_REB4      = VERSION_3,

        REG_TEMP_SLCT     = 0x600100,
        REG_TEMP_READ     = 0x600101,
        REG_TEMP_MUX      = 0x600200,
        TEMP_CABAC_TOP    = 0,
        TEMP_CABAC_BOTTOM = 1,
        TEMP_ASPIC_TOP    = 2,
        TEMP_ASPIC_BOTTOM = 3,
        NUM_TEMPS         = 4,
        MUX_T_B_SLOW      = 0,
        MUX_T_FAST_B_SLOW = 1,
        MUX_T_SLOW_B_FAST = 2,
        MUX_T_B_FAST      = 3,
        NUM_MUX_SET       = 4,

        REG_ASPIC_START   = 0x600100,
        REG_ASPIC_READ    = 0x601000,
        REG_MUX_CONFIG    = 0x600101,
        REG_MUX_READ      = 0x601010,
        NUM_ASPIC_TEMPS   = 6,
        NUM_CURR_CHANS    = 8,

        REG_MON_START     = 0xc00000,
        REG_MON_CONFIG    = 0xc00001,
        REG_MON_READ      = 0xc00010,
        OFF_MON_ADCS      = 0x100,
        NUM_MON_ADCS      = 2,
        OFF_MON_OD        = 0,
        OFF_MON_GD        = 1,
        OFF_MON_OG        = 2,
        OFF_MON_RD        = 3,
        OFF_MON_CKP       = 4,
        OFF_MON_CKS       = 5,
        OFF_MON_RG        = 6,
        OFF_MON_VREF25    = 7,
        OFF_MON_SCK_L     = 0,
        OFF_MON_SCK_U     = 1,
        OFF_MON_RG_L      = 2,
        OFF_MON_RG_U      = 3,
        OFF_MON_CKP_SH    = 4,
        OFF_MON_I_OD      = 5,

        CHAN_OD_0         = 0,
        CHAN_OG_0         = 1,
        CHAN_RD_0         = 2,
        CHAN_GD_0         = 3,
        CHAN_OD_1         = 4,
        CHAN_OG_1         = 5,
        CHAN_RD_1         = 6,
        CHAN_GD_1         = 7,
        CHAN_OD_2         = 8,
        CHAN_OG_2         = 9,
        CHAN_RD_2         = 10,
        CHAN_GD_2         = 11,
        CHAN_VREF5_0      = 12,
        CHAN_VREF5_1      = 13,
        CHAN_VREF5_2      = 14,
        CHAN_VREF25_1     = 15,
        CHAN_OD           = 16,
        CHAN_OG           = 17,
        CHAN_RD           = 18,
        CHAN_GD           = 19,
        CHAN_VREF25       = 20,
        CHAN_CKP          = 21,
        CHAN_CKS          = 22,
        CHAN_RG           = 23,
        CHAN_SCK_L        = 24,
        CHAN_SCK_U        = 25,
        CHAN_RG_L         = 26,
        CHAN_RG_U         = 27,
        CHAN_CKP_SH       = 28,
        CHAN_I_OD         = 29,
        CHAN_CKP_L        = 30,
        CHAN_CKP_U        = 31,
        CHAN_VP12         = 32,
        CHAN_VN12         = 33;

    public final static double
        VOLT_16_SCALE_0    = 1.5 / 65536,
        VOLT_12_SCALE      = 5.0 / 4096,
        VOLT_12_SCALE_HI   = VOLT_12_SCALE * 11,
        VOLT_16_SCALE      = 10.0 / 65536,
        VOLT_16_SCALE_MD   = VOLT_16_SCALE * 4,
        VOLT_16_SCALE_HI   = VOLT_16_SCALE * 11,
        CURR_16_SCALE_OD   = VOLT_16_SCALE / 60,
        CURR_12_SCALE      = VOLT_12_SCALE / 1000,
        CURR_16_SCALE      = VOLT_16_SCALE / 1000,
        TEMP_OFFSET        = 2.477 / 0.0056,                  // From Claire Juramy
        TEMP_12_SCALE      = VOLT_12_SCALE * (-1.0 / 0.0056), //        "
        TEMP_16_SCALE      = VOLT_16_SCALE * (-1.0 / 0.0056); //        "

    private final static int[]
        voltsCfgReb3 = {0x3c, 0x5c, 0x7c, 0x1c, 0x9c, 0xdc, 0xfc, 0x1d,
                        0xbc, 0x3d, 0x5d, 0x9d, 0xbd, 0xdd, 0x7d, 0xfd};

    private final static double[]
        voltsScaleReb3 = {VOLT_12_SCALE_HI, VOLT_12_SCALE_HI, VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE,    VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE,    VOLT_12_SCALE,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE_HI, VOLT_12_SCALE_HI,
                          VOLT_12_SCALE_HI, VOLT_12_SCALE};

    private final static Map<Integer, Integer> chanMapReb3 = new HashMap<>();
    static {
        chanMapReb3.put(CHAN_OD_0,     0);
        chanMapReb3.put(CHAN_OG_0,     1);
        chanMapReb3.put(CHAN_RD_0,     2);
        chanMapReb3.put(CHAN_GD_0,     3);
        chanMapReb3.put(CHAN_VREF5_0,  4);
        chanMapReb3.put(CHAN_OD_1,     5);
        chanMapReb3.put(CHAN_OG_1,     6);
        chanMapReb3.put(CHAN_RD_1,     7);
        chanMapReb3.put(CHAN_GD_1,     8);
        chanMapReb3.put(CHAN_VREF5_1,  9);
        chanMapReb3.put(CHAN_VREF25_1, 10);
        chanMapReb3.put(CHAN_OD_2,     11);
        chanMapReb3.put(CHAN_OG_2,     12);
        chanMapReb3.put(CHAN_RD_2,     13);
        chanMapReb3.put(CHAN_GD_2,     14);
        chanMapReb3.put(CHAN_VREF5_2,  15);
    }

    private final static int[]
        voltsCfgReb4 = {0x1411, 0x2411, 0x3411, 0x0411, 0x4411, 0x6411,
                        0x7411, 0x0511, 0x5411, 0x1511, 0x2511, 0x4511,
                        0x5511, 0x6511, 0x3511, 0x7511, 0x0001, 0x0101,
                        0x0201, 0x0301, 0x0401, 0x0501, 0x0601, 0x0701};

    private final static double[]
        voltsScaleReb4 = {VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                          VOLT_16_SCALE_HI, VOLT_16_SCALE,    VOLT_16_SCALE_HI,
                          VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                          VOLT_16_SCALE,    VOLT_16_SCALE,    VOLT_16_SCALE_HI,
                          VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                          VOLT_16_SCALE,    VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                          VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                          VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI};

    private final static Map<Integer, Integer> chanMapReb4 = new HashMap<>();
    static {
        chanMapReb4.put(CHAN_OD_0,     0);
        chanMapReb4.put(CHAN_OG_0,     1);
        chanMapReb4.put(CHAN_RD_0,     2);
        chanMapReb4.put(CHAN_GD_0,     3);
        chanMapReb4.put(CHAN_VREF5_0,  4);
        chanMapReb4.put(CHAN_OD_1,     5);
        chanMapReb4.put(CHAN_OG_1,     6);
        chanMapReb4.put(CHAN_RD_1,     7);
        chanMapReb4.put(CHAN_GD_1,     8);
        chanMapReb4.put(CHAN_VREF5_1,  9);
        chanMapReb4.put(CHAN_VREF25_1, 10);
        chanMapReb4.put(CHAN_OD_2,     11);
        chanMapReb4.put(CHAN_OG_2,     12);
        chanMapReb4.put(CHAN_RD_2,     13);
        chanMapReb4.put(CHAN_GD_2,     14);
        chanMapReb4.put(CHAN_VREF5_2,  15);
        chanMapReb4.put(CHAN_CKP_U,    16);
        chanMapReb4.put(CHAN_CKP_L,    17);
        chanMapReb4.put(CHAN_SCK_U,    18);
        chanMapReb4.put(CHAN_SCK_L,    19);
        chanMapReb4.put(CHAN_RG_U,     20);
        chanMapReb4.put(CHAN_RG_L,     21);
        chanMapReb4.put(CHAN_VP12,     22);
        chanMapReb4.put(CHAN_VN12,     23);
    }

    private final static double[][]
        voltsScaleWgreb2 = {{VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                             VOLT_16_SCALE_HI, VOLT_16_SCALE_HI, VOLT_16_SCALE_HI,
                             VOLT_16_SCALE_MD, VOLT_16_SCALE},
                            {VOLT_16_SCALE_MD, VOLT_16_SCALE_MD, VOLT_16_SCALE_MD,
                             VOLT_16_SCALE_MD, VOLT_16_SCALE_MD, CURR_16_SCALE_OD}};

    private final static Map<Integer, Integer> chanMapWgreb2 = new HashMap<>();
    static {
        chanMapWgreb2.put(CHAN_OD,     OFF_MON_OD);
        chanMapWgreb2.put(CHAN_GD,     OFF_MON_GD);
        chanMapWgreb2.put(CHAN_OG,     OFF_MON_OG);
        chanMapWgreb2.put(CHAN_RD,     OFF_MON_RD);
        chanMapWgreb2.put(CHAN_CKP,    OFF_MON_CKP);
        chanMapWgreb2.put(CHAN_CKS,    OFF_MON_CKS);
        chanMapWgreb2.put(CHAN_RG,     OFF_MON_RG);
        chanMapWgreb2.put(CHAN_VREF25, OFF_MON_VREF25);
        chanMapWgreb2.put(CHAN_SCK_L,  0x10 | OFF_MON_SCK_L);
        chanMapWgreb2.put(CHAN_SCK_U,  0x10 | OFF_MON_SCK_U);
        chanMapWgreb2.put(CHAN_RG_L,   0x10 | OFF_MON_RG_L);
        chanMapWgreb2.put(CHAN_RG_U,   0x10 | OFF_MON_RG_U);
        chanMapWgreb2.put(CHAN_CKP_SH, 0x10 | OFF_MON_CKP_SH);
        chanMapWgreb2.put(CHAN_I_OD,   0x10 | OFF_MON_I_OD);
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public SlowAdcs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public SlowAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Reads the REB3/4 slow ADC directly.
    **
    **  @param  config  The ADC/MUX configuration value to use
    **
    **  @return  The ADC value (counts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int readDirect(int config) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4);
        write(REG_MUX_CONFIG, config);
        return read(REG_MUX_READ) & 0xffff;
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of REB3/4 ASPIC temperature values.
    **
    **  @param  first  The number of the first temperature to read
    **
    **  @param  count  The number of temperatures to read
    **
    **  @return  The array of values (Celsius)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAspicTemps(int first, int count) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4);
        if (first < 0 || count < 0 || first + count > NUM_ASPIC_TEMPS) {
            throw new REBException("Invalid ASPIC temperature range");
        }
        int version = getVersion(OPTN_SLOW_ADCS);
        double[] data = new double[count];
        write(REG_ASPIC_START, 0);
        for (int j = 0; j < count; j++) {
            int value = read(REG_ASPIC_READ + first + j);
            if (version == VERSION_REB3) {
                data[j] = TEMP_OFFSET + TEMP_12_SCALE * (value & 0xfff);
            }
            else {
                data[j] = TEMP_OFFSET + TEMP_16_SCALE * (value & 0xffff);
            }
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Reads a REB3/4 ASPIC temperature directly.
    **
    **  @param  strip  The strip number
    **
    **  @param  side   The side: 0 (top) or 1 (bottom)
    **
    **  @return  The temperature
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readAspicTemp(int strip, int side) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4);
        checkStrip(strip);
        checkSide(side);
        if (getVersion(OPTN_SLOW_ADCS) == VERSION_REB3) {
            write(REG_MUX_CONFIG, 4 * strip + side + 2); // Improve this!
            return TEMP_OFFSET + (read(REG_MUX_READ) & 0xfff) * TEMP_12_SCALE;
        }
        else {
            int samChan = 4 * strip + side + 2;
            setMuxReb4(0, samChan & 7, (samChan >> 3) + 1, 1);
            return TEMP_OFFSET + (read(REG_MUX_READ) & 0xffff) * TEMP_16_SCALE;
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a REB3 bias or WGREB2/REB4 clock/bias voltage.
    **
    **  @param  chan   The overall channel number
    **
    **  @return  The voltage
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readVoltage(int chan) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4, VERSION_WGREB2);
        int version = getVersion(OPTN_SLOW_ADCS);
        Map chanMap = version == VERSION_REB3 ? chanMapReb3 :
                      version == VERSION_REB4 ? chanMapReb4 : chanMapWgreb2;
        Integer desc = (Integer)chanMap.get(chan);
        if (desc == null) {
            throw new REBException(("Invalid voltage channel: " + chan));
        }
        if (version == VERSION_REB3) {
            write(REG_MUX_CONFIG, voltsCfgReb3[desc]);
            return (read(REG_MUX_READ) & 0xfff) * voltsScaleReb3[desc];
        }
        else if (version == VERSION_REB4) {
            int cfg = voltsCfgReb4[desc];
            setMuxReb4(cfg >> 12, (cfg >> 8) & 0xf, (cfg >> 4) & 0xf, cfg & 0xf);
            return (read(REG_MUX_READ) & 0xffff) * voltsScaleReb4[desc];
        }
        else {
            int adc = desc >> 4, ch = desc & 0x0f;
            int rawValue = read(REG_MON_READ + adc * OFF_MON_ADCS + ch) & 0xffff;
            return (rawValue - 0x8000) * voltsScaleWgreb2[adc][ch];
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a REB3/4 CCD current.
    **
    **  @param  strip  The strip number
    **
    **  @param  side   The side: 0 (top) or 1 (bottom)
    **
    **  @param  chan   The channel number
    **
    **  @return  The current (amps)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readCurrent(int strip, int side, int chan) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4);
        checkStrip(strip);
        checkSide(side);
        if (chan < 0 || chan >= NUM_CURR_CHANS) {
            throw new REBException(("Invalid current channel: " + chan));
        }
        if (getVersion(OPTN_SLOW_ADCS) == VERSION_REB3) {
            write(REG_MUX_CONFIG, (4 * strip + side) | (chan << 5) | 0x10);
            return (read(REG_MUX_READ) & 0xfff) * CURR_12_SCALE;
        }
        else {
            int samChan = 4 * strip + side;
            setMuxReb4(chan, samChan & 7, (samChan >> 3) + 1, 1);
            return (read(REG_MUX_READ) & 0xffff) * CURR_16_SCALE;
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a REB3/4 CCD current.
    **
    **  @param  chan   The overall channel number: 16*strip + 8*side + chan
    **
    **  @return  The current (amps)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readCurrent(int chan) throws REBException
    {
        return readCurrent(chan >> 4, (chan >> 3) & 1, chan & 7);
    }


   /**
    ***************************************************************************
    **
    **  Fetches the WGREB2 monitoring ADC values.
    **
    **  This causes the values from the specified ADCs to be read into
    **  registers to be retrieved by subsequent readVoltage calls
    **
    **  @param  mask  The mask of ADC numbers to be fetched
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void fetchVoltages(int mask) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_WGREB2);
        for (int j = 0; j < NUM_MON_ADCS; j++) {
            if ((mask & (1 << j)) != 0) {
                write(REG_MON_START + j * OFF_MON_ADCS, 0);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Configures a WGREB2 monitoring ADC.
    **
    **  @param  adc    The ADC number (0 or 1)
    **
    **  @param  value  The value to be sent to the ADC
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void configure(int adc, int value) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_WGREB2);
        checkAdc(adc);
        write(REG_MON_CONFIG + adc * OFF_MON_ADCS, value);
    }


   /**
    ***************************************************************************
    **
    **  Reads the WGREB1 slow ADC.
    **
    **  @param  chan  The number of the channel to read
    **
    **  @return  The converted ADC value (volts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readSlow(int chan) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_WGREB1);
        if (chan < 0 || chan >= NUM_TEMPS) {
            throw new REBException("Invalid slow ADC channel number");
        }
        write(REG_TEMP_SLCT, chan);
        int value = ((read(REG_TEMP_READ) ^ 0x00800000) << 8) >> 14;

        return VOLT_16_SCALE_0 * value;
    }


   /**
    ***************************************************************************
    **
    **  Sets the WGREB1 ASIC MUX.
    **
    **  @param  set  The value to set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setMux(int set) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_WGREB1);
        if (set < 0 || set >= NUM_MUX_SET) {
            throw new REBException("Invalid MUX setting value: " + set);
        }
        write(REG_TEMP_MUX, set);
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of ASPIC temperatures.
    **
    **  @return  The number of ASPIC temperatures
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumAspicTemps() throws REBException
    {
        return isVersion(OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4)
                 ? NUM_ASPIC_TEMPS : 0;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of CCD currents.
    **
    **  @return  The number of CCD currents
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumCcdCurrents() throws REBException
    {
        return isVersion(OPTN_SLOW_ADCS, VERSION_REB3, VERSION_REB4)
                 ? Asic.NUM_ASICS * NUM_CURR_CHANS : 0;
    }


   /**
    ***************************************************************************
    **
    **  Tests a channel number for validity.
    **
    **  @param  chan  The channel number
    **
    ***************************************************************************
    */
    public boolean testChannel(int chan)
    {
        try {
            int version = getVersion(OPTN_SLOW_ADCS);
            Map chanMap = (version == VERSION_REB3) ? chanMapReb3 :
                          (version == VERSION_REB4) ? chanMapReb4 :
                          (version == VERSION_WGREB2) ? chanMapWgreb2 : null;
            return chanMap != null && chanMap.get(chan) != null;
        }
        catch (REBException e) {
            return false;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the REB4 MUX value.
    **
    **  @param  bias   The bias MUX channel
    **
    **  @param  sam    The SAM MUX channel
    **
    **  @param  adc    The ADC channel
    **
    **  @param  range  The encoded range
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void setMuxReb4(int bias, int sam, int adc, int range)
        throws REBException
    {
        write(REG_MUX_CONFIG, (sam << 19) | (bias << 16) | (4 << 9) | (adc << 5) | (range << 1));
    }


   /**
    ***************************************************************************
    **
    **  Checks a strip number.
    **
    **  @param  strip  The strip number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private static void checkStrip(int strip) throws REBException
    {
        if (strip < 0 || strip >= Asic.NUM_STRIPS) {
            throw new REBException("Invalid strip number: " + strip);
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a side number.
    **
    **  @param  side  The side number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private static void checkSide(int side) throws REBException
    {
        if (side < 0 || side >= Asic.NUM_SIDES) {
            throw new REBException(("Invalid board side: " + side));
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks an ADC number.
    **
    **  @param  adc  The ADC number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private static void checkAdc(int adc) throws REBException
    {
        if (adc < 0 || adc >= NUM_MON_ADCS) {
            throw new REBException(("Invalid ADC number: " + adc));
        }
    }

}
