package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  Power ADCs reading routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class PowerAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_POWER_ADCS   = 0x600000,
        ADC_V6_VOLTAGE   = 0,
        ADC_V6_CURRENT   = 1,
        ADC_V9_VOLTAGE   = 2,
        ADC_V9_CURRENT   = 3,
        ADC_V24_VOLTAGE  = 4,
        ADC_V24_CURRENT  = 5,
        ADC_V40_VOLTAGE  = 6,
        ADC_V40_CURRENT  = 7,
        ADC_V5_VOLTAGE   = ADC_V6_VOLTAGE,
        ADC_V5_CURRENT   = ADC_V6_CURRENT,
        ADC_V7_VOLTAGE   = ADC_V9_VOLTAGE,
        ADC_V7_CURRENT   = ADC_V9_CURRENT,
        ADC_V15_VOLTAGE  = ADC_V24_VOLTAGE,
        ADC_V15_CURRENT  = ADC_V24_CURRENT,
        NUM_POWER_REGS_S = 8,

        ADC_DREB_VOLTAGE = 0,
        ADC_DREB_CURRENT = 1,
        ADC_CLKH_VOLTAGE = 2,
        ADC_CLKH_CURRENT = 3,
        ADC_DPHI_VOLTAGE = 4,
        ADC_DPHI_CURRENT = 5,
        ADC_HTR_VOLTAGE  = 6,
        ADC_HTR_CURRENT  = 7,
        ADC_ANA_VOLTAGE  = 8,
        ADC_ANA_CURRENT  = 9,
        ADC_OD_VOLTAGE   = 10,
        ADC_OD_CURRENT   = 11,
        NUM_POWER_REGS_C = 12,
        NUM_POWER_REGS_M = NUM_POWER_REGS_C,

        ERROR_MASK       = 0x00010000,
        VALUE_MASK       = 0x0000ffff;

    public final static double
        VOLTAGE_SCALE    = 1.0 / 640.0,
        CURRENT_SCALE_LO = 10.0 / 640000.0,
        CURRENT_SCALE_HI = 10.0 / 640000.0 / 3.0,
        POWER_ERROR      = -8888.0;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public PowerAdcs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public PowerAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the power ADC reading.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws REBException
    {
        getVersion(OPTN_BOARD_POWER);
        enable(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the data to be available.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void waitDone() throws REBException
    {
        getVersion(OPTN_BOARD_POWER);
        waitDone(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time of the read enable.
    **
    **  @return  The Unix millisecond time of the read enable
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws REBException
    {
        checkNotVersion(OPTN_BOARD_POWER, VERSION_0);
        return getTriggerTime(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Reads one power value.
    **
    **  @param  adc  The number of the ADC to read
    **
    **  @return  The ADC value (volts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readAdc(int adc) throws REBException
    {
        if (adc < 0 || adc >= getNumRegs()) {
            throw new REBException("Invalid power ADC number");
        }
        enable();
        waitDone();
        int value = read(REG_POWER_ADCS + adc);

        return (value & ERROR_MASK) != 0
                 ? POWER_ERROR : getScale(adc) * (value & VALUE_MASK);
    }


   /**
    ***************************************************************************
    **
    **  Reads all the power values.
    **
    **  @return  The array of ADC values (volts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAdcs() throws REBException
    {
        return readAdcs(0, getNumRegs());
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of power values.
    **
    **  @param  first  The number of the first ADC to read
    **
    **  @param  count  The number of ADCs to read
    **
    **  @return  The array of ADC values (volts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAdcs(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > getNumRegs()) {
            throw new REBException("Invalid power ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[count];
        double[] data = new double[count];
        read(REG_POWER_ADCS + first, rawData);
        for (int j = 0, adc = first; j < count; j++, adc++) {
            int value = rawData[j];
            data[j] = (value & ERROR_MASK) != 0
                        ? POWER_ERROR : getScale(adc) * (value & VALUE_MASK);
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of ADC registers.
    **
    **  @return  The number of ADC registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumRegs() throws REBException
    {
        return getVersion(OPTN_BOARD_POWER) == VERSION_2 ? NUM_POWER_REGS_C
                                                         : NUM_POWER_REGS_S;
    }


   /**
    ***************************************************************************
    **
    **  Gets the scale factor.
    **
    **  @param  adc  The ADC number
    **
    **  @return  The scale factor for the measured quantity
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private double getScale(int adc) throws REBException
    {
        if ((adc & 1) == 0) {
            return VOLTAGE_SCALE;
        }
        else if (getVersion(OPTN_BOARD_POWER) != VERSION_2) {
            return adc <= ADC_V9_CURRENT ? CURRENT_SCALE_LO : CURRENT_SCALE_HI;
        }
        else {
            return CURRENT_SCALE_LO;
        }
    }

}
