package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  Java interface to the image client routines
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class GlobalClient {

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    public static final int
        HDW_TYPE_DAQ0 = RegClient.HDW_TYPE_DAQ0,
        HDW_TYPE_DAQ1 = RegClient.HDW_TYPE_DAQ1,
        HDW_TYPE_PCI  = RegClient.HDW_TYPE_PCI;

    private Impl impl;
    private ClientFactory clientFactory = new ClientFactory();



   /**
    ***************************************************************************
    **
    **  Finalizer.
    **
    **  @throws  Throwable
    **
    ***************************************************************************
    */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        if (impl != null) {
            impl.deleteGlobalClient();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets an alternative client factory.
    **
    **  The client factory is used to create the objects that implement
    **  access to either real (by default) or simulated hardware.
    **
    **  @param  clientFactory  The ClientFactory to use
    **
    ***************************************************************************
    */
    public void setClientFactory(ClientFactory clientFactory) {
        this.clientFactory = clientFactory;
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to a set of REBs.
    **
    **  @param  hdw  The hardware type
    **
    **  @param  ids  The array of IDs of the REBs to connect to
    **
    **  @param  ifc  The name of the hardware interface to use.  If null or
    **               empty, the default interface is used.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void open(int hdw, int[] ids, String ifc) throws REBException
    {
        if (impl != null) {
            throw new REBException("Global connection already open");
        }
        impl = clientFactory.createGlobalClient(hdw);
        impl.newGlobalClient(ids, ifc);
    }


   /**
    ***************************************************************************
    **
    **  Closes a connection.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void close() throws REBException
    {
        checkOpen();
        impl.deleteGlobalClient();
        impl = null;
    }


   /**
    ***************************************************************************
    **
    **  Acquires an image (by setting the trigger).
    **
    **  @param  name  The name of the image
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void acquireImage(String name) throws REBException
    {
        checkOpen();
        impl.triggerImage(name);
    }


   /**
    ***************************************************************************
    **
    **  Checks that connection is open
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void checkOpen() throws REBException
    {
        if (impl == null) {
            throw new REBException("Global connection not open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner interface to support choosing the hardware.
    **
    ***************************************************************************
    */
    public interface Impl {

       /**
        ***********************************************************************
        **
        **  Creates a new object.
        **
        **  Creates needed C++ objects.
        **
        **  @param  ids  The IDs of the REBs to connect to
        **
        **  @param  ifc  The name of the hardware interface to use.  If null or
        **               empty, the default interface is used.
        **
        ***********************************************************************
        */
        public default void newGlobalClient(int[] ids, String ifc)
            throws REBException {
        }


       /**
        ***********************************************************************
        **
        **  Deletes an object.
        **
        **  Deletes referenced C++ objects.
        **
        ***********************************************************************
        */
        public default void deleteGlobalClient() {
        }


       /**
        ***********************************************************************
        **
        **  Triggers image acquisition.
        **
        **  @param  name  The image name
        **
        ***********************************************************************
        */
        public default void triggerImage(String name) {
        }
    }

}
