package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 ******************************************************************************
 **
 **  Routines for setting the on-board DACs.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class BoardDacs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        VERSION_REB0_0   = VERSION_0,
        VERSION_REB0_1   = VERSION_1,
        VERSION_REB3     = VERSION_3,
        VERSION_WGREB1   = VERSION_2,
        VERSION_WGREB2   = VERSION_4,

        REG_BOARD_DACS   = 0x400000,
        REG_BOARD_DACS_0 = 0x500001,
        OFF_CLOCK_RAILS  = 0,
        OFF_CS_GATE      = 0x10,
        OFF_HEATER       = 0X10,
        OFF_CABAC_ALT    = 0x100,
        OFF_BIAS_GATE    = 0X100,
        INC_STRIPS       = 0x10,
        DACS_SET         = 0,
        DACS_LOAD        = 1,
        DAC_HAS_STRIP    = 0x10,
        DAC_CLOCK_RAILS  = 0,
        DAC_CS_GATE      = 1,
        DAC_CABAC_ALT    = 2,
        DAC_HEATER       = 3,
        DAC_BIAS_GATE    = DAC_HAS_STRIP,
        NUM_STRIPS       = Asic.NUM_STRIPS,

        CHAN_SCLK_L      = 0,
        CHAN_SCLK_L_SH   = 1,
        CHAN_SCLK_H      = 2,
        CHAN_SCLK_H_SH   = 3,
        CHAN_RG_L        = 4,
        CHAN_RG_L_SH     = 5,
        CHAN_RG_H        = 6,
        CHAN_RG_H_SH     = 7,
        CHAN_PCLK_L      = 8,
        CHAN_PCLK_L_SH   = 9,
        CHAN_PCLK_H      = 10,
        CHAN_PCLK_H_SH   = 11,
        CHAN_HEATER_1    = 12,
        CHAN_HEATER_2    = 13,
        CHAN_CSGATE_1    = 14,
        CHAN_CSGATE_2    = 15,
        CHAN_CSGATE_3    = 16,
        CHAN_FSB_CTRL    = 17,
        CHAN_OD          = 18,
        CHAN_GD          = 19,
        CHAN_RD          = 20,
        CHAN_OG          = 21,
        CHAN_OG_SH       = 22,
        CHAN_HEATER      = 23,
        CHAN_CSGATE      = 24,
        CHAN_OD_1        = 25,
        CHAN_OD_2        = 26,
        CHAN_OD_3        = 27,
        CHAN_OD_CTRL     = CHAN_OD;
    
   /**
    ***************************************************************************
    **
    **  Maps.
    **
    ***************************************************************************
    */
    private static final Map<Integer, Integer> dacMap_REB0 = new HashMap<>();
    static {
        dacMap_REB0.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMap_REB0.put(DAC_CS_GATE,     OFF_CS_GATE);
    }

    private static final Map<Integer, Integer> dacMap_WGREB1 = new HashMap<>();
    static {
        dacMap_WGREB1.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMap_WGREB1.put(DAC_CS_GATE,     OFF_CS_GATE);
        dacMap_WGREB1.put(DAC_CABAC_ALT,   OFF_CABAC_ALT);
    }

    private static final Map<Integer, Integer> dacMap_REB3 = new HashMap<>();
    static {
        dacMap_REB3.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMap_REB3.put(DAC_HEATER,      OFF_HEATER);
        dacMap_REB3.put(DAC_BIAS_GATE,   OFF_BIAS_GATE);
    }

    private static final Map<Integer, Integer> chanMap_REB0 = new HashMap<>();
    static {
        chanMap_REB0.put(CHAN_SCLK_L,   (DAC_CLOCK_RAILS << 16) | 0);
        chanMap_REB0.put(CHAN_SCLK_H,   (DAC_CLOCK_RAILS << 16) | 1);
        chanMap_REB0.put(CHAN_RG_L,     (DAC_CLOCK_RAILS << 16) | 2);
        chanMap_REB0.put(CHAN_RG_H,     (DAC_CLOCK_RAILS << 16) | 3);
        chanMap_REB0.put(CHAN_PCLK_L,   (DAC_CLOCK_RAILS << 16) | 4);
        chanMap_REB0.put(CHAN_PCLK_H,   (DAC_CLOCK_RAILS << 16) | 5);
        chanMap_REB0.put(CHAN_HEATER_1, (DAC_CLOCK_RAILS << 16) | 6);
        chanMap_REB0.put(CHAN_HEATER_2, (DAC_CLOCK_RAILS << 16) | 7);
        chanMap_REB0.put(CHAN_CSGATE_1, (DAC_CS_GATE << 16) | 0);
        chanMap_REB0.put(CHAN_CSGATE_2, (DAC_CS_GATE << 16) | 1);
        chanMap_REB0.put(CHAN_CSGATE_3, (DAC_CS_GATE << 16) | 2);
    }

    private static final Map<Integer, Integer> chanMap_REB3 = new HashMap<>();
    static {
        chanMap_REB3.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMap_REB3.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMap_REB3.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMap_REB3.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMap_REB3.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMap_REB3.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMap_REB3.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMap_REB3.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMap_REB3.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMap_REB3.put(CHAN_HEATER,    (DAC_HEATER << 16) | 0);
        chanMap_REB3.put(CHAN_GD,        (DAC_BIAS_GATE << 16) | 0);
        chanMap_REB3.put(CHAN_OD,        (DAC_BIAS_GATE << 16) | 1);
        chanMap_REB3.put(CHAN_OG_SH,     (DAC_BIAS_GATE << 16) | 2);
        chanMap_REB3.put(CHAN_OG,        (DAC_BIAS_GATE << 16) | 3);
        chanMap_REB3.put(CHAN_RD,        (DAC_BIAS_GATE << 16) | 4);
        chanMap_REB3.put(CHAN_CSGATE,    (DAC_BIAS_GATE << 16) | 5);
    }

    private static final Map<Integer, Integer> chanMap_WGREB1 = new HashMap<>();
    static {
        chanMap_WGREB1.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMap_WGREB1.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMap_WGREB1.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMap_WGREB1.put(CHAN_SCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMap_WGREB1.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMap_WGREB1.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMap_WGREB1.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x06);
        chanMap_WGREB1.put(CHAN_RG_H_SH,   (DAC_CLOCK_RAILS << 16) | 0x07);
        chanMap_WGREB1.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMap_WGREB1.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMap_WGREB1.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMap_WGREB1.put(CHAN_PCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x13);
        chanMap_WGREB1.put(CHAN_CSGATE,    (DAC_CS_GATE << 16) | 0);
        chanMap_WGREB1.put(CHAN_CSGATE_1,  (DAC_CS_GATE << 16) | 0);
        chanMap_WGREB1.put(CHAN_CSGATE_2,  (DAC_CS_GATE << 16) | 1);
        chanMap_WGREB1.put(CHAN_CSGATE_3,  (DAC_CS_GATE << 16) | 2);
        chanMap_WGREB1.put(CHAN_FSB_CTRL,  (DAC_CS_GATE << 16) | 3);
        chanMap_WGREB1.put(CHAN_HEATER_1,  (DAC_CS_GATE << 16) | 4);
        chanMap_WGREB1.put(CHAN_HEATER_2,  (DAC_CS_GATE << 16) | 5);
        chanMap_WGREB1.put(CHAN_OD,        (DAC_CS_GATE << 16) | 6);
        chanMap_WGREB1.put(CHAN_GD,        (DAC_CABAC_ALT << 16) | 0);
        chanMap_WGREB1.put(CHAN_RD,        (DAC_CABAC_ALT << 16) | 1);
        chanMap_WGREB1.put(CHAN_OG,        (DAC_CABAC_ALT << 16) | 2);
        chanMap_WGREB1.put(CHAN_OG_SH,     (DAC_CABAC_ALT << 16) | 3);
    }

    private static final Map<Integer, Integer> chanMap_WGREB2 = new HashMap<>();
    static {
        chanMap_WGREB2.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMap_WGREB2.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMap_WGREB2.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMap_WGREB2.put(CHAN_SCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMap_WGREB2.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMap_WGREB2.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMap_WGREB2.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x06);
        chanMap_WGREB2.put(CHAN_RG_H_SH,   (DAC_CLOCK_RAILS << 16) | 0x07);
        chanMap_WGREB2.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMap_WGREB2.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMap_WGREB2.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMap_WGREB2.put(CHAN_PCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x13);
        chanMap_WGREB2.put(CHAN_CSGATE,    (DAC_CS_GATE << 16) | 2);
        chanMap_WGREB2.put(CHAN_CSGATE_1,  (DAC_CS_GATE << 16) | 0);
        chanMap_WGREB2.put(CHAN_CSGATE_2,  (DAC_CS_GATE << 16) | 1);
        chanMap_WGREB2.put(CHAN_CSGATE_3,  (DAC_CS_GATE << 16) | 2);
        chanMap_WGREB2.put(CHAN_HEATER,    (DAC_CS_GATE << 16) | 4);
        chanMap_WGREB2.put(CHAN_OD,        (DAC_CS_GATE << 16) | 5);
        chanMap_WGREB2.put(CHAN_OD_1,      (DAC_CS_GATE << 16) | 5);
        chanMap_WGREB2.put(CHAN_OD_2,      (DAC_CS_GATE << 16) | 6);
        chanMap_WGREB2.put(CHAN_OD_3,      (DAC_CS_GATE << 16) | 7);
        chanMap_WGREB2.put(CHAN_GD,        (DAC_CABAC_ALT << 16) | 0);
        chanMap_WGREB2.put(CHAN_RD,        (DAC_CABAC_ALT << 16) | 1);
        chanMap_WGREB2.put(CHAN_OG,        (DAC_CABAC_ALT << 16) | 2);
        chanMap_WGREB2.put(CHAN_OG_SH,     (DAC_CABAC_ALT << 16) | 3);
    }

    private static final Set<Integer> stripSet = new HashSet<>();
    static {
        stripSet.add(CHAN_GD);
        stripSet.add(CHAN_RD);
        stripSet.add(CHAN_OG);
        stripSet.add(CHAN_OG_SH);
        stripSet.add(CHAN_OD);
        stripSet.add(CHAN_CSGATE);
    }

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public BoardDacs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public BoardDacs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Gets the DAC version.
    **
    **  @return  The version number: VERSION_0 & VERSION_1 for REB0; VERSION_2
    **           for WGREB1; VERSION_3 for REB3/4; VERSION_4 for WGREB2
    **
    ***************************************************************************
    */
    public int getVersion()
    {
        try {
            return getVersion(OPTN_BOARD_DACS);
        }
        catch (REBException e) {
            return VERSION_UNSUPP;
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads all DAC values.
    **
    **  All the values previously set in the DACs are loaded into their output
    **  registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void loadAll() throws REBException
    {
        loadGlobal();
        loadStrip();
    }


   /**
    ***************************************************************************
    **
    **  Loads all global DAC values.
    **
    **  All the values previously set in the global DACs are loaded into their
    **  output registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void loadGlobal() throws REBException
    {
        for (int dac : getDacMap(getVersion()).keySet()) {
            if ((dac & DAC_HAS_STRIP) == 0) {
                load(dac);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads all strip-associated DAC values.
    **
    **  All the values previously set in the strip-associated DACs are loaded
    **  into their output registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void loadStrip() throws REBException
    {
        for (int strip = 0; strip < NUM_STRIPS; strip++) {
            loadStrip(strip);
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads all DAC values for a strip.
    **
    **  All the values previously set for a particular strip are loaded into
    **  their output registers
    **
    **  @param  strip  The strip number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void loadStrip(int strip) throws REBException
    {
        int version = getVersion();
        if (version == VERSION_WGREB1 || version == VERSION_WGREB2) { // Fake it
            load(DAC_CS_GATE);
            load(DAC_CABAC_ALT);
        }
        else {
            for (int dac : getDacMap(version).keySet()) {
                if ((dac & DAC_HAS_STRIP) != 0) {
                    load(strip, dac);
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads one DAC's values.
    **
    **  All the values previously set in a DAC are loaded into its output
    **  registers.  This is a no-op for VERSION_REB0_0
    **
    **
    **  @param  dac  The DAC number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void load(int dac) throws REBException
    {
        if ((dac & DAC_HAS_STRIP) == 0) {
            int version = getVersion();
            if (version == VERSION_REB0_0) return;
            write(getBase(version, dac, false) + DACS_LOAD, 0);
        }
        else {
            for (int strip = 0; strip < NUM_STRIPS; strip++) {
                load(strip, dac);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads the values for one DAC and strip.
    **
    **  All the values previously set in a DAC are loaded into its output
    **  registers.
    **
    **  @param  strip  The strip number.
    **
    **  @param  dac    The DAC number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void load(int strip, int dac) throws REBException
    {
        write(getBase(getVersion(), dac, true) + getStripOffset(strip) + DACS_LOAD, 0);
    }


   /**
    ***************************************************************************
    **
    **  Sets a DAC value.
    **
    **  The supplied value is set into its DAC, but not loaded into the DAC's
    **  output register until the load method is called
    **
    **  @param  chan   The overall channel number
    **
    **  @param  value  The value to be set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void set(int chan, int value) throws REBException
    {
        int version = getVersion();
        int desc = getDescriptor(version, chan, false);
        write(getBase(version, desc >> 16, false) + DACS_SET,
              ((desc & 0xffff) << 12) | (value & 0x0fff));
    }


   /**
    ***************************************************************************
    **
    **  Sets a DAC value into a strip.
    **
    **  The supplied value is set into the its DAC, but not loaded into the
    **  DAC's output register until the load method is called
    **
    **  @param  strip  The strip number.
    **
    **  @param  chan   The overall channel number
    **
    **  @param  value  The value to be set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void set(int strip, int chan, int value) throws REBException
    {
        int version = getVersion();
        if (version == VERSION_WGREB1 || version == VERSION_WGREB2) { // Fake it
            int desc = getWgrebDescriptor(version, strip, chan);
            write(getBase(version, desc >> 16, false) + DACS_SET,
                  ((desc & 0xffff) << 12) | (value & 0x0fff));
        }
        else {
            int desc = getDescriptor(version, chan, true);
            write(getBase(version, desc >> 16, true) + getStripOffset(strip) + DACS_SET,
                  ((desc & 0xffff) << 12) | (value & 0x0fff));
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets a DAC's base register address.
    **
    **  @param  version   The board DACS system version
    **
    **  @param  dac       The DAC number
    **
    **  @param  hasStrip  Whether DAC is associated with a strip
    **
    **  @return  The base address of the DAC
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int getBase(int version, int dac, boolean hasStrip)
        throws REBException
    {
        Integer base = getDacMap(version).get(dac);
        if (base == null || ((dac & DAC_HAS_STRIP) != 0) != hasStrip) {
            throw new REBException("Invalid DAC number (" + dac + ")");
        }
        return (version == VERSION_REB0_0 ? REG_BOARD_DACS_0 : REG_BOARD_DACS) + base;
    }


   /**
    ***************************************************************************
    **
    **  Gets a strip's register address offset.
    **
    **  @param  strip  The strip number
    **
    **  @return  The strip address offset
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int getStripOffset(int strip) throws REBException
    {
        if (strip < 0 || strip >= NUM_STRIPS) {
            throw new REBException("Invalid strip number (" + strip + ")");
        }
        return INC_STRIPS * strip;
    }


   /**
    ***************************************************************************
    **
    **  Gets the applicable DAC map.
    **
    **  @param  version   The option version number
    **
    **  @return  The DAC map
    **
    ***************************************************************************
    */
    private Map<Integer, Integer> getDacMap(int version)
    {
        return version <= VERSION_REB0_1 ? dacMap_REB0 :
               version == VERSION_WGREB1 ? dacMap_WGREB1 :
               version == VERSION_REB3 ? dacMap_REB3 : dacMap_WGREB1;
    }


   /**
    ***************************************************************************
    **
    **  Gets the appropriate channel map.
    **
    **  @param  version   The option version number
    **
    **  @return  The channel map
    **
    ***************************************************************************
    */
    private Map<Integer, Integer> getChanMap(int version)
    {
        return version <= VERSION_REB0_1 ? chanMap_REB0 :
               version == VERSION_WGREB1 ? chanMap_WGREB1 :
               version == VERSION_REB3 ? chanMap_REB3 : chanMap_WGREB2;
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel descriptor.
    **
    **  @param  version   The option version number
    **
    **  @param  chan      The channel number
    **
    **  @param  hasStrip  Whether channel is associated with a strip
    **
    **  @return  The channel descriptor
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int getDescriptor(int version, int chan, boolean hasStrip)
        throws REBException
    {
        Map<Integer, Integer> chanMap = getChanMap(version);
        Integer desc = chanMap.get(chan);
        if (desc == null || (((desc >> 16) & DAC_HAS_STRIP) != 0) != hasStrip) {
            throw new REBException("Invalid channel number (" + chan + ")");
        }
        return desc;
    }


   /**
    ***************************************************************************
    **
    **  Gets a channel descriptor for a WGREB "strip".
    **
    **  @param  version   The DAC version number
    **
    **  @param  strip     The strip number
    **
    **  @param  chan      The channel number
    **
    **  @return  The channel descriptor
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int getWgrebDescriptor(int version, int strip, int chan)
        throws REBException
    {
        if (strip != 0) {
            throw new REBException("Invalid strip number (" + strip + ")");
        }
        if (!stripSet.contains(chan)) {
            throw new REBException("Invalid channel number (" + chan + ")");
        }

        return getChanMap(version).get(chan);
    }

}
