package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 *  Power ADCs reading routines.
 *
 *  There are several supported board versions:
 *
 *    VERSION_0: REB0
 *    VERSION_1: REB1-4
 *    VERSION_2: WREB
 *    VERSION_3: GREB
 *    VERSION_4: REB5
 *    VERSION_5: REB5 with two busy bits
 *
 *  @author Owen Saxton
 */
public class PowerAdcs {

   /**
    *  Public constants.
    */
    public static final int
        VERSION_REB0 = BaseSet.VERSION_0,
        VERSION_REB1 = BaseSet.VERSION_1,
        VERSION_REB5 = BaseSet.VERSION_4,
        VERSION_REB5X = BaseSet.VERSION_5,
        VERSION_WREB = BaseSet.VERSION_2,
        VERSION_GREB = BaseSet.VERSION_3,

        REG_POWER_ADCS   = 0x600000,
        NUM_POWER_REGS_S  = 8,
        NUM_POWER_REGS_S5 = 10,
        NUM_POWER_REGS_W  = 12,
        NUM_POWER_REGS_G  = 10,
        NUM_POWER_REGS_M  = NUM_POWER_REGS_W,

        ADC_DIG_VOLTAGE  = 0,
        ADC_DIG_CURRENT  = 1,
        ADC_ANA_VOLTAGE  = 2,
        ADC_ANA_CURRENT  = 3,
        ADC_CLKH_VOLTAGE = 4,
        ADC_CLKH_CURRENT = 5,
        ADC_OD_VOLTAGE   = 6,
        ADC_OD_CURRENT   = 7,
        ADC_HTR_VOLTAGE  = 8,
        ADC_HTR_CURRENT  = 9,
        ADC_DPHI_VOLTAGE = 10,
        ADC_DPHI_CURRENT = 11,
        ADC_CLKL_VOLTAGE = 8,
        ADC_CLKL_CURRENT = 9,

        ERROR_MASK       = 0x00010000,
        VALUE_MASK       = 0x0000ffff;

    public static final double
        VOLTAGE_SCALE    = 1.0 / 640.0,
        CURRENT_SCALE_LO = 10.0 / 640000.0,
        CURRENT_SCALE_MD = 10.0 / 640000.0 / 2.0,
        CURRENT_SCALE_HI = 10.0 / 640000.0 / 3.0;
 
   /**
    *  Private data
    */
    private static final Map<Integer, Integer> chanMapS = new HashMap<>();
    static {
        chanMapS.put(ADC_DIG_VOLTAGE, 0);
        chanMapS.put(ADC_DIG_CURRENT, 1);
        chanMapS.put(ADC_ANA_VOLTAGE, 2);
        chanMapS.put(ADC_ANA_CURRENT, 3);
        chanMapS.put(ADC_CLKH_VOLTAGE, 4);
        chanMapS.put(ADC_CLKH_CURRENT, 5);
        chanMapS.put(ADC_OD_VOLTAGE, 6);
        chanMapS.put(ADC_OD_CURRENT, 7);
    }

    private static final Map<Integer, Integer> chanMapS5 = new HashMap<>();
    static {
        chanMapS5.put(ADC_DIG_VOLTAGE, 0);
        chanMapS5.put(ADC_DIG_CURRENT, 1);
        chanMapS5.put(ADC_ANA_VOLTAGE, 2);
        chanMapS5.put(ADC_ANA_CURRENT, 3);
        chanMapS5.put(ADC_CLKH_VOLTAGE, 4);
        chanMapS5.put(ADC_CLKH_CURRENT, 5);
        chanMapS5.put(ADC_OD_VOLTAGE, 6);
        chanMapS5.put(ADC_OD_CURRENT, 7);
        chanMapS5.put(ADC_CLKL_VOLTAGE, 8);
        chanMapS5.put(ADC_CLKL_CURRENT, 9);
    }

    private static final Map<Integer, Integer> chanMapW = new HashMap<>();
    static {
        chanMapW.put(ADC_DIG_VOLTAGE, 0);
        chanMapW.put(ADC_DIG_CURRENT, 1);
        chanMapW.put(ADC_ANA_VOLTAGE, 8);
        chanMapW.put(ADC_ANA_CURRENT, 9);
        chanMapW.put(ADC_CLKH_VOLTAGE, 2);
        chanMapW.put(ADC_CLKH_CURRENT, 3);
        chanMapW.put(ADC_OD_VOLTAGE, 10);
        chanMapW.put(ADC_OD_CURRENT, 11);
        chanMapW.put(ADC_HTR_VOLTAGE, 6);
        chanMapW.put(ADC_HTR_CURRENT, 7);
        chanMapW.put(ADC_DPHI_VOLTAGE, 4);
        chanMapW.put(ADC_DPHI_CURRENT, 5);
    }

    private static final Map<Integer, Integer> chanMapG = new HashMap<>();
    static {
        chanMapG.put(ADC_DIG_VOLTAGE, 0);
        chanMapG.put(ADC_DIG_CURRENT, 1);
        chanMapG.put(ADC_ANA_VOLTAGE, 6);
        chanMapG.put(ADC_ANA_CURRENT, 7);
        chanMapG.put(ADC_CLKH_VOLTAGE, 2);
        chanMapG.put(ADC_CLKH_CURRENT, 3);
        chanMapG.put(ADC_OD_VOLTAGE, 8);
        chanMapG.put(ADC_OD_CURRENT, 9);
        chanMapG.put(ADC_HTR_VOLTAGE, 4);
        chanMapG.put(ADC_HTR_CURRENT, 5);
    }

    private static final Map<Integer, Map> chanMapMap = new HashMap<>();
    static {
        chanMapMap.put(VERSION_REB0, chanMapS);
        chanMapMap.put(VERSION_REB1, chanMapS);
        chanMapMap.put(VERSION_REB5, chanMapS5);
        chanMapMap.put(VERSION_REB5X, chanMapS5);
        chanMapMap.put(VERSION_WREB, chanMapW);
        chanMapMap.put(VERSION_GREB, chanMapG);
    }

    private static final Map<Integer, Integer> numRegsMap = new HashMap<>();
    static {
        numRegsMap.put(VERSION_REB0, NUM_POWER_REGS_S);
        numRegsMap.put(VERSION_REB1, NUM_POWER_REGS_S);
        numRegsMap.put(VERSION_REB5, NUM_POWER_REGS_S5);
        numRegsMap.put(VERSION_REB5X, NUM_POWER_REGS_S5);
        numRegsMap.put(VERSION_WREB, NUM_POWER_REGS_W);
        numRegsMap.put(VERSION_GREB, NUM_POWER_REGS_G);
    }

   /**
    *  Fields
    */
    BaseSet bss;


   /**
    *  Constructor.
    *
    *  @param  bss  The associated base set object
    */
    public PowerAdcs(BaseSet bss)
    {
        this.bss = bss;
    }


   /**
    *  Enables the power ADC reading.
    *
    *  @throws  REBException 
    */
    public synchronized void enable() throws REBException
    {
        bss.getVersion(BaseSet.OPTN_BOARD_POWER);
        bss.enable(BaseSet.RSET_POWER_ADCS);
    }


   /**
    *  Waits for the data to be available.
    *
    *  @throws  REBException 
    */
    public synchronized void waitDone() throws REBException
    {
        int version = bss.getVersion(BaseSet.OPTN_BOARD_POWER);
        bss.waitDone(BaseSet.RSET_POWER_ADCS);
        if (version == VERSION_REB5X) {
            bss.waitDone(BaseSet.RSET_TEMP_ADCS);
        }
    }


   /**
    *  Gets the time of the read enable.
    *
    *  @return  The Unix millisecond time of the read enable
    *  @throws  REBException 
    */
    public long getTriggerTime() throws REBException
    {
        bss.checkNotVersion(BaseSet.OPTN_BOARD_POWER, VERSION_REB0);
        return bss.getTriggerTime(BaseSet.RSET_POWER_ADCS);
    }


   /**
    *  Reads one power value.
    *
    *  @param  adc  The number of the ADC to read
    *  @return  The ADC value (volts)
    *  @throws  REBException 
    */
    public synchronized double readAdc(int adc) throws REBException
    {
        Map<Integer, Integer> chanMap = getChanMap();
        Integer regNum = chanMap.get(adc);
        if (regNum == null) {
            throw new REBException("Invalid power ADC number");
        }
        enable();
        waitDone();
        int value = bss.read(REG_POWER_ADCS + regNum);

        return (value & ERROR_MASK) != 0 && !isClockLow(adc)
                 ? BaseSet.ERROR_VALUE : getScale(adc) * (value & VALUE_MASK);
    }


   /**
    *  Reads all the power values.
    *
    *  @return  The array of converted ADC values
    *  @throws  REBException 
    */
    public double[] readAdcs() throws REBException
    {
        return readAdcs(0, getNumRegs());
    }


   /**
    *  Reads a range of power values.
    *
    *  @param  first  The number of the first ADC to read
    *  @param  count  The number of ADCs to read
    *  @return  The array of converted ADC values
    *  @throws  REBException 
    */
    public synchronized double[] readAdcs(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > getNumRegs()) {
            throw new REBException("Invalid power ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[getNumRegs()];
        double[] data = new double[count];
        bss.read(REG_POWER_ADCS, rawData);
        Map<Integer, Integer> chanMap = getChanMap();
        for (int j = 0; j < count; j++) {
            int adc = first + j;
            int value = rawData[chanMap.get(adc)];
            data[j] = (value & ERROR_MASK) != 0 && (!isClockLow(adc) || (value & VALUE_MASK) == 0xffff)
                        ? BaseSet.ERROR_VALUE : getScale(adc) * (value & VALUE_MASK);
        }

        return data;
    }


   /**
    *  Gets the number of ADC registers.
    *
    *  @return  The number of ADC registers
    *  @throws  REBException 
    */
    public int getNumRegs() throws REBException
    {
        return numRegsMap.get(bss.getVersion(BaseSet.OPTN_BOARD_POWER));
    }


   /**
    *  Gets the ADC channel map.
    *
    *  @return  The map of ADC channels to register offsets
    *  @throws  REBException 
    */
    private Map<Integer, Integer> getChanMap() throws REBException
    {
        return chanMapMap.get(bss.getVersion(BaseSet.OPTN_BOARD_POWER));
    }


   /**
    *  Gets the scale factor.
    *
    *  @param  adc  The ADC number
    *  @return  The scale factor for the measured quantity
    *  @throws  REBException 
    */
    private double getScale(int adc) throws REBException
    {
        if ((adc & 1) == 0) {
            return VOLTAGE_SCALE;
        }
        int version = bss.getVersion(BaseSet.OPTN_BOARD_POWER);
        if (version == VERSION_WREB || version == VERSION_GREB) {
            return CURRENT_SCALE_LO;
        }
        else if (version == VERSION_REB5 || version == VERSION_REB5X) {
            return adc <= ADC_ANA_CURRENT ? CURRENT_SCALE_LO
                    : adc == ADC_CLKL_CURRENT ? CURRENT_SCALE_MD : CURRENT_SCALE_HI;
        }
        else {
            return adc <= ADC_ANA_CURRENT ? CURRENT_SCALE_LO : CURRENT_SCALE_HI;
        }
    }


   /**
    *  Tests whether a clock low quantity is being measured.
    *
    *  @param  adc  The ADC number
    *  @return  Whether clock low
    *  @throws  REBException 
    */
    private boolean isClockLow(int adc) throws REBException
    {
        int version = bss.getVersion(BaseSet.OPTN_BOARD_POWER);
        return (version == VERSION_REB5 || version == VERSION_REB5X)
                 && (adc == ADC_CLKL_VOLTAGE || adc == ADC_CLKL_CURRENT);
    }

}
