package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 *  Base register set access routines.
 *
 *  @author Owen Saxton
 */
public class BaseSet extends RegClient {

    /**
     *  Inner class for version numbers.
     */
    public static class Version {

        private final int rebType;
        private final int sequencer;
        private final int cabac;
        private final int aspic;
        private final int rebSn;
        private final int drebSn;
        private final int dcdcSync;
        private final int dacs;
        private final int power;
        private final int boardTemp;
        private final int status;
        private final int ccdTemp;
        private final int slowAdcs;
        private final int fastAdcs;
        private final int backBias;

        public Version(int rebType, int sequencer, int cabac, int aspic,
                       int rebSn, int drebSn, int dcdcSync, int dacs,
                       int power, int boardTemp, int status, int ccdTemp,
                       int slowAdcs, int fastAdcs, int backBias)
        {
            this.rebType = rebType;
            this.sequencer = sequencer;
            this.cabac = cabac;
            this.aspic = aspic;
            this.rebSn = rebSn;
            this.drebSn = drebSn;
            this.dcdcSync = dcdcSync;
            this.dacs = dacs;
            this.power = power;
            this.boardTemp = boardTemp;
            this.status = status;
            this.ccdTemp = ccdTemp;
            this.slowAdcs = slowAdcs;
            this.fastAdcs = fastAdcs;
            this.backBias = backBias;
        }

    }


    /**
     *  Public constants.
     */
    public static final int
        RSET_STATUS       = 0,
        RSET_TIME_BASE    = 1,
        RSET_SEQUENCER    = 2,
        RSET_POWER_ADCS   = 3,
        RSET_TEMP_ADCS    = 4,
        RSET_FAST_ADCS    = 5,

        REG_SCHEMA        = 0x000000,
        REG_VERSION       = 0x000001,
        REG_ID            = 0x000002,
        REG_TIME_BASE     = 0x000004,
        REG_STATE         = 0x000008,
        REG_TRIGGER       = 0x000009,
        REG_TRIG_TIME     = 0x00000a,

        REG_SN_REB_START  = 0x800000,
        REG_SN_REB_VALUE  = 0x800001,
        REG_SN_DREB_START = 0x800010,
        REG_SN_DREB_VALUE = 0x800011,

        REG_DCDC_SYNC     = 0x900000,

        REG_BACK_BIAS     = 0xd00000,
        BACK_BIAS_ON      = 0x01,
        BACK_BIAS_OFF     = 0x02,
        BACK_BIAS_ERROR   = 0x04,

        REB_TYPE_UNKNOWN   = -1,
        REB_TYPE_SCIENCE   = 0,
        REB_TYPE_GUIDER    = 1,
        REB_TYPE_WAVEFRONT = 2,

        MAX_STRIPS         = 3,  // Maximum number of strips (CCDs)

        OPTN_SEQUENCER    = 0,
        OPTN_CABAC        = 1,
        OPTN_ASPIC        = 2,
        OPTN_REB_SN       = 3,
        OPTN_DREB_SN      = 4,
        OPTN_DCDC_SYNC    = 5,
        OPTN_BOARD_DACS   = 6,
        OPTN_BOARD_POWER  = 7,
        OPTN_BOARD_TEMP   = 8,
        OPTN_STATUS       = 9,
        OPTN_CCD_TEMP     = 10,
        OPTN_SLOW_ADCS    = 11,
        OPTN_FAST_ADCS    = 12,
        OPTN_BACK_BIAS    = 13,

        FW_VERSION_INITIAL = -3,
        FW_VERSION_UNSET   = -2,
        FW_VERSION_UNKNOWN = -1,
        FW_VERSION_REB0_0  = 0,
        FW_VERSION_REB0_1  = 1,
        FW_VERSION_REB1_2  = 2,
        FW_VERSION_REB1_3  = 3,
        FW_VERSION_REB1_4  = 4,
        FW_VERSION_WREB_1  = 5,
        FW_VERSION_REB3    = 6,
        FW_VERSION_WREB_2  = 7,
        FW_VERSION_REB4_0  = 8,
        FW_VERSION_GREB_0  = 9,
        FW_VERSION_WREB_3  = 10,
        FW_VERSION_REB5_0  = 11,
        FW_VERSION_REB4_1  = 12,
        FW_VERSION_REB5_1  = 13,
        FW_VERSION_GREB_1  = 14,
        FW_VERSION_WREB_4  = 15,
        FW_VERSION_REB4_2  = 16,
        FW_VERSION_GREB_2  = 17,
        FW_VERSION_WREB_5  = 18,

        VERSION_UNSUPP    = -1,
        VERSION_0         = 0,
        VERSION_1         = 1,
        VERSION_2         = 2,
        VERSION_3         = 3,
        VERSION_4         = 4,
        VERSION_5         = 5,
        VERSION_6         = 6,
        VERSION_7         = 7,
        VERSION_8         = 8,

        //TYPE_UNKNOWN      = -1,
        //TYPE_SCIENCE      = 0,
        //TYPE_CORNER       = 1,

        CLOCK_PERIOD_0    = 5,      // Nanoseconds
        CLOCK_PERIOD_1    = 10;     // Nanoseconds

    public static final long
        SN_READ_OKAY      = 0x0001000000000000L,
        SN_READ_ERROR     = 0x0002000000000000L,
        SN_READ_TIMEOUT   = 0x0004000000000000L,
        SN_VALUE_MASK     = 0x0000ffffffffffffL;

    static double
        ERROR_VALUE = Double.NaN;

    /**
     *  Private data.
     */
    private static final Map<Integer, Version> versionMap = new HashMap<>();
    static {
        versionMap.put(FW_VERSION_REB0_0,
                       new Version(REB_TYPE_SCIENCE, VERSION_0, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_0, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_0, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP));
        versionMap.put(FW_VERSION_REB0_1,
                       new Version(REB_TYPE_SCIENCE, VERSION_1, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_1, VERSION_0,
                                   VERSION_0, VERSION_0, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP));
        versionMap.put(FW_VERSION_REB1_2,
                       new Version(REB_TYPE_SCIENCE, VERSION_1, VERSION_0,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_1, VERSION_1,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP));
        versionMap.put(FW_VERSION_REB1_3,
                       new Version(REB_TYPE_SCIENCE, VERSION_1, VERSION_0,
                                   VERSION_UNSUPP, VERSION_0, VERSION_0,
                                   VERSION_0, VERSION_1, VERSION_1,
                                   VERSION_1, VERSION_1, VERSION_UNSUPP,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP));
        versionMap.put(FW_VERSION_REB1_4,
                       new Version(REB_TYPE_SCIENCE, VERSION_2, VERSION_0,
                                   VERSION_0, VERSION_0, VERSION_0,
                                   VERSION_0, VERSION_1, VERSION_1,
                                   VERSION_1, VERSION_1, VERSION_0,
                                   VERSION_UNSUPP, VERSION_UNSUPP, VERSION_UNSUPP));
        versionMap.put(FW_VERSION_REB3,
                       new Version(REB_TYPE_SCIENCE, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_3, VERSION_1,
                                   VERSION_3, VERSION_1, VERSION_0,
                                   VERSION_1, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_REB4_0,
                       new Version(REB_TYPE_SCIENCE, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_3, VERSION_1,
                                   VERSION_5, VERSION_1, VERSION_0,
                                   VERSION_3, VERSION_UNSUPP, VERSION_1));
        versionMap.put(FW_VERSION_REB4_1,
                       new Version(REB_TYPE_SCIENCE, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_3, VERSION_1,
                                   VERSION_3, VERSION_1, VERSION_0,
                                   VERSION_3, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_REB4_2,
                       new Version(REB_TYPE_SCIENCE, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_3, VERSION_1,
                                   VERSION_3, VERSION_1, VERSION_0,
                                   VERSION_3, VERSION_UNSUPP, VERSION_2));
        versionMap.put(FW_VERSION_REB5_0,
                       new Version(REB_TYPE_SCIENCE, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_6, VERSION_5,
                                   VERSION_5, VERSION_1, VERSION_1,
                                   VERSION_5, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_REB5_1,
                       new Version(REB_TYPE_SCIENCE, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_6, VERSION_4,
                                   VERSION_3, VERSION_1, VERSION_1,
                                   VERSION_5, VERSION_UNSUPP, VERSION_2));
        versionMap.put(FW_VERSION_WREB_1,
                       new Version(REB_TYPE_WAVEFRONT, VERSION_3, VERSION_1,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_2, VERSION_2,
                                   VERSION_2, VERSION_1, VERSION_0,
                                   VERSION_0, VERSION_0, VERSION_0));
        versionMap.put(FW_VERSION_WREB_2,
                       new Version(REB_TYPE_WAVEFRONT, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_4, VERSION_2,
                                   VERSION_2, VERSION_1, VERSION_1,
                                   VERSION_2, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_WREB_3,
                       new Version(REB_TYPE_WAVEFRONT, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_5, VERSION_3,
                                   VERSION_4, VERSION_1, VERSION_1,
                                   VERSION_4, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_WREB_4,
                       new Version(REB_TYPE_WAVEFRONT, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_8, VERSION_3,
                                   VERSION_4, VERSION_1, VERSION_1,
                                   VERSION_4, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_WREB_5,
                       new Version(REB_TYPE_WAVEFRONT, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_8, VERSION_3,
                                   VERSION_4, VERSION_1, VERSION_1,
                                   VERSION_4, VERSION_UNSUPP, VERSION_2));
        versionMap.put(FW_VERSION_GREB_0,
                       new Version(REB_TYPE_GUIDER, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_5, VERSION_3,
                                   VERSION_4, VERSION_1, VERSION_1,
                                   VERSION_4, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_GREB_1,
                       new Version(REB_TYPE_GUIDER, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_7, VERSION_3,
                                   VERSION_4, VERSION_1, VERSION_1,
                                   VERSION_4, VERSION_UNSUPP, VERSION_0));
        versionMap.put(FW_VERSION_GREB_2,
                       new Version(REB_TYPE_GUIDER, VERSION_3, VERSION_UNSUPP,
                                   VERSION_1, VERSION_0, VERSION_UNSUPP,
                                   VERSION_0, VERSION_7, VERSION_3,
                                   VERSION_4, VERSION_1, VERSION_1,
                                   VERSION_4, VERSION_UNSUPP, VERSION_2));
    }
    private static final int WAIT_TIMEOUT = 4000;
    private int fwVersion = FW_VERSION_INITIAL;
    private int rebType, defaultRebType = REB_TYPE_SCIENCE;
    private int numStrips;
    private int handleInst;
    private final Map<Integer, Integer> optionVersion = new HashMap<>();


    /**
     *  Constructor.
     */
    public BaseSet()
    {
        super();
    }


    /**
     *  Constructor.
     *
     *  @param  reg  The associated register client object
     */
    public BaseSet(RegClient reg)
    {
        ctxt = reg.ctxt;
    }


    /**
     *  Opens a connection.
     *
     *  @param  hdw  The hardware type to use (DAQ0, DAQ1, DAQ2, PCI0 or PCI1)
     *  @param  id   The ID of the REB to connect to
     *  @param  ifc  The name of the DAQ2 partition or hardware interface to use.
     *  @throws  REBException 
     */
    @Override
    public void open(int hdw, int id, String ifc) throws REBException
    {
        super.open(hdw, id, ifc);
        //setAllVersions();
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version number
     *  @throws  REBException if the version register can't be read
     */
    public int getHwVersion() throws REBException
    {
        return read(REG_VERSION);
    }


    /**
     *  Gets the firmware version.
     *
     *  @return  The firmware version number
     *  @throws  REBException if the firmware version is unknown
     */
    public int getFwVersion() throws REBException
    {
    //checkOpen();
    setAllVersions();
    return fwVersion;
    }


    /**
     *  Gets the REB type.
     *
     *  @return  The REB type
     *  @throws  REBException if the firmware version is unknown
     */
    public int getRebType() throws REBException
    {
        //checkOpen();
        setAllVersions();
        return rebType;
    }


    /**
     *  Sets the default REB type.
     *
     *  This is used for version information if no connection is open
     *
     *  @param  type  The REB type to set
     */
    public void setDefaultRebType(int type)
    {
        defaultRebType = type;
    }


    /**
     *  Gets the number of strips.
     *
     *  @return  The number of strips
     *  @throws  REBException if the firmware version is unknown
     */
    public int getNumStrips() throws REBException
    {
        //checkOpen();
        setAllVersions();
        return numStrips;
    }


    /**
     *  Converts a canonical strip (CCD) number to its hardware version (or vice versa).
     *
     *  This is needed because the hardware numbering of strips on science REBs
     *  is in the opposite order from the canonical numbering.
     *
     *  @param  strip  The canonical strip number
     *  @return  The hardware strip number
     *  @throws  REBException if the firmware version is unknown
     */
    public int convertStripNum(int strip) throws REBException
    {
        //checkOpen();
        setAllVersions();
        return rebType == REB_TYPE_SCIENCE ? 2 - strip : strip;
    }


    /**
     *  Gets the version of a particular option.
     *
     *  @param  option  The ID of the option
     *  @return  The option version number
     *  @throws  REBException if the option ID is invalid or is not supported
     *           by the firmware, or the firmware version is unknown
     */
    public int getVersion(int option) throws REBException
    {
        return getVersion(option, false);
    }


    /**
     *  Gets the version of a particular option.
     *
     *  @param  option   The ID of the option
     *  @param  unsOkay  If true, returns VERSION_UNSUPP if unsupported
     *  @return  The option version number
     *  @throws  REBException if the option ID is invalid or is not supported
     *           by the firmware, or the firmware version is unknown
     */
    public int getVersion(int option, boolean unsOkay) throws REBException
    {
        //checkOpen();
        setAllVersions();
        if (fwVersion == FW_VERSION_UNKNOWN) {
            throw new REBException("Unknown firmware version");
        }
        Integer version = optionVersion.get(option);
        if (version == null) {
            throw new REBException("Unrecognized option");
        }
        if (version == VERSION_UNSUPP && !unsOkay) {
            throw new REBException("Option not supported by the firmware");
        }

        return version;
    }


    /**
     *  Tests for a particular option version.
     *
     *  @param  option   The ID of the option
     *  @param  version  The array of possible versions to check against
     *  @return  Whether any of the versions match
     *  @throws  REBException if the firmware version is not recognized
     */
    public boolean isVersion(int option, int... version) throws REBException
    {
        int currVersion = getVersion(option, true);
        for (int testVersion : version) {
            if (testVersion == currVersion) return true;
        }

        return false;
        
    }


    /**
     *  Checks for particular option versions.
     *
     *  @param  option   The ID of the option
     *  @param  version  The array of possible versions to check against
     *  @throws  REBException if the firmware version is not recognized
     *           or matches none of the specified versions
     */
    public void checkVersion(int option, int... version) throws REBException
    {
        if (!isVersion(option, version)) {
            throwIncompatException();
        }
    }


    /**
     *  Checks for not particular option versions.
     *
     *  @param  option   The ID of the option
     *  @param  version  The array of possible versions to check against
     *  @throws  REBException if the firmware version is not recognized
     *           or matches any of the specified versions
     */
    public void checkNotVersion(int option, int... version) throws REBException
    {
        if (isVersion(option, version)) {
            throwIncompatException();
        }
    }


    /**
     *  Throws incompatible firmware version exception.
     *
     *  @throws  REBException
     */
    public void throwIncompatException() throws REBException
    {
        throw new REBException("Incompatible firmware version");
    }


    /**
     *  Enables the time-base.
     *
     *  @throws  REBException 
     */
    public void enable() throws REBException
    {
        enable(RSET_TIME_BASE);
    }


    /**
     *  Disables the time-base.
     *
     *  @throws  REBException 
     */
    public void disable() throws REBException
    {
        disable(RSET_TIME_BASE);
    }


    /**
     *  Gets the raw time-base value.
     *
     *  @return  The time-base value as clock ticks
     *  @throws  REBException 
     */
    public long getTimeRaw() throws REBException
    {
        return readLong(REG_TIME_BASE);
    }


    /**
     *  Gets the time-base value as millisecond Unix time.
     *
     *  @return  The time-base value as millisecond Unix time
     *  @throws  REBException 
     */
    public long getTime() throws REBException
    {
        return convertRawTime(getTimeRaw());
    }


    /**
     *  Converts a raw time-base value to milliseconds.
     *
     *  @param  raw  The raw time in clock ticks
     *  @return  The time-base value as millisecond Unix time
     *  @throws  REBException 
     */
    public long convertRawTime(long raw) throws REBException {
        return getPeriod() * raw / 1000000;
    }

    /**
     *  Sets the raw time-base value.
     *
     *  @param  time  The time-base value as clock ticks
     *  @throws  REBException 
     */
    public void setTimeRaw(long time) throws REBException
    {
        writeLong(REG_TIME_BASE, time);
    }


    /**
     *  Sets the time-base value as millisecond Unix time.
     *
     *  @param  time  The time-base value as millisecond Unix time
     *  @throws  REBException 
     */
    public void setTime(long time) throws REBException
    {
        setTimeRaw(1000000 * time / getPeriod());
    }


    /**
     *  Sets the time-base value to the current system time.
     *
     *  @throws  REBException 
     */
    public void setTime() throws REBException
    {
        setTime(System.currentTimeMillis());
    }


    /**
     *  Enables a register set.
     *
     *  @param  regSet  The register set identifier
     *  @throws  REBException 
     */
    public void enable(int regSet) throws REBException
    {
        //        update(REG_TRIGGER, 1<< regSet, 1 << regSet);
        //        write(REG_TRIGGER, read(REG_STATE) | (1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                | (1 << regSet));
    }


    /**
     *  Disables a register set.
     *
     *  @param  regSet  The register set identifier
     *  @throws  REBException 
     */
    public void disable(int regSet) throws REBException
    {
        //        update(REG_TRIGGER, 1<< regSet, 0);
        //        write(REG_TRIGGER, read(REG_STATE) & ~(1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                & ~(1 << regSet));
    }


    /**
     *  Checks whether a register set is enabled.
     *
     *  @param  regSet  The register set identifier
     *  @return  Whether the register set is enabled
     *  @throws  REBException 
     */
    public boolean isEnabled(int regSet) throws REBException
    {
        return (read(REG_STATE) & (1 << regSet)) != 0;
    }


    /**
     *  Waits for a register set to become disabled.
     *
     *  This generally indicates the completion of a read operation.  The
     *  default timeout of 1 second is used.
     *
     *  @param  regSet  The register set identifier
     *  @throws  REBException 
     */
    public void waitDone(int regSet) throws REBException
    {
        waitDone(regSet, WAIT_TIMEOUT);
    }


    /**
     *  Waits for a register set to become disabled.
     *
     *  This generally indicates the completion of a read operation
     *
     *  @param  regSet   The register set identifier
     *  @param  timeout  The timeout period (ms)
     *  @throws  REBException 
     */
    public void waitDone(int regSet, int timeout) throws REBException
    {
        long limit = System.currentTimeMillis() + timeout;
        boolean busy = true;
        int count = 0;
        while (busy && System.currentTimeMillis() < limit) {
            busy = isEnabled(regSet);
            if (busy) {
                try {
                    Thread.sleep(1);
                }
                catch (InterruptedException e) {
                }
            }
            count++;
        }
        //System.out.println("Wait done count = " + count);
        if (busy) {
            throw new REBTimeoutException("Completion wait timeout");
        }
    }


    /**
     *  Gets the raw trigger time for a register set.
     *
     *  @param  regSet  The register set identifier
     *  @return  The trigger time as clock ticks
     *  @throws  REBException 
     */
    public long getTriggerTimeRaw(int regSet) throws REBException
    {
        return readLong(REG_TRIG_TIME + 2 * regSet);
    }


    /**
     *  Gets the trigger time for a register set as Unix millisecond time.
     *
     *  @param  regSet  The register set identifier
     *  @return  The trigger time as Unix millisecond time
     *  @throws  REBException 
     */
    public long getTriggerTime(int regSet) throws REBException
    {
        return getPeriod() * getTriggerTimeRaw(regSet) / 1000000;
    }


    /**
     *  Gets the REB serial number.
     *
     *  @return  The 48-bit REB serial number
     *  @throws  REBException 
     */
    public long getRebSerial() throws REBException
    {
        getVersion(OPTN_REB_SN);
        write(REG_SN_REB_START, 0);
        long value = readLong(REG_SN_REB_VALUE);
        if ((value & SN_READ_OKAY) == 0) {
            String problem = (value & SN_READ_ERROR) != 0 ? "error" : "timeout";
            throw new REBException("Serial number read " + problem);
        }
        return value & SN_VALUE_MASK;
    }


    /**
     *  Gets the DREB serial number.
     *
     *  @return  The 48-bit DREB serial number
     *  @throws  REBException 
     */
    public long getDrebSerial() throws REBException
    {
        getVersion(OPTN_DREB_SN);
        write(REG_SN_DREB_START, 0);
        long value = readLong(REG_SN_DREB_VALUE);
        if ((value & SN_READ_OKAY) == 0) {
            String problem = (value & SN_READ_ERROR) != 0 ? "error" : "timeout";
            throw new REBException("Serial number read " + problem);
        }
        return value & SN_VALUE_MASK;
    }


    /**
     *  Sets the DC/DC synchronization.
     *
     *  @param  enable  If true, enable synchronization; otherwise disable it
     *  @throws  REBException 
     */
    public void setDcdcSync(boolean enable) throws REBException
    {
        getVersion(OPTN_DCDC_SYNC);
        write(REG_DCDC_SYNC, enable ? 1 : 0);
    }


    /**
     *  Gets the DC/DC synchronization.
     *
     *  @return  Whether DC/DC synchronization is in effect
     *  @throws  REBException 
     */
    public boolean isDcdcSync() throws REBException
    {
        getVersion(OPTN_DCDC_SYNC);
        return (read(REG_DCDC_SYNC) & 1) != 0;
    }


    /**
     *  Sets the back bias state.
     *
     *  @param  on  If true, turn the back bias on; otherwise turn it off
     *  @throws  REBException 
     */
    public void setBackBias(boolean on) throws REBException
    {
        int version = getVersion(OPTN_BACK_BIAS);
        write(REG_BACK_BIAS, (on ^ (version == VERSION_1)) ? BACK_BIAS_ON : 0);
        if (on && version == VERSION_2 && (read(REG_BACK_BIAS) & BACK_BIAS_ERROR) != 0) {
            String unders = (new BoardDacs(this)).getUndersString();
            throw new REBException("Error enabling back bias.  Values under threshold:" + unders);
        }
    }


    /**
     *  Gets the back bias setting.
     *
     *  @return  Whether the back bias is turned on
     *  @throws  REBException 
     */
    public boolean isBackBiasOn() throws REBException
    {
        int version = getVersion(OPTN_BACK_BIAS);
        return ((read(REG_BACK_BIAS) & BACK_BIAS_ON) != 0) ^ (version == VERSION_1);
    }


    /**
     *  Sets the firmware version and the version of each option.
     *
     *  @throws  REBException if the firmware version is not recognized
     */
    private void setAllVersions() throws REBException
    {
        try {
            checkOpen();
        }
        catch (REBException e) {
            if (defaultRebType == REB_TYPE_UNKNOWN) {
                throw e;
            }
            if (fwVersion == FW_VERSION_UNSET) return;
            fwVersion = FW_VERSION_UNSET;
            setVersions();
            return;
        }
        if (fwVersion > FW_VERSION_UNSET && handleInst == ctxt.instance) return;
        handleInst = ctxt.instance;
        int schema = read(REG_SCHEMA);
        int hwVersion = read(REG_VERSION);
        int pgpVersion = hwVersion >>> 16;
        int type = pgpVersion >> 12;
        hwVersion &= 0xffff;
        fwVersion = FW_VERSION_UNKNOWN;
        if (schema == 0x00f1de00) {
            fwVersion = FW_VERSION_REB0_0;
        }
        else if (pgpVersion == 0xb01c) {
            if (schema == 0x00000001) {
                fwVersion = FW_VERSION_REB0_1;
            }
            else if (schema == 0x00000002) {
                fwVersion = FW_VERSION_REB1_2;
            }
            else if (schema == 0x00000003) {
                fwVersion = FW_VERSION_REB1_3;
            }
            else if (schema == 0x00000004) {
                fwVersion = FW_VERSION_REB1_4;
            }
        }
        else if (type == 0x0b) {
            if (hwVersion < 0x100) {
                fwVersion = hwVersion < 0x20 ? FW_VERSION_WREB_1 : FW_VERSION_WREB_2;
            }
            else {
                fwVersion = hwVersion < 0x200 ? FW_VERSION_REB3 : FW_VERSION_REB4_0;
            }
        }
        else if (type == 0x01) {
            fwVersion = hwVersion < 0x4000 ? FW_VERSION_WREB_3
                          : hwVersion < 0x4002 ? FW_VERSION_WREB_4 : FW_VERSION_WREB_5;
        }
        else if (type == 0x02) {
            fwVersion = hwVersion >= 0x3000 ? FW_VERSION_WREB_3 : hwVersion < 0x2000 ? FW_VERSION_GREB_0
                          : hwVersion < 0x2002 ? FW_VERSION_GREB_1 : FW_VERSION_GREB_2;
        }
        else if (type == 0x03) {
            fwVersion = hwVersion < 0x3000 ? FW_VERSION_GREB_0 : hwVersion < 0x4000 ? FW_VERSION_REB3
                          : hwVersion < 0x4008 ? FW_VERSION_REB4_0 : hwVersion < 0x4009 ? FW_VERSION_REB4_1
                          : hwVersion < 0x5000 ? FW_VERSION_REB4_2
                          : hwVersion < 0x5003 ? FW_VERSION_REB5_0 : FW_VERSION_REB5_1;
        }
        if (fwVersion == FW_VERSION_UNKNOWN && defaultRebType == REB_TYPE_UNKNOWN) {
            throw new REBException("Unknown firmware version");
        }
        setVersions();
    }


    /**
     *  Sets the version of each option.
     */
    private void setVersions()
    {
        int fwVersn = fwVersion;
        if (fwVersion == FW_VERSION_UNSET || fwVersion == FW_VERSION_UNKNOWN) {
            fwVersn = defaultRebType == REB_TYPE_WAVEFRONT ? FW_VERSION_WREB_4
                        : defaultRebType == REB_TYPE_GUIDER ? FW_VERSION_GREB_1
                        : FW_VERSION_REB5_1;
        }
        Version versions = versionMap.get(fwVersn);
        rebType = versions.rebType;
        optionVersion.put(OPTN_SEQUENCER, versions.sequencer);
        optionVersion.put(OPTN_CABAC, versions.cabac);
        optionVersion.put(OPTN_ASPIC, versions.aspic);
        optionVersion.put(OPTN_REB_SN, versions.rebSn);
        optionVersion.put(OPTN_DCDC_SYNC, versions.dcdcSync);
        optionVersion.put(OPTN_DREB_SN, versions.drebSn);
        optionVersion.put(OPTN_BOARD_DACS, versions.dacs);
        optionVersion.put(OPTN_BOARD_POWER, versions.power);
        optionVersion.put(OPTN_BOARD_TEMP, versions.boardTemp);
        optionVersion.put(OPTN_STATUS, versions.status);
        optionVersion.put(OPTN_CCD_TEMP, versions.ccdTemp);
        optionVersion.put(OPTN_SLOW_ADCS, versions.slowAdcs);
        optionVersion.put(OPTN_FAST_ADCS, versions.fastAdcs);
        optionVersion.put(OPTN_BACK_BIAS, versions.backBias);
        numStrips = rebType == REB_TYPE_SCIENCE ? 3 : rebType == REB_TYPE_GUIDER ? 2 : 1;
    }


    /**
     *  Gets the clock period.
     *
     *  @return  The board's clock period in 10 ns units
     *  @throws  REBException 
     */
    private int getPeriod() throws REBException
    {
        return fwVersion == VERSION_0 ? CLOCK_PERIOD_0 : CLOCK_PERIOD_1;
    }

}
