package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 **********************************************
 *
 *  Temperature ADCs reading routines.
 *
 *  These are the supported board versions:
 *
 *    VERSION_0: REB0
 *    VERSION_1: REB1
 *    VERSION_2: WREB
 *    VERSION_3: REB3+
 *    VERSION_4: GREB
 *    VERSION_5: REB4/5 with wrong busy bit
 *
 *  @author Owen Saxton
 *
 **********************************************
 */
public class TempAdcs {

   /**
    *  Public constants
    */
    public static final int
        VERSION_REB0 = BaseSet.VERSION_0,
        VERSION_REB1 = BaseSet.VERSION_1,
        VERSION_REB3 = BaseSet.VERSION_3,
        VERSION_REB4X = BaseSet.VERSION_5,
        VERSION_WREB = BaseSet.VERSION_2,
        VERSION_GREB = BaseSet.VERSION_4,

        REG_TEMP_ADCS    = 0x600010,
/*      ADC_DREB_T1      = 0,
        ADC_DREB_T2      = 1,
        ADC_REB_T1       = 2,
        ADC_REB_T2       = 3,
        ADC_REB_T3       = 4,
        ADC_REB_T4       = 5,
        ADC_REB_T5       = 6,
        ADC_REB_T6       = 7,
        ADC_REB_T7       = 8,
        ADC_REB_T8       = 9,
        ADC_REB_T9       = 10, */
        NUM_TEMP_REGS_S  = 10,
        NUM_TEMP_REGS_S0 = 11,
        NUM_TEMP_REGS_W  = 6,
        NUM_TEMP_REGS_G  = 10,
        NUM_TEMP_REGS_M  = NUM_TEMP_REGS_S0,
        ERROR_MASK       = 0x00010000;

    public static final double
        TEMP_SCALE       = 1.0 / 128.0;

   /**
    *  Private data
    */
    private static final Map<Integer, Integer> numRegsMap = new HashMap<>();
    static {
        numRegsMap.put(VERSION_REB0, NUM_TEMP_REGS_S0);
        numRegsMap.put(VERSION_REB1, NUM_TEMP_REGS_S0);
        numRegsMap.put(VERSION_REB3, NUM_TEMP_REGS_S);
        numRegsMap.put(VERSION_REB4X, NUM_TEMP_REGS_S);
        numRegsMap.put(VERSION_WREB, NUM_TEMP_REGS_W);
        numRegsMap.put(VERSION_GREB, NUM_TEMP_REGS_G);
    }
    private final BaseSet bss;


   /**
    *  Constructor.
    *
    *  @param  bss  The associated base set object
    */
    public TempAdcs(BaseSet bss)
    {
        this.bss = bss;
    }


   /**
    *  Enables the temperature ADC reading.
    *
    *  @throws  REBException 
    */
    public synchronized void enable() throws REBException
    {
        bss.getVersion(BaseSet.OPTN_BOARD_TEMP);
        bss.enable(BaseSet.RSET_TEMP_ADCS);
    }


   /**
    *  Waits for the data to be available.
    *
    *  @throws  REBException 
    */
    public synchronized void waitDone() throws REBException
    {
        int version = bss.getVersion(BaseSet.OPTN_BOARD_TEMP);
        bss.waitDone(version == VERSION_REB4X ? BaseSet.RSET_FAST_ADCS : BaseSet.RSET_TEMP_ADCS);
    }


   /**
    *  Gets the time of the read enable.
    *
    *  @return  The Unix millisecond time of the read enable
    *
    *  @throws  REBException 
    */
    public long getTriggerTime() throws REBException
    {
        bss.checkNotVersion(BaseSet.OPTN_BOARD_TEMP, VERSION_REB0);
        return bss.getTriggerTime(BaseSet.RSET_TEMP_ADCS);
    }


   /**
    *  Reads one temperature value.
    *
    *  @param  adc  The number of the ADC to read
    *
    *  @return  The ADC value (Celsius)
    *
    *  @throws  REBException 
    */
    public synchronized double readAdc(int adc) throws REBException
    {
        if (adc < 0 || adc >= getNumRegs()) {
            throw new REBException("Invalid temperature ADC number");
        }
        enable();
        waitDone();
//        int value = bss.read(REG_TEMP_ADCS + adc);
//        return (value & ERROR_MASK) != 0 ? BaseSet.ERROR_VALUE : TEMP_SCALE * (short)value;
        return TEMP_SCALE * (short)bss.read(REG_TEMP_ADCS + adc);
    }


   /**
    *  Reads all the temperature values.
    *
    *  @return  The array of ADC values (Celsius)
    *
    *  @throws  REBException 
    */
    public double[] readAdcs() throws REBException
    {
        return readAdcs(0, getNumRegs());
    }


   /**
    *  Reads a range of temperature values.
    *
    *  @param  first  The number of the first ADC to read
    *
    *  @param  count  The number of ADCs to read
    *
    *  @return  The array of ADC values (Celsius)
    *
    *  @throws  REBException 
    */
    public synchronized double[] readAdcs(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > getNumRegs()) {
            throw new REBException("Invalid temperature ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[count];
        double[] data = new double[count];
        bss.read(REG_TEMP_ADCS + first, rawData);
        for (int j = 0; j < count; j++) {
//            int value = rawData[j];
//            data[j] = (value & ERROR_MASK) != 0
//                        ? BaseSet.ERROR_VALUE : TEMP_SCALE * (short)value;
            data[j] = TEMP_SCALE * (short)rawData[j];
        }

        return data;
    }


   /**
    *  Gets the number of ADC registers.
    *
    *  @return  The number of ADC registers
    *
    *  @throws  REBException 
    */
    public int getNumRegs() throws REBException
    {
        return numRegsMap.get(bss.getVersion(BaseSet.OPTN_BOARD_TEMP));
    }

}
