package org.lsst.ccs.drivers.reb;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 *  Program to test the Java global client interface routines
 *
 *  @author Owen Saxton
 */
public class TestGlobal {

    /*
     *  Private fields
     */
    private final GlobalClient gbl = new GlobalClient();


    /**
     *  Enumeration for hardware type.
     */
    public enum HdwType {

        DAQ(RegClient.HDW_TYPE_DAQ),
        DAQ0(RegClient.HDW_TYPE_DAQ0),
        DAQ1(RegClient.HDW_TYPE_DAQ1),
        DAQ2(RegClient.HDW_TYPE_DAQ2),
        DAQ4(RegClient.HDW_TYPE_DAQ4),
        PCI0(RegClient.HDW_TYPE_PCI0),
        PCI1(RegClient.HDW_TYPE_PCI1);

        int value;

        HdwType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


    /**
     *  Connects to a partition via DAQ V2/4.
     *
     *  @param  part  The partition name
     *  @throws  REBException
     */
    @Command(name="connect", description="Connect to a partition")
    public void connect(@Argument(description="Partition name") String part) throws REBException
    {
        gbl.open(part);
    }


    /**
     *  Connects to a REB.
     *
     *  @param  type     The hardware type (DAQ0, DAQ1, PCI0, PCI1)
     *  @param  id       The REB ID
     *  @param  ifcName  The network interface or PCI device name
     *  @throws  REBException
     */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(description="Hardware type") HdwType type,
                        @Argument(description="REB ID") int id,
                        @Argument(description="Network interface name") String ifcName) throws REBException
    {
        gbl.open(type.getValue(), new int[]{id}, new String[]{ifcName});
    }


    /**
     *  Disconnects from a partition or REB.
     *
     *  @throws  REBException
     */
    @Command(name="disconnect", description="Disconnect from a partition/REB")
    public void disconnect() throws REBException
    {
        gbl.close();
    }


    /**
     *  Acquires an image.
     *
     *  @param  name  The image name
     *  @throws  REBException
     */
    @Command(name="acquire", description="Acquire an image")
    public void trigger(@Argument(description="Image name") String name) throws REBException
    {
        showTime(gbl.acquireImage(name));
    }


    /**
     *  Acquires an image.
     *
     *  @param  name  The image name
     *  @param  folder  The image folder to use
     *  @param  opcode  The sequencer opcode
     *  @param  annotation  The annotation string
     *  @param  omittedIds  REB ids to omit from image data
     *  @throws  REBException
     */
    @Command(name="acquire", description="Acquire an image")
    public void trigger(@Argument(description="Image name") String name,
                        @Argument(description="Folder name") String folder,
                        @Argument(description="Sequencer opcode") int opcode,
                        @Argument(description="Annotation") String annotation,
                        @Argument(description="Omitted REB ids") int... omittedIds) throws REBException
    {
        showTime(gbl.acquireImage(name, folder, opcode, annotation, omittedIds));
    }


    /**
     *  Starts the sequencer(s).
     *
     *  @param  opcode  The opcode
     *  @throws  REBException
     */
    @Command(name="sequence", description="Start the sequrncers")
    public void sequence(@Argument(description="Sequencer opcode") int opcode) throws REBException
    {
        showTime(gbl.startSequencer(opcode));
    }


    /**
     *  Displays a trigger time.
     *
     *  @param  The time (nsec since the Epoch)
     */
    private void showTime(long time)
    {
        System.out.format("Time = %tY-%<tm-%<td %<tH:%<tM:%<tS\n", time / 1_000_000);
    }

}
