package org.lsst.ccs.drivers.reb;

/**
 *  Java interface to the REB registers.
 *
 *  @author Owen Saxton
 */
public class RegClient {

    /**
     *  Inner class to contain a copy of the register access handle, etc.
     */
    class Context {

        long handle;
        int instance;
        Impl impl;

    }


    /**
     *  Data fields.
     */
    public static final int
        HDW_TYPE_DAQ0 = 0,
        HDW_TYPE_DAQ1 = 1,
        HDW_TYPE_DAQ2 = 2,
        HDW_TYPE_PCI  = 4,
        HDW_TYPE_PCI0 = 3,
        HDW_TYPE_PCI1 = 4,
        HDW_TYPE_DAQ4 = 5,
        HDW_TYPE_DAQ  = 6;

    Context ctxt;
    private ClientFactory clientFactory = new ClientFactory();


    /**
     *  Constructor.
     */
    public RegClient()
    {
        ctxt = new Context();
    }
    

    /**
     * Can be used to override the default clientFactory, for example to 
     * substitute a clientFactory which simulates real clients.
     * 
     * @param clientFactory 
     */
    public void setClientFactory(ClientFactory clientFactory) {
        this.clientFactory = clientFactory;
    }


    /**
     *  Finalizer.
     */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        if (ctxt.impl != null && ctxt.handle != 0) {
            ctxt.impl.deleteRegClient(ctxt.handle);
        }
    }


    /**
     *  Opens a DAQ1 connection.
     *
     *  @param  id  The ID of the REB to connect to
     *  @throws  REBException 
     */
    public void open(int id) throws REBException
    {
        open(HDW_TYPE_DAQ1, id, null);
    }


    /**
     *  Opens a DAQ1/2/4 connection using the specified partition.
     *
     *  @param  id    The ID of the REB to connect to
     *  @param  part  The name of the partition
     *  @throws  REBException 
     */
    public void open(int id, String part) throws REBException
    {
        open(HDW_TYPE_DAQ, id, part);
    }


    /**
     *  Opens a connection.
     *
     *  @param  hdw   The hardware type to use (DAQ0, DAQ1, DAQ2, DAQ4, DAQ, PCI0 or PCI1)
     *  @param  id    The ID of the REB to connect to
     *  @param  part  The name of the partition or interface to use.
     *  @throws  REBException 
     */
    public synchronized void open(int hdw, int id, String part) throws REBException
    {
        if (ctxt.handle != 0) {
            throw new REBException("Register connection already open");
        }
        ctxt.impl = clientFactory.createRegClient(hdw);
        ctxt.handle = ctxt.impl.newRegClient(id, part);
        ctxt.instance++;
    }


    /**
     *  Closes a connection.
     *
     *  @throws  REBException 
     */
    public synchronized void close() throws REBException
    {
        checkOpen();
        ctxt.impl.deleteRegClient(ctxt.handle);
        ctxt.handle = 0;
    }


    /**
     *  Checks that connection is open.
     *
     *  @throws  REBException 
     */
    protected void checkOpen() throws REBException
    {
        if (ctxt.impl == null || ctxt.handle == 0) {
            throw new REBException("Register connection not open");
        }
    }


    /**
     *  Reads from a register.
     *
     *  @param  address  The address of the register to read
     *  @return  The value contained in the register
     *  @throws  REBException 
     */
    public synchronized int read(int address) throws REBException
    {
        checkOpen();
        return ctxt.impl.readReg(ctxt.handle, address);
    }


    /**
     *  Reads from a set of registers.
     *
     *  @param  address  The address of the first register to read
     *  @param  values   An array to receive the register values
     *  @param  offset   The offset to the array element where the first value is to be put
     *  @param  count    The number of registers to read
     *  @throws  REBException 
     */
    public synchronized void read(int address, int[] values, int offset, int count) throws REBException
    {
        checkOpen();
        ctxt.impl.readRegs(ctxt.handle, address, values, offset, count);
    }


    /**
     *  Reads from a set of registers.
     *
     *  @param  address  The address of the first register to read
     *  @param  values   An array to receive the register values.  The size
     *                   of the array determines how many registers to read.
     *  @throws  REBException 
     */
    public void read(int address, int[] values) throws REBException
    {
        read(address, values, 0, values.length);
    }


    /**
     *  Writes to a register.
     *
     *  @param  address  The address of the register to write
     *  @param  value    The value to write to the register
     *  @throws  REBException 
     */
    public synchronized void write(int address, int value) throws REBException
    {
        checkOpen();
        ctxt.impl.writeReg(ctxt.handle, address, value);
    }


    /**
     *  Writes to a set of registers.
     *
     *  @param  address  The address of the first register to write
     *  @param  values   An array containing the values to write
     *  @param  offset   The offset to the array element containing the first value to write
     *  @param  count    The number of registers to write
     *  @throws  REBException 
     */
    public synchronized void write(int address, int[] values, int offset, int count) throws REBException
    {
        checkOpen();
        ctxt.impl.writeRegs(ctxt.handle, address, values, offset, count);
    }


    /**
     *  Writes to a set of registers.
     *
     *  @param  address  The address of the first register to write
     *  @param  values   An array containing the values to write.  The size of
     *                   array is the number of registers to write.
     *  @throws  REBException 
     */
    public void write(int address, int[] values) throws REBException
    {
        write(address, values, 0, values.length);
    }


    /**
     *  Updates a register.
     *
     *  @param  address  The address of the register to update
     *  @param  mask     A mask in which set bits indicate which bits of
     *                   value are to be written
     *  @param  value    The value to write to the register
     *  @return  The previous value of the register
     *  @throws  REBException 
     */
    public synchronized int update(int address, int mask, int value) throws REBException
    {
        checkOpen();
        return ctxt.impl.updateReg(ctxt.handle, address, mask, value);
    }


    /**
     *  Reads a register pair as a long value.
     *
     *  @param  address  The address of the first register to read
     *  @return  The value obtained by combining the values as a little-endian pair
     *  @throws  REBException 
     */
    public long readLong(int address) throws REBException
    {
        checkOpen();
        int[] vals = new int[2];
        read(address, vals, 0, vals.length);
        return ((long)vals[1] << 32) | (vals[0] & 0xffffffffL);
    }


    /**
     *  Writes a long value to a register pair.
     *
     *  @param  address  The address of the first register to write
     *  @param  value    The value to write, as a little-endian pair
     *  @throws  REBException 
     */
    public void writeLong(int address, long value) throws REBException
    {
        checkOpen();
        int[] vals = {(int)value, (int)(value >> 32)};
        write(address, vals, 0, vals.length);
    }


    /**
     *  Resets the hardware.
     *
     *  @param  type     The type of reset to perform
     *  @throws  REBException 
     */
    public synchronized void reset(int type) throws REBException
    {
        checkOpen();
        ctxt.impl.reset(ctxt.handle, type);
    }


    /**
     *  Inner interface to support choosing the hardware.
     */
    public interface Impl {

        /**
         *  Creates a new register client object.
         *
         *  @param  id   The ID of the REB to be accessed by the object
         *  @param  ifc  The name of the partition or hardware interface to use.
         *  @return  The handle for the created object
         */
        long newRegClient(int id, String ifc);


        /**
         *  Deletes the object.
         *
         *  @param  handle  The handle of the register client object
         */
        void deleteRegClient(long handle);


        /**
         *  Reads from a register.
         *
         *  @param  handle   The handle of the register client object
         *  @param  address  The address of the register to read
         *  @return  The value contained in the register
         *  @throws  REBException 
         */
        int readReg(long handle, int address) throws REBException;


        /**
         *  Reads from a set of registers.
         *
         *  @param  handle   The handle of the register client object
         *  @param  address  The address of the first register to read
         *  @param  values   An array to receive the register values
         *  @param  offset   The offset to the array element where the first value is to be put
         *  @param  count    The number of registers to read
         *  @throws  REBException 
         */
        void readRegs(long handle, int address, int[] values, int offset, int count) throws REBException;


        /**
         *  Writes to a register.
         *
         *  @param  handle   The handle of the register client object
         *  @param  address  The address of the register to write
         *  @param  value    The value to write to the register
         *  @throws  REBException 
         */
        void writeReg(long handle, int address, int value) throws REBException;


        /**
         *  Writes to a set of registers.
         *  @param  handle   The handle of the register client object
         *  @param  address  The address of the first register to write
         *  @param  values   An array containing the values to write
         *  @param  offset   The offset to the array element containing the first value to write
         *  @param  count    The number of registers to write
         *  @throws  REBException 
         */
        void writeRegs(long handle, int address, int[] values, int offset, int count) throws REBException;


        /**
         *  Updates a register.
         *
         *  @param  handle   The handle of the register client object
         *  @param  address  The address of the register to update
         *  @param  mask     A mask in which set bits specify which bits of
         *                   value are to be written
         *  @param  value    The value to write to the register
         *  @return  The previous register value
         *  @throws  REBException 
         */
        int updateReg(long handle, int address, int mask, int value) throws REBException;


        /**
         *  Resets the hardware.
         *
         *  @param  handle   The handle of the register client object
         *  @param  type     The type of reset to perform
         *  @throws  REBException 
         */
        default void reset(long handle, int type) throws REBException {
        }

    }

}
