package org.lsst.ccs.drivers.reb.sim;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.zip.CRC32;
import org.lsst.ccs.drivers.reb.sim.AutoCloseableReentrantLock;
import org.lsst.ccs.drivers.reb.sim.SequencerSimulation;

/**
 * A general purpose sequencer listener useful for testing the sequencer
 * simulation by counting the number of occurrences of each function/subroutine.
 *
 * @author tonyj
 */
public class SequencerListener implements SequencerSimulation.SubroutineListener, SequencerSimulation.FunctionListener, SequencerSimulation.WaveformListener, SequencerSimulation.StateListener {

    private final CountMap subroutineCounts = new CountMap();
    private final CountMap functionCounts = new CountMap();
    private final CountMap subroutineReturns = new CountMap();
    private long accumulatedNanos;
    private final CRC32 checksum = new CRC32();
    private final AutoCloseableReentrantLock stateLock = new AutoCloseableReentrantLock();
    private final Condition stateChanged = stateLock.newCondition();
    private volatile SequencerSimulation.State currentState;

    @Override
    public void subroutineCalled(int address, int repetitions) {
        subroutineCounts.get(address).add(repetitions);
    }

    @Override
    public void subroutineReturned(int address) {
        subroutineReturns.get(address).increment();
    }

    @Override
    public void functionCalled(int function, boolean infiniteLoop, int repetitions) {
        functionCounts.get(function).add(repetitions);
    }

    public void waitForState(SequencerSimulation.State state, Duration timeout) throws InterruptedException, TimeoutException {
        try (AutoCloseableReentrantLock lock = stateLock.open()) {
            for (;;) {
                if (currentState == state) {
                    return;
                }
                if (!stateChanged.await(timeout.toMillis(), TimeUnit.MILLISECONDS)) {
                    throw new TimeoutException("Timed out waiting for state: " + state);
                }
            }
        }
    }
    @Override
    public void stateChanged(SequencerSimulation.State oldState, SequencerSimulation.State newState) {
        try (AutoCloseableReentrantLock lock = stateLock.open()) {
            currentState = newState;
            stateChanged.signalAll();
        }
    }

    public Map<? extends Number, ? extends Number> getSubroutineCounts() {
        return subroutineCounts;
    }

    public Map<? extends Number, ? extends Number> getFunctionCounts() {
        return functionCounts;
    }

    public Map<? extends Number, ? extends Number> getSubroutineReturns() {
        return subroutineReturns;
    }

    @Override
    public void transition(int oldState, int newState, int nanos) {
        accumulatedNanos += nanos;
        checksum.update(newState);
    }

    public long getAccumulatedNanos() {
        return accumulatedNanos;
    }

    public long getChecksum() {
        return checksum.getValue();
    }

    public void clear() {
        subroutineCounts.clear();
        functionCounts.clear();
        subroutineReturns.clear();
        accumulatedNanos = 0;
        checksum.reset();
    }

    private static class Count extends Number {

        private int count;

        void add(int n) {
            count += n;
        }

        void increment() {
            count++;
        }

        @Override
        public String toString() {
            return String.valueOf(count);
        }

        @Override
        public int intValue() {
            return count;
        }

        @Override
        public long longValue() {
            return count;
        }

        @Override
        public float floatValue() {
            return count;
        }

        @Override
        public double doubleValue() {
            return count;
        }
    }

    private static class CountMap extends ConcurrentHashMap<Integer, Count> {

        @Override
        public Count get(Object key) {
            Count result = super.get(key);
            if (result == null && key instanceof Integer) {
                result = new Count();
                put((Integer) key, result);
            }
            return result;
        }

    }
}
